# Intacct API Usage - Technical Documentation

## Table of Contents

1. [Overview](#overview)
2. [Architecture](#architecture)
3. [Platform Compatibility](#platform-compatibility)
4. [Project Structure](#project-structure)
5. [Backend Components](#backend-components)
   - [Server & Routes](#server--routes)
   - [Services](#services)
   - [Database Layer](#database-layer)
6. [Frontend Components](#frontend-components)
   - [Tab System](#tab-system)
   - [JavaScript Modules](#javascript-modules)
   - [HTML Partials](#html-partials)
7. [Theming System](#theming-system)
8. [Data Storage & Encryption](#data-storage--encryption)
9. [Key Workflows](#key-workflows)
   - [Data Fetch (Reports Tab)](#data-fetch-reports-tab)
   - [Scheduled Jobs](#scheduled-jobs)
   - [Report Generation](#report-generation)
   - [Notification System](#notification-system)
10. [Configuration](#configuration)
11. [API Reference](#api-reference)
12. [Security Considerations](#security-considerations)
13. [Troubleshooting](#troubleshooting)
14. [Dependencies](#dependencies)

---

## Overview

Intacct API Usage is a Node.js web application that monitors and reports on Sage Intacct API usage across multiple company instances. The application queries the Intacct APIUSAGEDETAIL object to collect usage metrics, stores them in a local SQLite database, and provides reporting through a web interface with optional email notifications.

### Key Features

- **Multi-Company Support**: Query API usage data across multiple Intacct company instances
- **Filter-Based Queries**: Configure custom filters based on DOCCONTROLID prefixes (e.g., EC_ for ExcelConnect, FUSAPI_ for Fusion API)
- **Scheduled Reports**: Automated data fetching and email report delivery on configurable schedules
- **Multiple Notification Providers**: Support for SMTP and Microsoft Graph email delivery
- **Excel/CSV Export**: Download detailed, summary, and exception reports in multiple formats
- **Company Management**: Load company lists from external APIs or manage manually
- **Exception Tracking**: Identify and manage companies that return errors or no data

### Version

Current version: 4.1.5

---

## Architecture

```
┌─────────────────────────────────────────────────────────────────────┐
│                         Web Browser (Client)                         │
│                                                                      │
│  ┌──────────────┐  ┌──────────────┐  ┌──────────────┐               │
│  │ Schedule Tab │  │ Reports Tab  │  │  Config Tab  │  ...          │
│  └──────────────┘  └──────────────┘  └──────────────┘               │
└─────────────────────────────────────────────────────────────────────┘
                                │
                                │ HTTP/REST
                                ▼
┌─────────────────────────────────────────────────────────────────────┐
│                      Express.js Server (Node.js)                     │
│                                                                      │
│  ┌─────────────────────────────────────────────────────────────┐    │
│  │                      Route Handlers                          │    │
│  │  ┌─────────┬──────────┬────────┬──────────┬──────────────┐  │    │
│  │  │  /job   │  /data   │/config │/schedules│/notifications│  │    │
│  │  └─────────┴──────────┴────────┴──────────┴──────────────┘  │    │
│  └─────────────────────────────────────────────────────────────┘    │
│                                                                      │
│  ┌─────────────────────────────────────────────────────────────┐    │
│  │                       Services Layer                         │    │
│  │  ┌──────────┬────────────┬───────────┬────────────────┐     │    │
│  │  │ intacct  │  scheduler │  database │   notification │     │    │
│  │  └──────────┴────────────┴───────────┴────────────────┘     │    │
│  └─────────────────────────────────────────────────────────────┘    │
│                                                                      │
│  ┌─────────────────────────────────────────────────────────────┐    │
│  │                   CLI Worker (cli.js)                        │    │
│  │  (Spawned as subprocess for data fetching)                   │    │
│  └─────────────────────────────────────────────────────────────┘    │
└─────────────────────────────────────────────────────────────────────┘
                                │
                                ▼
┌─────────────────────────────────────────────────────────────────────┐
│                      SQLite Database (better-sqlite3)                │
│                                                                      │
│  ┌──────────────┬────────────┬──────────┬─────────────────────┐     │
│  │  credentials │  settings  │  filters │    api_usage        │     │
│  │  (encrypted) │            │          │                     │     │
│  ├──────────────┼────────────┼──────────┼─────────────────────┤     │
│  │  companies   │ job_errors │schedules │notification_configs │     │
│  ├──────────────┴────────────┴──────────┴─────────────────────┤     │
│  │                     skip_companies                          │     │
│  └─────────────────────────────────────────────────────────────┘     │
└─────────────────────────────────────────────────────────────────────┘
                                │
                                ▼
┌─────────────────────────────────────────────────────────────────────┐
│                      External Systems                                │
│                                                                      │
│  ┌────────────────────┐  ┌────────────────┐  ┌────────────────┐     │
│  │  Sage Intacct API  │  │   SMTP Server  │  │ Microsoft Graph│     │
│  │  (XML/SOAP)        │  │                │  │     API        │     │
│  └────────────────────┘  └────────────────┘  └────────────────┘     │
└─────────────────────────────────────────────────────────────────────┘
```

### Request Flow

1. **User Interaction**: User interacts with the web UI (tabs, forms, buttons)
2. **API Calls**: Frontend JavaScript makes REST API calls to Express server
3. **Route Handling**: Express routes validate requests and call appropriate services
4. **Data Processing**: Services handle business logic, database operations, and external API calls
5. **CLI Subprocess**: Data fetching spawns cli.js as a subprocess for isolation and process control
6. **Database Storage**: All data persisted to SQLite using better-sqlite3 (synchronous operations)
7. **Response**: Results returned to frontend for display

---

## Platform Compatibility

| Platform | Support |
|----------|---------|
| Windows  | Full support (development and production) |
| Linux    | Full support (Docker recommended) |
| macOS    | Full support |
| Docker   | Fully containerized with docker-compose |

### Runtime Requirements

- **Node.js**: v18 or higher (v24 recommended for Docker)
- **npm**: v8 or higher
- **Python 3**: Required for better-sqlite3 native addon compilation
- **C++ Build Tools**: Required for better-sqlite3 compilation

---

## Project Structure

```
IntacctAPIUsage/
├── source/                      # Source code directory
│   ├── js/                      # Backend JavaScript (Node.js)
│   │   ├── server.js           # Express server entry point
│   │   ├── cli.js              # CLI worker for data fetching
│   │   ├── routes/             # Express route handlers
│   │   │   ├── about.js        # About/version endpoint
│   │   │   ├── config.js       # Configuration endpoints
│   │   │   ├── credentials.js  # Credential management
│   │   │   ├── data.js         # Data/reporting endpoints
│   │   │   ├── job.js          # Job execution (run/stop/status)
│   │   │   ├── logs.js         # Log file access
│   │   │   ├── notifications.js # Notification config CRUD
│   │   │   ├── schedules.js    # Schedule CRUD and execution
│   │   │   └── upload.js       # Company management endpoints
│   │   └── services/           # Business logic services
│   │       ├── config.js       # Configuration management
│   │       ├── filters.js      # DOCCONTROLID prefix filtering
│   │       ├── database.js     # Database barrel export
│   │       ├── export.js       # Report generation utilities
│   │       ├── intacct.js      # Intacct API client
│   │       ├── job.js          # Job state management
│   │       ├── logger.js       # Winston logging with rotation
│   │       ├── notification.js # Email sending (SMTP/Graph)
│   │       ├── proxy.js        # HTTP/HTTPS proxy configuration
│   │       ├── scheduler.js    # Scheduled job execution
│   │       └── db/             # Database modules
│   │           ├── core.js     # Schema, encryption, lifecycle
│   │           ├── api-usage.js # API usage records
│   │           ├── companies.js # Company list management
│   │           ├── credentials.js # Encrypted credential storage
│   │           ├── job-errors.js # Job error tracking
│   │           ├── notification-configs.js # Email configs
│   │           ├── queries.js  # Filter definitions
│   │           ├── schedules.js # Schedule definitions
│   │           ├── settings.js # Key/value settings
│   │           └── skip-companies.js # Exception list
│   │   ├── middleware/          # Express middleware
│   │   │   └── auth.js         # Session-based authentication
│   ├── public/                  # Frontend static files
│   │   ├── index.html          # Main SPA entry point
│   │   ├── login.html          # Login page (when AUTH_KEY is set)
│   │   ├── css/
│   │   │   └── styles.css      # Application styles
│   │   ├── js/                  # Frontend JavaScript (ES modules)
│   │   │   ├── main.js         # Entry point, tab registration
│   │   │   ├── tabs.js         # Tab system logic
│   │   │   ├── config.js       # Configuration tab
│   │   │   ├── reports.js      # Reports/data fetch tab
│   │   │   ├── schedule.js     # Schedules tab
│   │   │   ├── notifications.js # Notifications tab
│   │   │   ├── logs.js         # Logs viewer tab
│   │   │   ├── guide.js        # User guide tab
│   │   │   └── about.js        # About tab
│   │   ├── partials/           # HTML partial templates
│   │   │   ├── config.html     # Configuration form
│   │   │   ├── reports.html    # Reports/data fetch interface
│   │   │   ├── schedule.html   # Schedules management
│   │   │   ├── notifications.html # Notification configs
│   │   │   ├── logs.html       # Log viewer
│   │   │   ├── guide.html      # User guide content
│   │   │   └── about.html      # About page
│   │   └── images/             # Static images
│   └── package.json            # Node.js dependencies
├── TechnicalDocumentation/     # This documentation
├── Dockerfile                  # Docker build instructions
├── docker-compose.yml          # Docker compose configuration
├── env-sample.txt              # Environment variable template
└── example-companies.json      # Sample companies file
```

---

## Backend Components

### Server & Routes

#### server.js

The main Express.js server entry point that:

- Loads environment variables from `.env`
- Initializes the SQLite database
- Mounts route handlers
- Starts the scheduler service
- Handles graceful shutdown (SIGTERM, SIGINT)

```javascript
// Key initialization sequence:
1. Load .env configuration
2. Initialize database (initDatabase())
3. Connect database module to config service
4. Start scheduler service (startScheduler())
5. Listen on PORT (default 5050)
```

#### Route Handlers

| Route File | Mount Point | Purpose |
|------------|-------------|---------|
| `job.js` | `/` | Run, stop, status, download endpoints |
| `data.js` | `/` | Reporting data endpoints |
| `config.js` | `/config` | Configuration management |
| `credentials.js` | `/` | Credential CRUD |
| `upload.js` | `/` | Company management (add, remove, load from API, skip) |
| `schedules.js` | `/` | Schedule CRUD and execution |
| `notifications.js` | `/` | Notification config CRUD |
| `logs.js` | `/` | Log file access |
| `about.js` | `/` | Version endpoint |

### Services

#### intacct.js - Intacct API Client

Handles all communication with the Sage Intacct XML API:

- **XML Building**: Constructs readByQuery and readMore XML requests
- **API Communication**: Posts XML to Intacct endpoint via axios
- **Proxy Support**: Uses proxy.js for outbound connections through corporate proxies
- **Response Parsing**: Uses fast-xml-parser to parse XML responses
- **Pagination**: Handles automatic pagination with readMore requests
- **Error Handling**: Extracts control and operation errors from responses

```javascript
// IntacctClient class options:
{
  senderId: string,      // Partner/Sender ID
  senderPassword: string,// Partner/Sender password
  userId: string,        // Intacct user login
  userPassword: string,  // Intacct user password
  url: string,           // API endpoint URL
  pageSize: number,      // Records per page (default 1000)
  fields: string[],      // Fields to retrieve
  queryName: string,     // Query identifier
  queryPrefix: string,   // DOCCONTROLID prefix filter
  onData: function       // Callback for streaming data to database
}
```

#### scheduler.js - Scheduled Job Execution

Manages automated data fetching and report delivery:

- **Interval Checking**: Runs every minute to check for due schedules
- **Credential Validation**: Validates Intacct credentials before spawning CLI (uses 'Datel Group' as test company)
- **Data Fetching**: Spawns cli.js subprocess for each query
- **Data Fetch Modes**: Supports INCREMENTAL (top-up from last date) and FULL REFRESH modes with visible logging
- **Reports Only Mode**: Skips data fetching and sends reports from existing data
- **Report Generation**: Generates Excel reports after data fetch
- **Multi-Recipient Email Delivery**: Sends reports to multiple notification configs sequentially
- **Status Tracking**: Updates schedule status after each run
- **Filtered Email Content**: When a specific filter is selected, email breakdown tables only show that filter's data
- **One-off Schedules**: Automatically disables schedules with type 'oneoff' after successful execution

Key functions:
- `startScheduler()` - Begins the scheduler interval
- `stopScheduler()` - Stops the scheduler
- `executeSchedule(schedule)` - Runs a single scheduled job (supports multiple notification configs)
- `runDataFetch(queryFilter, scheduleName, useJobService, fullRefresh, reportsOnly)` - Core data fetch logic
- `generateReportAttachments(queryFilter, selectedAttachments, startDate, endDate)` - Creates Excel reports
- `buildEmailBody(queryFilter, selectedAttachments, startDate, endDate)` - Generates HTML email body with filtered breakdown table

#### notification.js - Email Sending

Supports two email delivery methods:

1. **SMTP**: Direct SMTP connection using nodemailer
   - SSL, STARTTLS, or no encryption
   - Connection pooling for performance
   - Custom from name and email

2. **Microsoft Graph**: OAuth2-based Azure email
   - Uses @azure/identity for authentication
   - Client credential flow
   - Send-as capability for mailbox users

#### logger.js - Winston Logging

Provides structured logging with:

- **Console Output**: Colorized, timestamped messages
- **File Output**: JSON format, monthly rotation
- **Log Files**: Named `app-YYYY-MM.log` for easy archival
- **API Access**: Endpoints to list and read log files from UI

#### proxy.js - HTTP/HTTPS Proxy Configuration

Provides centralized proxy configuration for outbound HTTP requests:

- **Database-Stored Configuration**: Proxy settings stored in the settings table
- **Axios Integration**: Returns configured HttpsProxyAgent for axios requests
- **Supports HTTP/HTTPS Proxies**: Works with both HTTP and HTTPS proxy servers
- **Optional Authentication**: Supports proxy username/password authentication

```javascript
// Proxy settings stored in database:
{
  proxyEnabled: "true" | "false",
  proxyHost: "proxy.example.com",
  proxyPort: "8080",
  proxyUsername: "user",      // Optional
  proxyPassword: "password"   // Optional (encrypted)
}

// Usage in services (e.g., intacct.js):
import { getProxyAgent } from './proxy.js';
const httpsAgent = getProxyAgent();
const response = await axios.post(url, data, { httpsAgent });
```

### Database Layer

#### Core Module (db/core.js)

Central database management with:

- **Schema Definitions**: Single source of truth for all table structures
- **Migration System**: Automatic schema updates on startup
- **Encryption**: AES-256-GCM encryption for credentials
- **WAL Mode**: Write-Ahead Logging for concurrent access

Tables managed:

| Table | Purpose |
|-------|---------|
| `credentials` | Encrypted Intacct credentials |
| `settings` | Key/value configuration pairs |
| `filters` | Query filter definitions |
| `api_usage` | Collected API usage records |
| `companies` | Company ID list with Sage Account Number |
| `job_errors` | Error tracking per job run |
| `notification_configs` | Email configuration (SMTP or MS Graph) |
| `schedules` | Scheduled job definitions (supports oneoff_datetime for one-off schedules, reports_only flag, notification_config_id can contain comma-separated IDs for multiple recipients) |
| `skip_companies` | Exception list (companies to skip) |

#### Database Barrel Export (database.js)

Re-exports all database functions for convenient importing:

```javascript
import {
  initDatabase,
  getApiUsageRecords,
  getAllCompanies,
  getSchedule,
  // ... all other database functions
} from './services/database.js';
```

#### Single Source of Truth: Exception Reports

The `getFilteredJobErrors()` function in `db/job-errors.js` is the single source of truth for exception reports. It excludes companies in the `skip_companies` table automatically, ensuring consistent exception handling across:

- Exception report downloads (detailed/summary/xlsx)
- Email attachment generation
- Error summary displays in the UI
- Exception count badges

Related functions:
- `getFilteredJobErrors(queryName)` - Returns filtered error records
- `getFilteredJobErrorSummary(queryName)` - Returns grouped summary with company lists
- `getFilteredJobErrorCount(queryName)` - Returns count excluding skip list

---

## Frontend Components

### Tab System

The application uses a single-page architecture with lazy-loaded tabs.

#### tabs.js

Core tab management functions:

- `registerTabInit(tabName, initFunction)` - Register initialization function
- `registerTabCleanup(tabName, cleanupFunction)` - Register cleanup function
- `switchTab(tabName)` - Switch to a tab, triggering init/cleanup
- `loadInitialTab(defaultTab)` - Load initial tab from URL hash

#### Tab Lifecycle

1. **Cleanup**: Previous tab's cleanup function called (stops polling, removes listeners)
2. **Load HTML**: Fetch partial HTML from `/partials/{tab}.html`
3. **Insert DOM**: Replace tab container content
4. **Initialize**: Call tab's init function

### JavaScript Modules

Each tab has a corresponding JavaScript module:

| Module | Tab | Key Functions |
|--------|-----|---------------|
| `schedule.js` | Schedules | CRUD schedules with multi-notification support, one-off schedules, reports only mode, run now, status display |
| `reports.js` | Reports | Run/stop jobs, live log, downloads, charts, error summary |
| `config.js` | Configuration | Settings, credentials, company management, API loading |
| `notifications.js` | Notifications | Email config CRUD with test-before-save |
| `logs.js` | Logs | Log file viewer |
| `guide.js` | User Guide | Static help content |
| `about.js` | About | Version and system info |

### HTML Partials

Located in `/public/partials/`, each partial contains:

- HTML structure for the tab
- Form elements
- Modal dialogs
- Inline event handlers pointing to JavaScript functions

---

## Theming System

### CSS Custom Properties

The application uses CSS custom properties for consistent theming:

```css
:root {
  /* Colors */
  --bg-light: #ffffff;
  --bg-dark: #1a1a1a;
  --primary: #00AFAA;      /* Sage/Intacct teal */
  --primary-dark: #008F8A;
  --text-dark: #333333;
  --text-light: #f0f0f0;
  --border-color: #e0e0e0;
  --success: #28a745;
  --error: #dc3545;
  --warning: #ffc107;

  /* Spacing */
  --spacing-sm: 8px;
  --spacing-md: 16px;
  --spacing-lg: 24px;

  /* Typography */
  --font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
  --font-size-base: 14px;
}
```

### Component Classes

Key component styles:

- `.card` - Content containers with shadow
- `.btn`, `.btn-primary`, `.btn-secondary` - Button styles
- `.btn-loading` - Button loading state with animated spinner
- `.form-group`, `.form-row` - Form layout
- `.toggle-row`, `.toggle-list` - Toggle switch lists for multi-select options
- `.toggle-option` - Radio button groups
- `.modal`, `.modal-content` - Modal dialogs
- `.status-*` - Status indicator colors

### Loading States

Buttons support a loading state via the `.btn-loading` class:
- Text becomes transparent
- Animated spinner appears centered on the button
- Pointer events are disabled
- Used during async operations like sending reports or saving notification configs

### Mobile Responsiveness

- Bottom navigation bar for mobile devices
- Collapsible admin menu
- Responsive grid layouts
- Touch-friendly button sizes

---

## Data Storage & Encryption

### SQLite Database

**File Location**: `{DATA_DIR}/intacct.db`

**Features**:
- WAL mode for concurrent access
- Automatic schema migration on startup
- better-sqlite3 for synchronous, fast operations

### Credential Encryption

All sensitive credentials are encrypted using:

- **Algorithm**: AES-256-GCM
- **Key Derivation**: PBKDF2 with 100,000 iterations
- **Salt**: 32 bytes, randomly generated per credential
- **IV**: 16 bytes, randomly generated per encryption
- **Auth Tag**: 16 bytes for authentication

**Encryption Key**: Must be provided via `ENCRYPTION_KEY` environment variable

```javascript
// Encryption process:
1. Generate random salt (32 bytes)
2. Derive key using PBKDF2(ENCRYPTION_KEY, salt)
3. Generate random IV (16 bytes)
4. Encrypt with AES-256-GCM
5. Store: salt, iv, auth_tag, encrypted_data
```

### Protected Tables

The `credentials` table is marked as `preserveData: true` in schema definitions. This ensures:
- Table is never dropped during migrations
- Only missing columns are added
- Existing encrypted data is preserved

---

## Key Workflows

### Data Fetch (Reports Tab)

```
User clicks "Run Process"
         │
         ▼
POST /run with credentials and options
         │
         ▼
Server validates credentials against Intacct API
(Uses 'Datel Group' as test company)
         │
         ▼
Spawns cli.js subprocess with env vars:
  - SENDER_ID, SENDER_PASSWORD
  - USER_ID, USER_PASSWORD
  - QUERY_NAME, QUERY_PREFIX
  - FULL_REFRESH (optional)
         │
         ▼
cli.js:
  1. Initialize database
  2. Load companies from database
  3. Log mode: FULL REFRESH or INCREMENTAL
  4. For each company:
     a. Query Intacct APIUSAGEDETAIL
     b. Filter by DOCCONTROLID prefix
     c. Store records to database
  5. Output summary
         │
         ▼
Server pipes stdout/stderr to job log
(Filtered patterns: Mode, Top-up from, Initial fetch)
         │
         ▼
Client polls GET /status for updates
         │
         ▼
On completion, reload database
         │
         ▼
Display results in UI
```

**Note**: Credential validation occurs at the scheduler/server level before CLI is spawned. The CLI process no longer performs its own credential validation to avoid duplicate data fetches.

### Scheduled Jobs

```
Scheduler interval (every 60 seconds)
         │
         ▼
Check getDueSchedules()
         │
         ▼
For each due schedule:
  1. Load companies (from CSV or API)
  2. Run data fetch (runDataFetch())
  3. Generate Excel reports
  4. Send email via notification config
  5. Update schedule status
         │
         ▼
Calculate and store next_run_at
```

### Report Generation

Three report types available:

1. **Detailed Report**: Every API usage record
   - Company ID, Sage Account Number, Partner ID, DocControlID, Transactions, Date/Time, Status, Filter

2. **Summary Report**: Aggregated by company and filter
   - Company ID, Sage Account Number, Filter, Total API Usage

3. **Exceptions Report**: Companies with errors or no data
   - Company ID, Sage Account Number, Query, Status, Error Number, Description

### Notification System

The notification system supports sending to multiple email configurations simultaneously. Both scheduled jobs and the Send Reports feature allow selecting multiple recipients.

```
Schedule or Manual Run completes
         │
         ▼
Build attachments (if configured)
         │
         ▼
Generate HTML email body with summary
(When a specific filter is selected, the
breakdown table only shows that filter)
         │
         ▼
Load notification config(s) from database
(Schedules can have multiple config IDs)
         │
         ▼
For each notification config:
  │
  ▼
Decrypt secret (password/client secret)
         │
         ▼
     ┌───┴───┐
     │       │
   SMTP   MS Graph
     │       │
     ▼       ▼
nodemailer  Azure SDK
     │       │
     └───┬───┘
         │
         ▼
Email sent with attachments
         │
         ▼
Track success/failure per config
```

**Multiple Recipients**: When multiple notification configs are selected, emails are sent sequentially to each. Partial success is possible (some configs succeed while others fail).

---

## Configuration

### Environment Variables

| Variable | Required | Default | Description |
|----------|----------|---------|-------------|
| `ENCRYPTION_KEY` | Yes | - | 32-byte hex key for credential encryption |
| `DATA_DIR` | No | `./data` | Directory for SQLite database |
| `LOG_DIR` | No | `./logs` | Directory for log files |
| `PORT` | No | `5050` | Server listening port |
| `NODE_ENV` | No | `development` | Environment mode |
| `LOG_LEVEL` | No | `info` | Logging verbosity for both server and CLI job output (error, warn, info, debug) |
| `AUTH_KEY` | No | - | Access key for web UI login (if empty, no login required) |
| `LOCAL_COMPANIES_API_ENABLED` | No | `false` | Enable local companies API endpoint |
| `LOCAL_COMPANIES_API_TOKEN` | No | - | Bearer token for local companies API |

### Database Settings

Key/value pairs stored in `settings` table:

| Key | Description |
|-----|-------------|
| `url` | Intacct API endpoint URL |
| `pageSize` | Records per API request page |
| `fields` | Comma-separated field list |
| `baseStartDate` | Earliest date to fetch data from |
| `includeCurrentMonth` | Include current month in reports |
| `companiesApiUrl` | External API URL for company list |
| `proxyEnabled` | Enable/disable proxy for outbound requests |
| `proxyHost` | Proxy server hostname |
| `proxyPort` | Proxy server port |
| `proxyUsername` | Proxy authentication username (optional) |
| `proxyPassword` | Proxy authentication password (optional, encrypted) |

### Filter Definitions

Stored in `filters` table:

| Field | Description |
|-------|-------------|
| `sort_order` | Display order |
| `name` | Filter display name (e.g., "ExcelConnect") |
| `prefix` | DOCCONTROLID prefix (e.g., "EC_") |

---

## API Reference

### Job Endpoints

#### POST /run
Start a data fetch job.

**Request Body**:
```json
{
  "senderId": "string",
  "senderPassword": "string",
  "userId": "string",
  "userPassword": "string",
  "queryName": "string (optional)",
  "runAllQueries": "boolean (optional)",
  "forceFullRefresh": "boolean (optional)",
  "notificationConfigId": "number (optional)",
  "attachments": ["detailed", "summary", "exceptions"],
  "reportStartDate": "YYYY-MM-DD (optional)",
  "reportEndDate": "YYYY-MM-DD (optional)"
}
```

#### POST /stop
Stop the running job.

#### GET /status
Get current job status.

**Response**:
```json
{
  "running": true,
  "scheduledJobRunning": false,
  "startedAt": "ISO timestamp",
  "exitCode": 0,
  "stopped": false,
  "log": "job output...",
  "queryName": "All Filters",
  "dataAvailable": true,
  "recordCount": 12345,
  "exceptionCount": 5,
  "errorCount": 2,
  "errorSummary": [...]
}
```

#### GET /download/:which
Download report (detailed, summary, exceptions).

**Query Parameters**:
- `format`: "csv" or "xlsx"

### Data Endpoints

#### GET /detailed.json
Get record count and aggregates.

#### GET /reporting.json
Get monthly aggregated data for charts.

**Query Parameters**:
- `startDate`: YYYY-MM-DD
- `endDate`: YYYY-MM-DD

#### GET /api-usage/stats.json
Get overall statistics.

#### GET /api-usage/summary.json
Get per-company summary.

### Config Endpoints

#### GET /config
Get all settings and queries.

#### POST /config
Save settings and queries.

#### POST /config/settings
Save individual setting.

### Schedule Endpoints

#### GET /schedules
List all schedules.

#### POST /schedules
Create new schedule.

#### PUT /schedules/:id
Update schedule.

#### DELETE /schedules/:id
Delete schedule.

#### POST /schedules/:id/run
Execute schedule immediately.

#### POST /schedules/stop
Stop running scheduled job.

### Notification Endpoints

#### GET /api/notifications/configs
List all notification configs (secrets masked).

#### GET /api/notifications/configs/:id
Get single notification config (secrets masked).

#### POST /api/notifications/configs
Create notification config.

#### PUT /api/notifications/configs/:id
Update notification config.

#### DELETE /api/notifications/configs/:id
Delete notification config (fails if linked to schedules).

#### POST /api/notifications/configs/:id/test
Test a saved notification configuration.

**Request Body** (optional - for testing with updated form values):
```json
{
  "name": "string",
  "recipients": "string",
  "cc": "string",
  "bcc": "string",
  "smtp": { "host": "...", "from": "..." },
  "msgraph": { "tenantId": "...", "from": "..." }
}
```

When form values are provided, they are merged with the stored config (stored secrets are preserved). This allows testing updated settings without saving first.

#### POST /api/notifications/test
Test unsaved notification configuration (from form data).

**Request Body**:
```json
{
  "name": "string",
  "provider": "smtp|msgraph",
  "recipients": "string",
  "smtp": { ... } | "msgraph": { ... }
}
```

**Frontend Test-Before-Save Behavior**: The UI automatically tests configurations before saving. If the test fails, the configuration is not saved. A loading spinner is shown during testing.

---

## Security Considerations

### Credential Protection

- All Intacct credentials stored with AES-256-GCM encryption
- Encryption key never stored in database
- PBKDF2 key derivation prevents rainbow table attacks
- Unique salt and IV per credential

### API Security

- Web UI protected by `AUTH_KEY` session-based authentication when configured
- Local Companies API disabled by default, with optional Bearer token authentication
- Companies API routes exempt from session auth (use their own token-based auth)

### Input Validation

- CSV file size limits
- Filename sanitization for log file access
- SQL injection prevention via parameterized queries
- XSS prevention (no user content rendered as HTML)

### Process Isolation

- Data fetch runs in subprocess
- SIGINT/SIGKILL handling for job termination
- Graceful shutdown on server termination

### Recommendations

1. Run behind reverse proxy with authentication for external access
2. Use strong, unique ENCRYPTION_KEY
3. Enable HTTPS in production
4. Restrict network access to Intacct API IPs
5. Regular backup of SQLite database
6. Monitor log files for errors

---

## Troubleshooting

### Common Issues

#### "Database not ready"
- Ensure DATA_DIR is writable
- Check for disk space
- Verify ENCRYPTION_KEY is set

#### "Missing credentials"
- Upload credentials via Configuration tab
- Verify ENCRYPTION_KEY matches original key used to save credentials

#### "No companies found"
- Upload CSV file or configure external API
- Check companies source setting

#### Job hangs or doesn't complete
- Check job log for errors
- Verify Intacct credentials are valid
- Check network connectivity to api.intacct.com

#### Email not sending
- Verify notification config settings
- Check SMTP server connectivity
- For MS Graph: verify Azure app registration and permissions

#### Proxy connection issues
- Verify proxy host and port are correct
- Check if proxy requires authentication (username/password)
- Test proxy connectivity independently: `curl -x http://proxy:port https://api.intacct.com`
- Review logs for "ECONNREFUSED" or "ETIMEDOUT" errors
- Ensure proxy allows HTTPS connections to api.intacct.com

#### Data fetch mode issues
- Check logs for "Mode: FULL REFRESH" or "Mode: INCREMENTAL" messages
- INCREMENTAL mode tops up from the last recorded date per company
- FULL REFRESH clears existing data for the date range
- If top-up isn't working, verify the `forceFullRefresh` setting

### Debug Mode

Set `LOG_LEVEL=debug` for verbose output:
- Query details
- Prefix filtering results
- Page-by-page progress
- Sample DOCCONTROLID values

### Log File Location

Logs are written to `{LOG_DIR}/app-YYYY-MM.log`

View via:
1. Logs tab in UI
2. Direct file access
3. Docker: `docker logs intacct-api-usage`

---

## Dependencies

### Backend (Node.js)

| Package | Version | Purpose |
|---------|---------|---------|
| express | ^4.19.2 | Web framework |
| cors | ^2.8.5 | CORS middleware |
| dotenv | ^16.4.5 | Environment variables |
| better-sqlite3 | ^11.7.0 | SQLite database |
| axios | ^1.7.2 | HTTP client |
| fast-xml-parser | ^4.4.0 | XML parsing |
| exceljs | ^4.4.0 | Excel generation |
| json2csv | ^5.0.7 | CSV generation |
| cron-parser | ^4.9.0 | Cron expression parsing |
| multer | ^1.4.5-lts.1 | File upload handling |
| nodemailer | ^6.9.0 | SMTP email |
| https-proxy-agent | ^7.0.0 | HTTPS proxy support |
| @azure/identity | ^4.0.0 | Azure authentication |
| @microsoft/microsoft-graph-client | ^3.0.0 | MS Graph API |
| winston | ^3.17.0 | Logging |

### Frontend

| Library | Version | Purpose |
|---------|---------|---------|
| Chart.js | CDN | Chart rendering |

### Build Dependencies

| Tool | Purpose |
|------|---------|
| Python 3 | better-sqlite3 compilation |
| node-gyp | Native addon compilation |
| make, g++ | C++ compilation |

---

## Licensing

This is an internal tool developed by Fusion/Datel for monitoring Sage Intacct API usage. All rights reserved.

---

*Documentation updated: February 2026*
*Version: 4.1.5*

### Version 4.1.5 Changes
- **Authentication**: Added `AUTH_KEY` session-based authentication for the web UI with login page
- **Companies API auth bypass**: Companies API routes exempt from session auth (use their own Bearer token auth)
- **CSV cleanup**: Removed unused `csv-parse` dependency; renamed `csv.js` to `filters.js` with only prefix-matching utilities
- **LOG_LEVEL consolidation**: Merged `LIVE_LOG_LEVEL` into `LOG_LEVEL` (single env var for both server and CLI)
- **Mobile support**: Removed mobile device blocking - app now works on all screen sizes
- **Responsive improvements**: Card-based table layouts, centred guide TOC pills, mobile bottom navigation

### Version 4.1.0 Changes
- **One-off schedules**: Added new schedule type that runs once at a specific date/time, then automatically disables
- **Reports Only mode**: New schedule option to skip data fetching and send reports from existing data
- **Sage Account Number**: Reports now include Sage Account Number column via LEFT JOIN from companies table
- **Schedule UI improvements**: Edit and Delete buttons are disabled when a schedule is enabled for safety
- **Enable reminder**: After saving a schedule, a reminder is displayed to enable it using the toggle

### Version 4.0.0 Changes
- **Proxy support**: Added HTTP/HTTPS proxy configuration for outbound Intacct API connections
- **Tab consolidation**: Merged Fetch Data and Reports functionality into consolidated Reports tab
- **Improved data fetch modes**: Visible logging of INCREMENTAL vs FULL REFRESH mode in job logs
- **Credential validation optimization**: Removed redundant CLI credential validation; scheduler validates before spawning CLI
- **Company search**: Added search/filter functionality to Manage Companies modal
- **Single source of truth**: Exception reports now use `getFilteredJobErrors()` consistently across all features
- **better-sqlite3 migration**: Moved from sqlite3 to better-sqlite3 for improved performance and memory handling

### Version 3.1.0 Changes
- **Multiple notification configs**: Schedules and Send Reports now support sending to multiple email configurations simultaneously
- **Test-before-save**: Notification configurations are automatically tested before saving; if the test fails, the configuration is not saved
- **Filtered email breakdown**: When sending reports for a specific filter, the email breakdown table only shows that filter's data
- **Loading spinners**: Added animated loading indicators for async operations (sending reports, saving notification configs)
- **Toggle lists**: Notification selection changed from dropdowns to toggle lists for better multi-select UX
