#!/bin/bash

# Push existing Docker image to remote registry (no build, no version bump)
# Usage: ./development_scripts/push-only.sh

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"

# Source configuration
source "$SCRIPT_DIR/docker-config.sh"

# Load registry credentials
ENV_FILE="$PROJECT_DIR/.env-docker-registry"
if [ ! -f "$ENV_FILE" ]; then
    echo "Error: $ENV_FILE not found. Copy .env-docker-registry.example to .env-docker-registry and fill in your credentials."
    exit 1
fi
source "$ENV_FILE"

# Validate required variables
if [ -z "$DOCKER_REGISTRY" ] || [ -z "$DOCKER_USERNAME" ] || [ -z "$DOCKER_PASSWORD" ]; then
    echo "Error: DOCKER_REGISTRY, DOCKER_USERNAME, and DOCKER_PASSWORD must be set in $ENV_FILE"
    exit 1
fi

# Get current version
VERSION=$("$SCRIPT_DIR/version.sh" get)
echo "Current version: $VERSION"

# Full image names
LOCAL_IMAGE="$IMAGE_NAME:$VERSION"
LOCAL_LATEST="$IMAGE_NAME:latest"
REMOTE_IMAGE="$DOCKER_REGISTRY/$IMAGE_NAME:$VERSION"
REMOTE_LATEST="$DOCKER_REGISTRY/$IMAGE_NAME:latest"

# Check if local image exists
if ! docker image inspect "$LOCAL_IMAGE" &>/dev/null; then
    echo "Error: Local image $LOCAL_IMAGE not found. Build it first."
    exit 1
fi

echo "Logging in to registry..."
echo "$DOCKER_PASSWORD" | docker login "$DOCKER_REGISTRY" -u "$DOCKER_USERNAME" --password-stdin

echo "Tagging images for registry..."
docker tag "$LOCAL_IMAGE" "$REMOTE_IMAGE"
docker tag "$LOCAL_LATEST" "$REMOTE_LATEST"

echo "Pushing to registry..."
docker push "$REMOTE_IMAGE"
docker push "$REMOTE_LATEST"

echo "Successfully pushed $REMOTE_IMAGE and $REMOTE_LATEST"
