#!/bin/bash

# Build and push Docker image to remote registry (linux/arm64)
# Usage: ./development_scripts/push-to-registry.sh [major|minor|patch]

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"

# Source configuration
source "$SCRIPT_DIR/docker-config.sh"

# Load registry credentials
ENV_FILE="$PROJECT_DIR/.env-docker-registry"
if [ ! -f "$ENV_FILE" ]; then
    echo "Error: $ENV_FILE not found. Copy .env-docker-registry.example to .env-docker-registry and fill in your credentials."
    exit 1
fi
source "$ENV_FILE"

# Validate required variables
if [ -z "$DOCKER_REGISTRY" ] || [ -z "$DOCKER_USERNAME" ] || [ -z "$DOCKER_PASSWORD" ]; then
    echo "Error: DOCKER_REGISTRY, DOCKER_USERNAME, and DOCKER_PASSWORD must be set in $ENV_FILE"
    exit 1
fi

# Get version bump type (default: patch)
BUMP_TYPE="${1:-patch}"

# Bump version and get new version
VERSION=$("$SCRIPT_DIR/version.sh" "$BUMP_TYPE")
echo "Version bumped to: $VERSION"

# Full image names
LOCAL_IMAGE="$IMAGE_NAME:$VERSION"
LOCAL_LATEST="$IMAGE_NAME:latest"
REMOTE_IMAGE="$DOCKER_REGISTRY/$IMAGE_NAME:$VERSION"
REMOTE_LATEST="$DOCKER_REGISTRY/$IMAGE_NAME:latest"

echo "Building Docker image..."
docker build -t "$LOCAL_IMAGE" -t "$LOCAL_LATEST" "$PROJECT_DIR"

echo "Logging in to registry..."
echo "$DOCKER_PASSWORD" | docker login "$DOCKER_REGISTRY" -u "$DOCKER_USERNAME" --password-stdin

echo "Tagging images for registry..."
docker tag "$LOCAL_IMAGE" "$REMOTE_IMAGE"
docker tag "$LOCAL_LATEST" "$REMOTE_LATEST"

echo "Pushing to registry..."
docker push "$REMOTE_IMAGE"
docker push "$REMOTE_LATEST"

echo "Successfully pushed $REMOTE_IMAGE and $REMOTE_LATEST"
