#!/usr/bin/env node
// cli.js - CLI wrapper for Intacct API Usage Export
import { IntacctClient, buildQuery } from './services/intacct.js';
import { mapToDbRecord, ErrorRecorder } from './services/export.js';
import {
  initDatabase,
  closeDatabase,
  insertApiUsageRecords,
  clearApiUsageRecords,
  clearCompanyQueryRecordsFromDate,
  getLastFetchedDate,
  getApiUsageMonthly,
  getApiUsageCount,
  getAllCompanies,
  insertJobError,
  getJobErrors,
  getAllSkipCompanies
} from './services/database.js';
import { isValidPrefixPattern } from './services/filters.js';

// ---- CLI args parsing ----
function parseArgs(argv) {
  const out = {};
  for (let i = 0; i < argv.length; i++) {
    const a = argv[i];
    if (a.startsWith('--')) {
      const eq = a.indexOf('=');
      if (eq > -1) {
        out[a.slice(2, eq)] = a.slice(eq + 1);
      } else {
        const k = a.slice(2);
        const v = (i + 1 < argv.length && !argv[i + 1].startsWith('--')) ? argv[++i] : true;
        out[k] = v;
      }
    }
  }
  return out;
}

function printHelp() {
  console.log(`
Usage:
  node cli.js

Companies are read from the SQLite database (uploaded via web UI).

Required environment variables:
  SENDER_ID         Partner/Sender ID
  SENDER_PASSWORD   Partner/Sender password
  USER_ID           Intacct user login
  USER_PASSWORD     Intacct user password

Optional environment variables:
  IA_URL            Defaults to https://api.intacct.com/ia/xml/xmlgw.phtml
  FIELDS            Comma list of fields (default: PARTNERID,DOCCONTROLID,N_TRANS,CREATED_DATE,CREATED_TIME,STATUS)
  QUERY_PREFIX      DOCCONTROLID prefix for filtering (e.g., "EC_" or "FUSAPI_")
  QUERY_NAME        Query name for database records
  BASE_START_DATE   Earliest date for data fetch (YYYY-MM-DD, default: 2025-08-01)
  PAGE_SIZE         Page size for readByQuery (default: 1000)
  FULL_REFRESH      Set to "true" to clear all data and re-fetch from base start date

Data is stored directly in the SQLite database. Use the web UI to view and export data.
`);
}

// Log level for live log output (info = basic, debug = verbose)
const LOG_LEVEL = (process.env.LOG_LEVEL || 'info').toLowerCase();

function logDebug(...args) {
  if (LOG_LEVEL === 'debug') console.log(...args);
}

// ====== MAIN ======
async function main() {
  const ARGS = parseArgs(process.argv.slice(2));

  if (process.argv.includes('--help') || process.argv.includes('-h') || ARGS.help) {
    printHelp();
    process.exit(0);
  }

  // Initialize database for storing API usage data
  console.log('Initializing database...');
  await initDatabase();

  // Configuration from environment
  const SENDER_ID = process.env.SENDER_ID;
  const SENDER_PASSWORD = process.env.SENDER_PASSWORD;
  const USER_ID = process.env.USER_ID;
  const USER_PASSWORD = process.env.USER_PASSWORD;

  if (!SENDER_ID || !SENDER_PASSWORD || !USER_ID || !USER_PASSWORD) {
    console.error('Missing required env vars: SENDER_ID, SENDER_PASSWORD, USER_ID, USER_PASSWORD');
    process.exit(1);
  }

  const IA_URL = process.env.IA_URL || 'https://api.intacct.com/ia/xml/xmlgw.phtml';

  const FIELDS = (process.env.FIELDS || 'PARTNERID,DOCCONTROLID,N_TRANS,CREATED_DATE,CREATED_TIME,STATUS')
    .split(',').map(s => s.trim()).filter(Boolean);
  const PAGE_SIZE = parseInt(process.env.PAGE_SIZE || '1000', 10);
  const QUERY_PREFIX = process.env.QUERY_PREFIX || 'EC_'; // Prefix for DOCCONTROLID filtering
  const QUERY_NAME = process.env.QUERY_NAME || 'Query'; // Query name for database records
  const FULL_REFRESH = process.env.FULL_REFRESH === 'true';
  const BASE_START_DATE = process.env.BASE_START_DATE || '2025-08-01'; // ISO format (YYYY-MM-DD)

  // Auto-build query from prefix (space before prefix in LIKE clause is intentional)
  const BASE_QUERY = `function not in 'getAPISession' and DOCCONTROLID like ' ${QUERY_PREFIX}%'`;
  // Add the base start date to the query
  const QUERY_BASE = buildQuery(BASE_QUERY, BASE_START_DATE, null);
  logDebug('Base query:', QUERY_BASE);
  logDebug('Filter prefix:', QUERY_PREFIX);
  logDebug('Query name:', QUERY_NAME);
  logDebug('Full refresh:', FULL_REFRESH);
  logDebug('Base start date:', BASE_START_DATE);

  // Get companies from database (uploaded via web UI)
  const companyValues = getAllCompanies();

  if (companyValues.length === 0) {
    console.error('No companies found in database. Upload a CSV file via the web UI first.');
    process.exit(1);
  }

  console.log(`Companies loaded from database: ${companyValues.length}`);
  console.log(`Mode: ${FULL_REFRESH ? 'FULL REFRESH (clearing all data)' : 'INCREMENTAL (top-up from last date)'}`);

  // Track database inserts
  let dbInsertedCount = 0;
  let dbFilteredOutCount = 0;

  // Callback for storing data directly to database as it's fetched
  const handleData = (items, companyId, queryName, prefix) => {
    // Debug: log sample DOCCONTROLIDs to diagnose filtering issues
    if (items.length > 0) {
      const sample = items.slice(0, 3).map(i => {
        const val = i.DOCCONTROLID;
        return `(type=${typeof val}, value=${JSON.stringify(val)})`;
      });
      logDebug(`[${companyId}] Received ${items.length} items. Sample DOCCONTROLIDs: ${sample.join(', ')}`);
    }

    // Filter items by prefix before storing
    let filteredItems;
    if (prefix) {
      filteredItems = items.filter(item => {
        const docId = String(item.DOCCONTROLID ?? '').trim();
        return isValidPrefixPattern(docId, prefix);
      });
      const rejected = items.length - filteredItems.length;
      if (rejected > 0) {
        dbFilteredOutCount += rejected;
        logDebug(`[${companyId}] Prefix filter '${prefix}': ${filteredItems.length} matched, ${rejected} rejected out of ${items.length}`);
        // Show sample of rejected items for debugging
        if (filteredItems.length === 0) {
          const rejectedSample = items.slice(0, 3).map(i => String(i.DOCCONTROLID ?? '').trim());
          logDebug(`[${companyId}] WARNING: All items rejected. Sample DOCCONTROLIDs: ${JSON.stringify(rejectedSample)}`);
        }
      }
    } else {
      filteredItems = items;
    }

    if (filteredItems.length === 0) return;

    // Convert to database records
    const records = filteredItems.map(item => mapToDbRecord(item, companyId, queryName));

    // Insert into database
    const inserted = insertApiUsageRecords(records);
    dbInsertedCount += inserted;

    console.log(`[${companyId}] Stored ${inserted} of ${records.length} records to database (total: ${dbInsertedCount})`);
  };

  // Create client with database storage callback
  const client = new IntacctClient({
    senderId: SENDER_ID,
    senderPassword: SENDER_PASSWORD,
    userId: USER_ID,
    userPassword: USER_PASSWORD,
    url: IA_URL,
    pageSize: PAGE_SIZE,
    fields: FIELDS,
    queryName: QUERY_NAME,
    queryPrefix: QUERY_PREFIX,
    onData: handleData
  });

  const errorRecorder = new ErrorRecorder();
  let totalProcessed = 0;

  // Full refresh: bulk-clear all records for this query up front
  if (FULL_REFRESH) {
    console.log('Mode: Full refresh (clearing all existing data)');
    const cleared = clearApiUsageRecords(QUERY_NAME);
    if (cleared > 0) {
      console.log(`Cleared ${cleared} existing records for query '${QUERY_NAME}'`);
    }
  } else {
    console.log('Mode: Top-up (per-company from last data date)');
  }

  // Build skip list: user-configured exceptions + companies with auth errors from earlier filters
  const skipCompanies = new Set();

  // Load user-configured exceptions
  const userSkipList = getAllSkipCompanies();
  for (const id of userSkipList) {
    skipCompanies.add(id);
  }
  if (userSkipList.length > 0) {
    console.log(`Skipping ${userSkipList.length} companies from exceptions list`);
  }

  // Add companies with auth errors from earlier filters in this run
  const previousErrors = getJobErrors();
  for (const err of previousErrors) {
    if (err.error_number === 'XL03000006') {
      skipCompanies.add(err.company_id);
    }
  }
  const authSkipCount = skipCompanies.size - userSkipList.length;
  if (authSkipCount > 0) {
    console.log(`Skipping ${authSkipCount} additional companies with auth errors from earlier filters`);
  }

  // Note: Credentials are validated by the scheduler before spawning this CLI process
  // See scheduler.js testCredentials() call - no need to duplicate validation here

  // Process each company
  for (const companyId of companyValues) {
    if (skipCompanies.has(companyId)) {
      const reason = userSkipList.includes(companyId) ? 'in exceptions list' : 'auth error from earlier filter';
      console.log(`\n=== Company: ${companyId} === SKIPPED (${reason})`);
      continue;
    }
    console.log(`\n=== Company: ${companyId} ===`);

    // Determine the query for this company based on its actual data
    let companyQuery;
    if (FULL_REFRESH) {
      companyQuery = QUERY_BASE;
    } else {
      // Check this company's latest data date from actual records
      const lastDate = getLastFetchedDate(QUERY_NAME, companyId);
      if (lastDate) {
        // Clear from last date onward and re-fetch from that date
        const cleared = clearCompanyQueryRecordsFromDate(QUERY_NAME, companyId, lastDate);
        if (cleared > 0) {
          console.log(`[${companyId}] Cleared ${cleared} records from ${lastDate} onward`);
        }
        companyQuery = buildQuery(BASE_QUERY, lastDate, null);
        console.log(`[${companyId}] Top-up from ${lastDate}`);
      } else {
        companyQuery = QUERY_BASE;
        console.log(`[${companyId}] Initial fetch (no existing data)`);
      }
    }

    logDebug(`[${companyId}] Query: ${companyQuery}`);

    try {
      const result = await client.queryCompany(companyId, companyQuery, (pageInfo) => {
        logDebug(`[${companyId}] page ${pageInfo.page}: count=${pageInfo.count}, remaining=${pageInfo.numRemaining}`);
      });

      if (!result.success) {
        errorRecorder.recordError(companyId, result.error, result.errors, 1, result.rawFile);

        // Handle auth errors - credentials are already validated against Datel Group,
        // so any auth errors here are company-specific access issues
        if (result.fatalAuth) {
          console.log(`[${companyId}] Skipped: User does not have access to this company`);
          // Add to skip list so it's skipped in subsequent filters in this run
          skipCompanies.add(companyId);
          // Error is recorded to job_errors for Exceptions Report - user can manually add to skip list
        }

        if (result.partialItems) {
          totalProcessed += result.partialItems.length;
        }
        continue;
      }

      totalProcessed += result.items.length;
      console.log(`[${companyId}] completed. Records fetched: ${result.items.length}`);

    } catch (e) {
      errorRecorder.recordHttpError(companyId, 'HTTP_ERROR', e, 1);
      continue;
    }
  }

  // Summary
  console.log(`\n=== JOB SUMMARY ===`);
  console.log(`Total records fetched from API: ${totalProcessed}`);
  console.log(`Records filtered out by prefix: ${dbFilteredOutCount}`);
  console.log(`Records stored in database: ${dbInsertedCount}`);
  console.log(`Total records in database: ${getApiUsageCount(QUERY_NAME)}`);

  if (errorRecorder.hasErrors()) {
    const errors = errorRecorder.getErrors();
    console.log(`Errors encountered: ${errors.length}`);
    for (const err of errors) {
      console.log(`  - ${err.Company}: ${err.Description}`);
      // Persist error to database
      insertJobError({
        company_id: err.Company,
        query_name: QUERY_NAME,
        error_type: err.Stage || 'UNKNOWN',
        error_number: err.ErrorNo || null,
        description: err.Description || 'Unknown error',
        description2: err.Description2 || null,
        page: err.Page || 1,
        recorded_at: new Date().toISOString()
      });
    }
  } else {
    console.log('No errors encountered.');
  }

  // Monthly totals
  const monthlyData = getApiUsageMonthly({ query_name: QUERY_NAME });
  if (monthlyData.length > 0) {
    console.log(`\n=== MONTHLY TOTALS (${QUERY_NAME}) ===`);
    let grandTotalRecords = 0;
    let grandTotalTransactions = 0;
    for (const m of monthlyData) {
      console.log(`  ${m.month}: ${m.record_count} records, ${m.total_transactions} transactions`);
      grandTotalRecords += m.record_count;
      grandTotalTransactions += m.total_transactions;
    }
    console.log(`  TOTAL: ${grandTotalRecords} records, ${grandTotalTransactions} transactions`);
  } else {
    console.log(`\nNo monthly data available for query '${QUERY_NAME}'.`);
  }

  // Close database connection
  closeDatabase();
  console.log('Database connection closed.');
}

main().then(() => {
  process.exit(0);
}).catch(err => {
  if (err.response) {
    console.error('HTTP error:', err.response.status, err.response.statusText);
    console.error(err.response.data);
  } else {
    console.error(err.stack || err.message || err);
  }
  process.exit(1);
});
