// auth.js - Simple auth_key authentication middleware
import crypto from 'crypto';

const AUTH_KEY = process.env.AUTH_KEY || '';
const COOKIE_NAME = 'intacct_session';
const SESSION_MAX_AGE = 7 * 24 * 60 * 60 * 1000; // 7 days

// In-memory session store (simple Map of token -> expiry)
const sessions = new Map();

function generateToken() {
  return crypto.randomBytes(32).toString('hex');
}

function parseCookies(cookieHeader) {
  const cookies = {};
  if (!cookieHeader) return cookies;
  cookieHeader.split(';').forEach(cookie => {
    const [name, ...rest] = cookie.trim().split('=');
    cookies[name] = rest.join('=');
  });
  return cookies;
}

/**
 * Check if auth is enabled
 */
export function isAuthEnabled() {
  return AUTH_KEY.length > 0;
}

/**
 * Validate an auth key and return a session token
 */
export function login(key) {
  if (key === AUTH_KEY) {
    const token = generateToken();
    sessions.set(token, Date.now() + SESSION_MAX_AGE);
    return token;
  }
  return null;
}

/**
 * Destroy a session
 */
export function logout(token) {
  sessions.delete(token);
}

/**
 * Express middleware - protects all routes when AUTH_KEY is set
 */
export function authMiddleware(req, res, next) {
  // No auth configured - allow everything
  if (!isAuthEnabled()) return next();

  // Allow login page (auth API routes are mounted before this middleware)
  if (req.path === '/login' || req.path === '/login.html') {
    return next();
  }

  // Allow static assets needed for login page
  if (req.path.startsWith('/css/') || req.path.startsWith('/images/') || req.path === '/favicon.ico') {
    return next();
  }

  // Allow companies API routes through - they have their own token-based auth
  if (req.path.startsWith('/companies/') || req.path === '/companies') {
    return next();
  }

  // Check session cookie
  const cookies = parseCookies(req.headers.cookie);
  const token = cookies[COOKIE_NAME];

  if (token && sessions.has(token)) {
    const expiry = sessions.get(token);
    if (Date.now() < expiry) {
      return next();
    }
    // Expired
    sessions.delete(token);
  }

  // Not authenticated
  if (req.path.startsWith('/api/') || req.headers.accept?.includes('application/json')) {
    return res.status(401).json({ error: 'Authentication required' });
  }

  // Redirect to login page
  return res.redirect('/login.html');
}
