// routes/about.js - About information endpoint
import express from "express";
import os from "os";
import fs from "fs";
import path from "path";
import { fileURLToPath } from "url";

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const router = express.Router();

// Read package.json for version and dependencies
function getPackageInfo() {
  try {
    const packagePath = path.join(__dirname, "..", "..", "package.json");
    const packageJson = JSON.parse(fs.readFileSync(packagePath, "utf8"));
    return packageJson;
  } catch (e) {
    return { version: "unknown", dependencies: {} };
  }
}

// Get network interfaces for IP addresses
function getIpAddresses() {
  const interfaces = os.networkInterfaces();
  const addresses = [];

  for (const name of Object.keys(interfaces)) {
    for (const iface of interfaces[name]) {
      // Skip internal (loopback) and non-IPv4 addresses
      if (!iface.internal && iface.family === "IPv4") {
        addresses.push(iface.address);
      }
    }
  }

  return addresses.length > 0 ? addresses : ["localhost"];
}

// Check if running in Docker
function isRunningInDocker() {
  try {
    // Check for .dockerenv file
    if (fs.existsSync("/.dockerenv")) {
      return true;
    }
    // Check cgroup for docker
    const cgroup = fs.readFileSync("/proc/1/cgroup", "utf8");
    return cgroup.includes("docker");
  } catch {
    return false;
  }
}

// Get Linux distribution info
function getDistroInfo() {
  try {
    if (os.platform() !== "linux") return null;

    // Try /etc/os-release first
    if (fs.existsSync("/etc/os-release")) {
      const content = fs.readFileSync("/etc/os-release", "utf8");
      const match = content.match(/PRETTY_NAME="([^"]+)"/);
      if (match) return match[1];
    }

    return null;
  } catch {
    return null;
  }
}

router.get("/api/about", (req, res) => {
  const packageInfo = getPackageInfo();
  const port = process.env.PORT || 5050;

  const aboutInfo = {
    version: packageInfo.version,
    description: packageInfo.description || "Intacct API Usage Export Tool",
    nodeVersion: process.version,
    dependencies: packageInfo.dependencies || {},
    environment: {
      hostname: os.hostname(),
      ipAddresses: getIpAddresses(),
      port: port,
      os: `${os.type()} ${os.release()}`,
      distro: getDistroInfo(),
      platform: os.platform(),
      architecture: os.arch(),
      isDocker: isRunningInDocker()
    }
  };

  res.json(aboutInfo);
});

export default router;
