// routes/config.js - Configuration management routes
import express from "express";
import { getConfig, saveConfig, DEFAULT_CONFIG } from "../services/config.js";
import { setCredential, getCredential } from "../services/database.js";

const router = express.Router();

// Helper to load current config (uses database-backed getConfig)
function loadConfig() {
  try {
    return getConfig();
  } catch (e) {
    console.warn("[Config] Failed to load config:", e.message);
    return null;
  }
}

// GET /config/queries - Get all queries
// Structure: { "1": { name: "Query Name", prefix: "EC_" }, "2": { ... } }
router.get("/queries", (req, res) => {
  const config = loadConfig();
  if (!config) {
    return res.status(500).json({ ok: false, message: "Failed to load config" });
  }

  const queries = config.intacct?.queries || {};
  // Convert to array format for frontend, sorted by order
  const queryList = Object.entries(queries)
    .map(([order, q]) => ({
      order: parseInt(order, 10),
      name: q.name,
      prefix: q.prefix || ""
    }))
    .sort((a, b) => a.order - b.order);

  return res.json({ ok: true, queries: queryList });
});

// PUT /config/queries - Create or update a query
// Body: { order: 1, name: "Query Name", prefix: "EC_" }
// Query text is auto-generated from the prefix and baseStartDate setting
router.put("/queries", (req, res) => {
  const { order, name, prefix } = req.body;

  if (!order || order < 1) {
    return res.status(400).json({ ok: false, message: "Order must be a positive number" });
  }

  if (!name || !prefix) {
    return res.status(400).json({ ok: false, message: "Name and prefix are required" });
  }

  const config = loadConfig();
  if (!config) {
    return res.status(500).json({ ok: false, message: "Failed to load config" });
  }

  // Ensure intacct.queries exists
  if (!config.intacct) {
    config.intacct = {};
  }
  if (!config.intacct.queries) {
    config.intacct.queries = {};
  }

  // Use order as the key, store name and prefix (query text auto-generated on save)
  config.intacct.queries[String(order)] = { name, prefix: prefix || "" };

  try {
    saveConfig(config);
    return res.json({ ok: true, message: "Query saved" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: "Failed to save config: " + e.message });
  }
});

// DELETE /config/queries/:order - Delete a query by order number
router.delete("/queries/:order", (req, res) => {
  const order = decodeURIComponent(req.params.order);

  const config = loadConfig();
  if (!config) {
    return res.status(500).json({ ok: false, message: "Failed to load config" });
  }

  if (!config.intacct?.queries?.[order]) {
    return res.status(404).json({ ok: false, message: "Query not found" });
  }

  // Don't allow deleting the last query
  const queryCount = Object.keys(config.intacct.queries).length;
  if (queryCount <= 1) {
    return res.status(400).json({ ok: false, message: "Cannot delete the last query" });
  }

  delete config.intacct.queries[order];

  try {
    saveConfig(config);
    return res.json({ ok: true, message: "Query deleted" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: "Failed to save config: " + e.message });
  }
});

// Default settings from config service
const DEFAULT_SETTINGS = DEFAULT_CONFIG.intacct;

// Default queries from config service
const DEFAULT_QUERIES = DEFAULT_CONFIG.intacct.queries;

// POST /config/queries/reset - Reset queries to defaults
router.post("/queries/reset", (req, res) => {
  const config = loadConfig();
  if (!config) {
    return res.status(500).json({ ok: false, message: "Failed to load config" });
  }

  // Ensure intacct object exists
  if (!config.intacct) {
    config.intacct = {};
  }

  // Replace queries with defaults
  config.intacct.queries = JSON.parse(JSON.stringify(DEFAULT_QUERIES));

  try {
    saveConfig(config);
    return res.json({ ok: true, message: "Queries reset to defaults" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: "Failed to save config: " + e.message });
  }
});

// GET /config/settings - Get API settings
router.get("/settings", (req, res) => {
  const config = loadConfig();
  if (!config) {
    return res.status(500).json({ ok: false, message: "Failed to load config" });
  }

  const settings = {
    url: config.intacct?.url || DEFAULT_SETTINGS.url,
    pageSize: config.intacct?.pageSize || DEFAULT_SETTINGS.pageSize,
    fields: config.intacct?.fields || DEFAULT_SETTINGS.fields,
    baseStartDate: config.intacct?.baseStartDate || DEFAULT_SETTINGS.baseStartDate,
    companyIdColumn: config.intacct?.companyIdColumn || DEFAULT_SETTINGS.companyIdColumn,
    includeCurrentMonth: config.intacct?.includeCurrentMonth ?? DEFAULT_SETTINGS.includeCurrentMonth,
    runAllQueries: config.intacct?.runAllQueries ?? DEFAULT_SETTINGS.runAllQueries,
    forceFullRefresh: config.intacct?.forceFullRefresh ?? DEFAULT_SETTINGS.forceFullRefresh,
    companiesSource: config.intacct?.companiesSource || 'csv',
    companiesApiUrl: config.intacct?.companiesApiUrl || '',
    companiesApiToken: getCredential('companiesApiToken') || ''
  };

  return res.json({ ok: true, settings });
});

// PUT /config/settings - Update API settings
router.put("/settings", (req, res) => {
  const { url, pageSize, fields, baseStartDate, companyIdColumn, includeCurrentMonth, runAllQueries, forceFullRefresh } = req.body;

  if (!url) {
    return res.status(400).json({ ok: false, message: "API URL is required" });
  }

  const config = loadConfig();
  if (!config) {
    return res.status(500).json({ ok: false, message: "Failed to load config" });
  }

  // Ensure intacct object exists
  if (!config.intacct) {
    config.intacct = {};
  }

  config.intacct.url = url;
  config.intacct.pageSize = parseInt(pageSize, 10) || 1000;
  config.intacct.fields = fields || DEFAULT_SETTINGS.fields;
  config.intacct.baseStartDate = baseStartDate || DEFAULT_SETTINGS.baseStartDate;
  config.intacct.companyIdColumn = companyIdColumn || DEFAULT_SETTINGS.companyIdColumn;
  config.intacct.includeCurrentMonth = !!includeCurrentMonth;
  config.intacct.runAllQueries = !!runAllQueries;
  config.intacct.forceFullRefresh = !!forceFullRefresh;

  try {
    saveConfig(config);
    return res.json({ ok: true, message: "Settings saved" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: "Failed to save config: " + e.message });
  }
});

// PUT /config/settings/companies-source - Update companies source settings only
router.put("/settings/companies-source", (req, res) => {
  const { companiesSource, companiesApiUrl, companiesApiToken } = req.body;

  const config = loadConfig();
  if (!config) {
    return res.status(500).json({ ok: false, message: "Failed to load config" });
  }

  if (!config.intacct) {
    config.intacct = {};
  }

  config.intacct.companiesSource = companiesSource || 'csv';
  config.intacct.companiesApiUrl = companiesApiUrl || '';
  // Don't store token in config - it goes to encrypted credentials

  try {
    saveConfig(config);

    // Store token encrypted in credentials table
    if (companiesApiToken) {
      setCredential('companiesApiToken', companiesApiToken);
    }

    return res.json({ ok: true, message: "Companies source saved" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: "Failed to save config: " + e.message });
  }
});

// POST /config/settings/reset - Reset settings to defaults
router.post("/settings/reset", (req, res) => {
  const config = loadConfig();
  if (!config) {
    return res.status(500).json({ ok: false, message: "Failed to load config" });
  }

  // Ensure intacct object exists
  if (!config.intacct) {
    config.intacct = {};
  }

  config.intacct.url = DEFAULT_SETTINGS.url;
  config.intacct.pageSize = DEFAULT_SETTINGS.pageSize;
  config.intacct.fields = DEFAULT_SETTINGS.fields;
  config.intacct.baseStartDate = DEFAULT_SETTINGS.baseStartDate;
  config.intacct.companyIdColumn = DEFAULT_SETTINGS.companyIdColumn;
  config.intacct.includeCurrentMonth = DEFAULT_SETTINGS.includeCurrentMonth;
  config.intacct.runAllQueries = DEFAULT_SETTINGS.runAllQueries;
  config.intacct.forceFullRefresh = DEFAULT_SETTINGS.forceFullRefresh;

  try {
    saveConfig(config);
    return res.json({ ok: true, message: "Settings reset to defaults" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: "Failed to save config: " + e.message });
  }
});

export default router;
