// routes/credentials.js - Credential management routes
import express from "express";
import {
  setCredential,
  getCredential,
  getAllCredentials,
  deleteAllCredentials,
  deleteCredential,
  isEncryptionConfigured,
  getSetting,
  setSetting
} from "../services/database.js";
import { clearProxyCache } from "../services/proxy.js";

const router = express.Router();

router.get("/credentials", (req, res) => {
  try {
    // Check if encryption is configured
    if (!isEncryptionConfigured()) {
      return res.json({
        ok: true,
        credentials: null,
        encryptionConfigured: false,
        message: "ENCRYPTION_KEY not configured"
      });
    }

    const credentials = getAllCredentials();
    if (credentials) {
      return res.json({ ok: true, credentials, encryptionConfigured: true });
    } else {
      return res.json({ ok: true, credentials: null, encryptionConfigured: true });
    }
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

router.post("/credentials", (req, res) => {
  try {
    // Check if encryption is configured
    if (!isEncryptionConfigured()) {
      return res.status(400).json({
        ok: false,
        message: "ENCRYPTION_KEY not configured. Set it in .env or Docker environment."
      });
    }

    const { senderId, senderPassword, userId, userPassword } = req.body || {};

    if (!senderId || !senderPassword || !userId || !userPassword) {
      return res.status(400).json({
        ok: false,
        message: "All credential fields are required"
      });
    }

    setCredential('senderId', senderId);
    setCredential('senderPassword', senderPassword);
    setCredential('userId', userId);
    setCredential('userPassword', userPassword);

    return res.json({ ok: true, message: "Credentials saved securely" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

router.delete("/credentials", (req, res) => {
  try {
    const deleted = deleteAllCredentials();
    if (deleted) {
      return res.json({ ok: true, message: "Credentials deleted" });
    } else {
      return res.json({ ok: true, message: "No credentials to delete" });
    }
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// Proxy configuration endpoints
router.get("/proxy", (req, res) => {
  try {
    // URL and enabled state are in settings (not sensitive)
    const proxyUrl = getSetting('proxyUrl');
    const proxyEnabled = getSetting('proxyEnabled');

    // Username and password are encrypted in credentials
    const proxyUsername = isEncryptionConfigured() ? getCredential('proxyUsername') : null;
    const proxyPassword = isEncryptionConfigured() ? getCredential('proxyPassword') : null;

    // Enabled defaults to true if URL exists but flag not set
    const isEnabled = proxyEnabled !== '0' && proxyEnabled !== 0;

    if (proxyUrl) {
      return res.json({
        ok: true,
        proxy: {
          url: proxyUrl,
          username: proxyUsername || '',
          hasPassword: !!proxyPassword,
          enabled: isEnabled
        },
        encryptionConfigured: isEncryptionConfigured()
      });
    } else {
      return res.json({ ok: true, proxy: null, encryptionConfigured: isEncryptionConfigured() });
    }
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

router.post("/proxy", (req, res) => {
  try {
    const { url, username, password, enabled } = req.body || {};

    // URL is required if saving proxy config
    if (!url || !url.trim()) {
      // If no URL, clear all proxy settings
      setSetting('proxyUrl', '');
      setSetting('proxyEnabled', '');
      if (isEncryptionConfigured()) {
        deleteCredential('proxyUsername');
        deleteCredential('proxyPassword');
      }
      clearProxyCache();
      return res.json({ ok: true, message: "Proxy configuration cleared" });
    }

    // Validate URL format
    try {
      new URL(url);
    } catch {
      return res.status(400).json({ ok: false, message: "Invalid proxy URL format" });
    }

    // URL and enabled state go in settings (not sensitive)
    setSetting('proxyUrl', url.trim());

    // Save enabled state (store as '1' or '0') - explicitly check for disabled
    const isEnabled = enabled !== false && enabled !== 'false' && enabled !== '0' && enabled !== 0;
    setSetting('proxyEnabled', isEnabled ? '1' : '0');

    // Username and password are encrypted in credentials (if encryption configured)
    if (isEncryptionConfigured()) {
      if (username && username.trim()) {
        setCredential('proxyUsername', username.trim());
      } else {
        deleteCredential('proxyUsername');
      }

      if (password && password.trim()) {
        setCredential('proxyPassword', password.trim());
      } else {
        deleteCredential('proxyPassword');
      }
    }

    clearProxyCache();
    return res.json({ ok: true, message: "Proxy configuration saved" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

router.delete("/proxy", (req, res) => {
  try {
    // Clear settings
    setSetting('proxyUrl', '');
    setSetting('proxyEnabled', '');
    // Clear encrypted credentials
    if (isEncryptionConfigured()) {
      deleteCredential('proxyUsername');
      deleteCredential('proxyPassword');
    }
    clearProxyCache();
    return res.json({ ok: true, message: "Proxy configuration cleared" });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// Toggle proxy enabled state
router.post("/proxy/toggle", (req, res) => {
  try {
    const proxyUrl = getSetting('proxyUrl');
    if (!proxyUrl) {
      return res.status(400).json({
        ok: false,
        message: "No proxy configured"
      });
    }

    const { enabled } = req.body || {};
    // Explicitly convert to boolean to handle string "false" edge case
    const isEnabled = enabled === true || enabled === 'true' || enabled === '1';

    // Save to settings table (not encrypted)
    setSetting('proxyEnabled', isEnabled ? '1' : '0');
    clearProxyCache();

    return res.json({
      ok: true,
      enabled: isEnabled,
      message: isEnabled ? "Proxy enabled" : "Proxy disabled"
    });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

export default router;
