// routes/data.js - Data/reporting routes (database only)
import express from "express";
import fs from "fs";
import path from "path";
import { getConfig, DATA_DIR } from "../services/config.js";
import { getJob } from "../services/job.js";
import {
  getApiUsageRecords,
  getApiUsageMonthly,
  getApiUsageMonthlyByFilter,
  getApiUsageSummary,
  getApiUsageCompanies,
  getApiUsageDateRange,
  getApiUsageCount,
  clearApiUsageRecords,
  getLastFetchedAtByQuery,
  isDatabaseReady
} from "../services/database.js";

const router = express.Router();

// ── Local Companies API middleware ──
// Check if local companies API is enabled and validate token if required
function localCompaniesApiMiddleware(req, res, next) {
  const enabled = process.env.LOCAL_COMPANIES_API_ENABLED;
  // Check if enabled (must be explicitly set to true, 1, or "true")
  if (!enabled || (enabled !== 'true' && enabled !== '1')) {
    return res.status(404).json({ ok: false, message: "Local companies API is not enabled" });
  }

  // Check token if configured
  const requiredToken = process.env.LOCAL_COMPANIES_API_TOKEN;
  if (requiredToken && requiredToken.trim()) {
    const authHeader = req.headers.authorization;
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return res.status(401).json({ ok: false, message: "Authorization header required" });
    }

    const providedToken = authHeader.substring(7); // Remove "Bearer " prefix
    if (providedToken !== requiredToken) {
      return res.status(403).json({ ok: false, message: "Invalid token" });
    }
  }

  next();
}

// Totals from database
router.get("/detailed.json", (req, res) => {
  try {
    if (!isDatabaseReady()) {
      return res.status(503).json({ ok: false, message: "Database not ready" });
    }

    const currentJob = getJob();
    const dbCount = getApiUsageCount();

    if (dbCount === 0) {
      return res.status(404).json({ ok: false, message: "No data available. Run a job first." });
    }

    // Query database
    const filters = {};
    const isAllQueriesMode = !currentJob.queryName || currentJob.queryName === "All Filters";
    if (!isAllQueriesMode && currentJob.queryName) {
      filters.query_name = currentJob.queryName;
    }

    const records = getApiUsageRecords(filters);
    const headers = ['CompanyID', 'PARTNERID', 'DOCCONTROLID', 'N_TRANS', 'CREATED_DATE', 'CREATED_TIME', 'STATUS'];

    // Calculate numeric sums
    const numericSums = {
      N_TRANS: records.reduce((sum, r) => sum + (r.n_trans || 0), 0)
    };

    return res.json({
      ok: true,
      headers,
      count: records.length,
      aggregates: { numericSums },
      queryName: currentJob.queryName
    });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// Reporting - usage counts aggregated by month, with optional date range filtering
router.get("/reporting.json", (req, res) => {
  try {
    if (!isDatabaseReady()) {
      return res.status(503).json({ ok: false, message: "Database not ready" });
    }

    const currentJob = getJob();
    const config = getConfig();
    const includeCurrentMonth = config.intacct?.includeCurrentMonth ?? false;

    // Get current month
    const now = new Date();
    const currentMonthKey = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, "0")}`;
    const defaultStartMonth = "2025-08";

    const dbCount = getApiUsageCount();
    if (dbCount === 0) {
      return res.status(404).json({ ok: false, message: "No data available. Run a job first." });
    }

    // Optional date range from query params (YYYY-MM-DD format)
    const { startDate, endDate } = req.query;

    // Query database for monthly data
    const isAllQueriesMode = !currentJob.queryName || currentJob.queryName === "All Filters";
    const filters = {
      exclude_current_month: !includeCurrentMonth
    };
    if (!isAllQueriesMode && currentJob.queryName) {
      filters.query_name = currentJob.queryName;
    }
    if (startDate) filters.start_date = startDate;
    if (endDate) filters.end_date = endDate;

    const monthlyData = getApiUsageMonthly(filters);

    // Determine effective start month filter
    const startMonthKey = startDate ? startDate.substring(0, 7) : defaultStartMonth;

    // Filter by start month and convert to expected format
    const monthly = monthlyData
      .filter(m => m.month >= startMonthKey)
      .map(m => ({ month: m.month, count: m.record_count }));

    // Get per-filter monthly data for chart (only in "All Filters" mode)
    let monthlyByFilter = null;
    if (isAllQueriesMode) {
      const byFilterData = getApiUsageMonthlyByFilter(filters);
      // Group into { filterName: [ { month, count }, ... ] }
      monthlyByFilter = {};
      for (const row of byFilterData) {
        if (row.month < startMonthKey) continue;
        if (!monthlyByFilter[row.query_name]) {
          monthlyByFilter[row.query_name] = [];
        }
        monthlyByFilter[row.query_name].push({ month: row.month, count: row.record_count });
      }
    }

    // Get query breakdown for "All Filters" mode
    let queryBreakdown = null;
    if (isAllQueriesMode) {
      const allRecords = getApiUsageRecords({});
      const lastFetchedMap = getLastFetchedAtByQuery();

      const counts = {};
      const companySets = {};
      for (const r of allRecords) {
        // Apply date filtering
        if (!r.created_date) continue;
        const monthKey = r.created_date.substring(0, 7); // YYYY-MM
        if ((!includeCurrentMonth && monthKey === currentMonthKey) || monthKey < startMonthKey) {
          continue;
        }
        // Apply end date filtering
        if (endDate && r.created_date > endDate) continue;

        const queryName = r.query_name || "Other";
        counts[queryName] = (counts[queryName] || 0) + 1;
        if (!companySets[queryName]) companySets[queryName] = new Set();
        if (r.company_id) companySets[queryName].add(r.company_id);
      }

      // Build breakdown with count, company count, and last fetched date
      queryBreakdown = {};
      for (const [name, count] of Object.entries(counts)) {
        queryBreakdown[name] = {
          count,
          companyCount: companySets[name] ? companySets[name].size : 0,
          lastFetched: lastFetchedMap[name] || null
        };
      }
    }

    return res.json({
      ok: true,
      dateHeader: 'created_date',
      monthly,
      monthlyByFilter,
      count: dbCount,
      currentMonth: currentMonthKey,
      startMonth: startMonthKey,
      excludedCurrentMonth: !includeCurrentMonth,
      queryName: currentJob.queryName || "All Filters",
      queryBreakdown,
      filters: { startDate: startDate || null, endDate: endDate || null }
    });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// Get API usage statistics from database
router.get("/api-usage/stats.json", (req, res) => {
  try {
    if (!isDatabaseReady()) {
      return res.status(503).json({ ok: false, message: "Database not ready" });
    }

    const count = getApiUsageCount();
    const companies = getApiUsageCompanies();
    const dateRange = getApiUsageDateRange();

    return res.json({
      ok: true,
      count,
      companies: companies.length,
      companyList: companies,
      dateRange
    });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// Get summary by company from database
router.get("/api-usage/summary.json", (req, res) => {
  try {
    if (!isDatabaseReady()) {
      return res.status(503).json({ ok: false, message: "Database not ready" });
    }

    const { queryName } = req.query;
    const summary = getApiUsageSummary(queryName || null);

    return res.json({
      ok: true,
      summary,
      queryName: queryName || "All Filters"
    });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// Clear API usage data from database
router.delete("/api-usage", (req, res) => {
  try {
    if (!isDatabaseReady()) {
      return res.status(503).json({ ok: false, message: "Database not ready" });
    }

    const { queryName } = req.body || {};
    const cleared = clearApiUsageRecords(queryName || null);

    return res.json({
      ok: true,
      cleared,
      message: queryName
        ? `Cleared ${cleared} records for query "${queryName}"`
        : `Cleared all ${cleared} records`
    });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// Get any JSON file from DATA_DIR that contains a companies array
router.get("/companies/:filename.json", localCompaniesApiMiddleware, (req, res) => {
  try {
    const { filename } = req.params;
    const filePath = path.join(DATA_DIR, `${filename}.json`);

    if (!fs.existsSync(filePath)) {
      return res.status(404).json({ ok: false, message: `${filename}.json not found` });
    }

    const data = fs.readFileSync(filePath, "utf8");
    const parsed = JSON.parse(data);

    // Only serve files that have a companies array
    if (!parsed.companies || !Array.isArray(parsed.companies)) {
      return res.status(400).json({ ok: false, message: "File does not contain a companies array" });
    }

    return res.json({ ok: true, ...parsed });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// Get all companies from all JSON files combined (deduplicated)
router.get("/companies/all", localCompaniesApiMiddleware, (req, res) => {
  try {
    const allCompanies = new Set();

    const files = fs.readdirSync(DATA_DIR).filter(f => f.endsWith(".json"));

    for (const f of files) {
      try {
        const data = fs.readFileSync(path.join(DATA_DIR, f), "utf8");
        const parsed = JSON.parse(data);
        if (parsed.companies && Array.isArray(parsed.companies)) {
          for (const company of parsed.companies) {
            allCompanies.add(company);
          }
        }
      } catch {
        // Skip invalid JSON files
      }
    }

    const companies = Array.from(allCompanies).sort();

    return res.json({
      ok: true,
      companies,
      count: companies.length
    });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// List all JSON files in DATA_DIR that contain a companies array
router.get("/companies", localCompaniesApiMiddleware, (req, res) => {
  try {
    const files = fs.readdirSync(DATA_DIR)
      .filter(f => f.endsWith(".json"))
      .map(f => {
        try {
          const data = fs.readFileSync(path.join(DATA_DIR, f), "utf8");
          const parsed = JSON.parse(data);
          if (parsed.companies && Array.isArray(parsed.companies)) {
            return {
              filename: f,
              count: parsed.companies.length
            };
          }
        } catch {
          // Skip invalid JSON files
        }
        return null;
      })
      .filter(Boolean);

    return res.json({ ok: true, files });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

export default router;
