// routes/job.js - Job status, reporting, and download routes
import express from "express";
import { Parser as Json2CsvParser } from "json2csv";
import ExcelJS from "exceljs";
import { getConfig } from "../services/config.js";
import {
  getJob,
  isScheduledJobRunning
} from "../services/job.js";
import {
  getApiUsageRecords,
  getApiUsageSummary,
  getApiUsageCount,
  getCompaniesWithNoData,
  getFilteredJobErrorCount,
  getFilteredJobErrorSummary,
  isDatabaseReady,
  reloadDatabase
} from "../services/database.js";
import { isoToMdy } from "../services/export.js";
import { generateReportAttachments, buildEmailBody, resolveReportDateRange, getExceptionRows } from "../services/scheduler.js";
import { sendNotification } from "../services/notification.js";
import logger from "../services/logger.js";
import { testCredentials } from "../services/intacct.js";

const router = express.Router();

// Get available queries from config
// Structure: { "1": { name: "QueryName", prefix: "EC_" }, "2": { ... } }
// Returns sorted by numeric key (sort order)
router.get("/queries", (req, res) => {
  const config = getConfig();
  const queries = config.intacct?.queries || {};
  // Convert to array, sort by numeric key, include the order number
  const queryList = Object.entries(queries)
    .map(([order, q]) => ({
      order: parseInt(order, 10),
      name: q.name,
      prefix: q.prefix || ""
    }))
    .sort((a, b) => a.order - b.order);
  return res.json({ ok: true, queries: queryList });
});

// Test Intacct credentials by making a simple API call
router.post("/test-credentials", async (req, res) => {
  const { senderId, senderPassword, userId, userPassword, testCompanyId } = req.body || {};

  if (!senderId || !senderPassword || !userId || !userPassword) {
    return res.status(400).json({
      ok: false,
      message: "Missing required credentials"
    });
  }

  try {
    const result = await testCredentials({
      senderId,
      senderPassword,
      userId,
      userPassword,
      testCompanyId
    });

    if (result.error === 'CONNECTION_ERROR') {
      return res.status(500).json(result);
    }

    return res.json(result);
  } catch (e) {
    logger.error(`[Credentials] Test failed: ${e.message}`);
    return res.status(500).json({
      ok: false,
      error: 'CONNECTION_ERROR',
      message: `Connection error: ${e.message}`
    });
  }
});

// Send reports using existing data (no data fetch)
router.post("/send-reports", async (req, res) => {
  const { notificationConfigId, attachments, reportStartDate, reportEndDate, queryFilter } = req.body || {};

  if (!notificationConfigId) {
    return res.status(400).json({ ok: false, message: "No email configuration selected." });
  }

  try {
    // Reload database to ensure we see latest data (including job_errors from subprocess)
    await reloadDatabase();

    const selectedAttachments = attachments || ['detailed', 'summary', 'exceptions'];
    const { startDate, endDate } = resolveReportDateRange(reportStartDate || '', reportEndDate || '');
    const reportAttachments = await generateReportAttachments(queryFilter || 'all', selectedAttachments, startDate, endDate);

    const dateStr = new Date().toISOString().slice(0, 10);
    const filterLabel = (queryFilter && queryFilter !== 'all') ? queryFilter : 'All';
    const subject = `Intacct API Usage Report (${filterLabel}) - ${dateStr}`;
    const body = buildEmailBody(queryFilter || 'all', selectedAttachments, startDate, endDate);

    await sendNotification(notificationConfigId, subject, body, reportAttachments);
    logger.info(`[Job] Manual send reports completed (${filterLabel})`);
    res.json({ ok: true, message: "Reports sent successfully." });
  } catch (error) {
    logger.error(`[Job] Manual send reports error: ${error.message}`);
    res.status(500).json({ ok: false, message: error.message });
  }
});

router.get("/status", (req, res) => {
  const currentJob = getJob();

  // Check database for data availability
  const dbReady = isDatabaseReady();
  const dbCount = dbReady ? getApiUsageCount() : 0;

  // Count companies with no data (exceptions) and job errors
  // Uses filtered functions that exclude skip_companies (single source of truth)
  let exceptionCount = 0;
  let errorCount = 0;
  let errorSummary = [];
  if (dbReady) {
    const isAllQueriesMode = !currentJob.queryName || currentJob.queryName === "All Filters";
    const queryName = isAllQueriesMode ? null : currentJob.queryName;

    if (dbCount > 0) {
      exceptionCount = getCompaniesWithNoData(queryName).length;
    }

    errorCount = getFilteredJobErrorCount(queryName);
    if (errorCount > 0) {
      errorSummary = getFilteredJobErrorSummary(queryName);
    }
  }

  return res.json({
    running: currentJob.running,
    scheduledJobRunning: isScheduledJobRunning(),
    startedAt: currentJob.startedAt,
    exitCode: currentJob.exitCode,
    stopped: currentJob.stopped,
    log: currentJob.log.slice(-80000),
    queryName: currentJob.queryName,
    queryPrefix: currentJob.queryPrefix,
    dataAvailable: dbCount > 0,
    recordCount: dbCount,
    exceptionCount,
    errorCount,
    errorSummary
  });
});

router.get("/download/:which", async (req, res) => {
  const which = req.params.which;
  const format = (req.query.format || "csv").toLowerCase();
  const currentJob = getJob();

  // Reload database to ensure we see latest data (including job_errors from subprocess)
  try {
    await reloadDatabase();
  } catch (err) {
    logger.warn(`[Job] Download reload warning: ${err.message}`);
  }

  if (!isDatabaseReady()) {
    return res.status(503).send("Database not ready.");
  }

  const dbCount = getApiUsageCount();
  if (dbCount === 0) {
    return res.status(404).send("No data available. Run a schedule first.");
  }

  const dateStr = new Date().toISOString().slice(0, 10);
  const queryLabel = currentJob.queryName || 'All';

  try {
    if (which === "detailed") {
      // Use filter from query param if provided, otherwise fall back to current job's queryName
      const filterParam = req.query.filter;
      const filters = {};
      if (filterParam && filterParam !== 'all') {
        filters.query_name = filterParam;
      } else if (!filterParam) {
        const isAllQueriesMode = !currentJob.queryName || currentJob.queryName === "All Filters";
        if (!isAllQueriesMode && currentJob.queryName) {
          filters.query_name = currentJob.queryName;
        }
      }

      const records = getApiUsageRecords(filters);

      if (format === "xlsx") {
        const workbook = new ExcelJS.Workbook();
        const sheet = workbook.addWorksheet("Detailed");

        sheet.columns = [
          { header: "Sage Account", key: "SageAccount", width: 15 },
          { header: "CompanyID", key: "CompanyID", width: 30 },
          { header: "PARTNERID", key: "PARTNERID", width: 15 },
          { header: "DOCCONTROLID", key: "DOCCONTROLID", width: 20 },
          { header: "N_TRANS", key: "N_TRANS", width: 10 },
          { header: "CREATED_DATE", key: "CREATED_DATE", width: 14 },
          { header: "CREATED_TIME", key: "CREATED_TIME", width: 14 },
          { header: "STATUS", key: "STATUS", width: 12 },
          { header: "FILTER", key: "FILTER", width: 18 }
        ];

        for (const r of records) {
          sheet.addRow({
            SageAccount: r.sage_account_number || '',
            CompanyID: r.company_id,
            PARTNERID: r.partner_id || '',
            DOCCONTROLID: r.doc_control_id,
            N_TRANS: r.n_trans,
            CREATED_DATE: r.created_date ? new Date(r.created_date + "T00:00:00") : null,
            CREATED_TIME: r.created_time || '',
            STATUS: r.status || '',
            FILTER: r.query_name || ''
          });
        }

        // Format the CREATED_DATE column as dates
        sheet.getColumn("CREATED_DATE").numFmt = "yyyy-mm-dd";

        const filename = `Intacct-Results-${queryLabel}-${dateStr}.xlsx`;
        res.setHeader("Content-Type", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        res.setHeader("Content-Disposition", `attachment; filename="${filename}"`);
        return workbook.xlsx.write(res).then(() => res.end());
      }

      // CSV format (default)
      const csvData = records.map(r => ({
        'Sage Account': r.sage_account_number || '',
        CompanyID: r.company_id,
        PARTNERID: r.partner_id || '',
        DOCCONTROLID: r.doc_control_id,
        N_TRANS: r.n_trans,
        CREATED_DATE: isoToMdy(r.created_date) || '',
        CREATED_TIME: r.created_time || '',
        STATUS: r.status || '',
        FILTER: r.query_name || ''
      }));

      const fields = ['Sage Account', 'CompanyID', 'PARTNERID', 'DOCCONTROLID', 'N_TRANS', 'CREATED_DATE', 'CREATED_TIME', 'STATUS', 'FILTER'];
      const csvParser = new Json2CsvParser({ fields });
      const csv = csvParser.parse(csvData);

      const filename = `Intacct-Results-${queryLabel}-${dateStr}.csv`;
      res.setHeader('Content-Type', 'text/csv');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      return res.send(csv);

    } else if (which === "summary") {
      // Use filter from query param if provided, otherwise fall back to current job's queryName
      const filterParam = req.query.filter;
      let queryName = null;
      if (filterParam && filterParam !== 'all') {
        queryName = filterParam;
      } else if (!filterParam) {
        queryName = currentJob.queryName === "All Filters" ? null : currentJob.queryName;
      }
      const summary = getApiUsageSummary(queryName);

      if (format === "xlsx") {
        const workbook = new ExcelJS.Workbook();
        const sheet = workbook.addWorksheet("Summary");

        sheet.columns = [
          { header: "Sage Account", key: "SageAccount", width: 15 },
          { header: "Company", key: "Company", width: 30 },
          { header: "Filter", key: "Filter", width: 18 },
          { header: "API Usage", key: "ApiUsage", width: 16 }
        ];

        for (const s of summary) {
          sheet.addRow({
            SageAccount: s.sage_account_number || '',
            Company: s.company_id,
            Filter: s.query_name || '',
            ApiUsage: s.total_trans
          });
        }

        const filename = `Intacct-Summary-${queryLabel}-${dateStr}.xlsx`;
        res.setHeader("Content-Type", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        res.setHeader("Content-Disposition", `attachment; filename="${filename}"`);
        return workbook.xlsx.write(res).then(() => res.end());
      }

      // CSV format (default)
      const csvData = summary.map(s => ({
        'Sage Account': s.sage_account_number || '',
        Company: s.company_id,
        Filter: s.query_name || '',
        'API Usage': s.total_trans
      }));

      const fields = ['Sage Account', 'Company', 'Filter', 'API Usage'];
      const csvParser = new Json2CsvParser({ fields });
      const csv = csvParser.parse(csvData);

      const filename = `Intacct-Summary-${queryLabel}-${dateStr}.csv`;
      res.setHeader('Content-Type', 'text/csv');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      return res.send(csv);

    } else if (which === "exceptions") {
      // Use filter from query param if provided, otherwise fall back to current job's queryName
      const filterParam = req.query.filter;
      let queryFilter = 'all';
      if (filterParam && filterParam !== 'all') {
        queryFilter = filterParam;
      } else if (!filterParam) {
        // No filter param - use current job's queryName for backward compatibility
        const isAllQueriesMode = !currentJob.queryName || currentJob.queryName === "All Filters";
        queryFilter = isAllQueriesMode ? 'all' : currentJob.queryName;
      }

      // Use shared function (single source of truth)
      const exceptionRows = getExceptionRows(queryFilter);

      if (exceptionRows.length === 0) {
        return res.status(404).send("No exceptions found. All companies returned data successfully.");
      }

      if (format === "xlsx") {
        const workbook = new ExcelJS.Workbook();
        const sheet = workbook.addWorksheet("Exceptions");

        sheet.columns = [
          { header: "Sage Account", key: "SageAccount", width: 15 },
          { header: "Company ID", key: "CompanyID", width: 30 },
          { header: "Filter", key: "Filter", width: 18 },
          { header: "Status", key: "Status", width: 20 },
          { header: "Error No", key: "ErrorNo", width: 15 },
          { header: "Description", key: "Description", width: 50 },
          { header: "Description 2", key: "Description2", width: 50 }
        ];

        for (const row of exceptionRows) {
          sheet.addRow({
            SageAccount: row.sageAccountNumber || '',
            CompanyID: row.companyId,
            Filter: row.queryName,
            Status: row.status,
            ErrorNo: row.errorNumber,
            Description: row.description,
            Description2: row.description2
          });
        }

        const filename = `Intacct-Exceptions-${queryLabel}-${dateStr}.xlsx`;
        res.setHeader("Content-Type", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        res.setHeader("Content-Disposition", `attachment; filename="${filename}"`);
        return workbook.xlsx.write(res).then(() => res.end());
      }

      // CSV format (default)
      const csvData = exceptionRows.map(row => ({
        'Sage Account': row.sageAccountNumber || '',
        'Company ID': row.companyId,
        'Filter': row.queryName,
        'Status': row.status,
        'Error No': row.errorNumber,
        'Description': row.description,
        'Description 2': row.description2
      }));

      const fields = ['Sage Account', 'Company ID', 'Filter', 'Status', 'Error No', 'Description', 'Description 2'];
      const csvParser = new Json2CsvParser({ fields });
      const csv = csvParser.parse(csvData);

      const filename = `Intacct-Exceptions-${queryLabel}-${dateStr}.csv`;
      res.setHeader('Content-Type', 'text/csv');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      return res.send(csv);

    } else {
      return res.status(404).send("Unknown download type. Use 'detailed', 'summary', or 'exceptions'.");
    }
  } catch (e) {
    logger.error(`[Download] Error generating export: ${e.message}`);
    return res.status(500).send(`Error generating export: ${e.message}`);
  }
});

export default router;
