// routes/notifications.js - Notification configuration API
import express from 'express';
import {
  getAllNotificationConfigs,
  getNotificationConfig,
  createNotificationConfig,
  updateNotificationConfig,
  deleteNotificationConfig,
  getAllSchedules
} from '../services/database.js';
import { sendTestNotification } from '../services/notification.js';
import logger from '../services/logger.js';

const router = express.Router();

// Get all notification configurations (secrets masked)
router.get('/api/notifications/configs', (req, res) => {
  try {
    const configs = getAllNotificationConfigs();
    res.json({ success: true, configs });
  } catch (error) {
    logger.error('Get notification configs error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get single notification configuration (secrets masked)
router.get('/api/notifications/configs/:id', (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const config = getNotificationConfig(id);

    if (!config) {
      return res.status(404).json({ success: false, error: 'Configuration not found' });
    }

    res.json({ success: true, config });
  } catch (error) {
    logger.error('Get notification config error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Create notification configuration
router.post('/api/notifications/configs', (req, res) => {
  try {
    const id = createNotificationConfig(req.body);
    res.json({ success: true, id });
  } catch (error) {
    logger.error('Create notification config error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Update notification configuration
router.put('/api/notifications/configs/:id', (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    updateNotificationConfig(id, req.body);
    res.json({ success: true });
  } catch (error) {
    logger.error('Update notification config error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Delete notification configuration
router.delete('/api/notifications/configs/:id', (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);

    // Check if any schedules are linked to this notification config
    const schedules = getAllSchedules();
    const linkedSchedules = schedules.filter(s => s.notificationConfigId === id);
    if (linkedSchedules.length > 0) {
      const names = linkedSchedules.map(s => s.name).join(', ');
      return res.status(400).json({
        success: false,
        error: `Cannot delete: this configuration is used by schedule(s): ${names}`
      });
    }

    const deleted = deleteNotificationConfig(id);

    if (!deleted) {
      return res.status(404).json({ success: false, error: 'Configuration not found' });
    }

    res.json({ success: true });
  } catch (error) {
    logger.error('Delete notification config error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Test saved notification configuration (optionally with updated form values)
router.post('/api/notifications/configs/:id/test', async (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const storedConfig = getNotificationConfig(id, true);

    if (!storedConfig) {
      return res.status(404).json({ success: false, error: 'Configuration not found' });
    }

    // If form values are provided, merge them with stored config (keeping stored secrets)
    let testConfig = storedConfig;
    if (req.body && Object.keys(req.body).length > 0) {
      const formData = req.body;
      testConfig = {
        ...storedConfig,
        name: formData.name || storedConfig.name,
        recipients: formData.recipients !== undefined ? formData.recipients : storedConfig.recipients,
        cc: formData.cc !== undefined ? formData.cc : storedConfig.cc,
        bcc: formData.bcc !== undefined ? formData.bcc : storedConfig.bcc
      };

      // Merge SMTP settings if provided (keep stored password)
      if (formData.smtp && storedConfig.smtp) {
        testConfig.smtp = {
          ...storedConfig.smtp,
          host: formData.smtp.host || storedConfig.smtp.host,
          port: formData.smtp.port || storedConfig.smtp.port,
          security: formData.smtp.security || storedConfig.smtp.security,
          ignoreTLS: formData.smtp.ignoreTLS !== undefined ? formData.smtp.ignoreTLS : storedConfig.smtp.ignoreTLS,
          username: formData.smtp.username || storedConfig.smtp.username,
          from: formData.smtp.from || storedConfig.smtp.from,
          fromName: formData.smtp.fromName !== undefined ? formData.smtp.fromName : storedConfig.smtp.fromName
          // password kept from storedConfig
        };
      }

      // Merge MS Graph settings if provided (keep stored clientSecret)
      if (formData.msgraph && storedConfig.msgraph) {
        testConfig.msgraph = {
          ...storedConfig.msgraph,
          tenantId: formData.msgraph.tenantId || storedConfig.msgraph.tenantId,
          clientId: formData.msgraph.clientId || storedConfig.msgraph.clientId,
          from: formData.msgraph.from || storedConfig.msgraph.from
          // clientSecret kept from storedConfig
        };
      }
    }

    await sendTestNotification(testConfig);
    res.json({ success: true, message: 'Test notification sent successfully' });
  } catch (error) {
    logger.error('Test notification config error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Test unsaved notification configuration (from form data)
router.post('/api/notifications/test', async (req, res) => {
  try {
    await sendTestNotification(req.body);
    res.json({ success: true, message: 'Test notification sent successfully' });
  } catch (error) {
    logger.error('Test notification error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

export default router;
