// routes/schedules.js - Schedule configuration API
import express from 'express';
import {
  getAllSchedules,
  getSchedule,
  createSchedule,
  updateSchedule,
  deleteSchedule,
  getAllNotificationConfigs
} from '../services/database.js';
import { stopScheduledJob } from '../services/scheduler.js';
import logger from '../services/logger.js';

const router = express.Router();

// Get all schedules
router.get('/api/schedules', (req, res) => {
  try {
    const schedules = getAllSchedules();
    res.json({ success: true, schedules });
  } catch (error) {
    logger.error('Get schedules error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get notification configs (for dropdown) - must be before :id route
router.get('/api/schedules/email-configs', (req, res) => {
  try {
    const configs = getAllNotificationConfigs();
    // Filter to only email providers (SMTP, Graph)
    const emailConfigs = configs.filter(c => c.provider === 'smtp' || c.provider === 'msgraph');
    res.json({ success: true, configs: emailConfigs });
  } catch (error) {
    logger.error('Get notification configs error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get single schedule
router.get('/api/schedules/:id', (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const schedule = getSchedule(id);

    if (!schedule) {
      return res.status(404).json({ success: false, error: 'Schedule not found' });
    }

    res.json({ success: true, schedule });
  } catch (error) {
    logger.error('Get schedule error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Create schedule
router.post('/api/schedules', (req, res) => {
  try {
    const id = createSchedule(req.body);
    const schedule = getSchedule(id);
    logger.info(`[Scheduler] Schedule created: "${req.body.name}" (ID: ${id}, next run: ${schedule?.nextRunAt || 'not scheduled'})`);
    res.json({ success: true, id });
  } catch (error) {
    logger.error('Create schedule error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Update schedule
router.put('/api/schedules/:id', (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    updateSchedule(id, req.body);
    const schedule = getSchedule(id);
    logger.info(`[Scheduler] Schedule updated: "${schedule?.name}" (ID: ${id}, enabled: ${schedule?.enabled}, next run: ${schedule?.nextRunAt || 'not scheduled'})`);
    res.json({ success: true });
  } catch (error) {
    logger.error('Update schedule error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Delete schedule
router.delete('/api/schedules/:id', (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const schedule = getSchedule(id);
    const deleted = deleteSchedule(id);

    if (!deleted) {
      return res.status(404).json({ success: false, error: 'Schedule not found' });
    }

    logger.info(`[Scheduler] Schedule deleted: "${schedule?.name}" (ID: ${id})`);
    res.json({ success: true });
  } catch (error) {
    logger.error('Delete schedule error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Stop a running scheduled job
router.post('/api/schedules/stop', (req, res) => {
  try {
    const result = stopScheduledJob();
    res.json({ success: true, message: result.message, reset: result.reset });
  } catch (error) {
    logger.error('Stop schedule error:', error.message);
    res.status(409).json({ success: false, error: error.message });
  }
});

// Toggle schedule enabled/disabled
router.post('/api/schedules/:id/toggle', (req, res) => {
  try {
    const id = parseInt(req.params.id, 10);
    const schedule = getSchedule(id);

    if (!schedule) {
      return res.status(404).json({ success: false, error: 'Schedule not found' });
    }

    const newEnabled = !schedule.enabled;
    updateSchedule(id, { enabled: newEnabled });
    const updated = getSchedule(id);
    logger.info(`[Scheduler] Schedule ${newEnabled ? 'enabled' : 'disabled'}: "${schedule.name}" (ID: ${id}, next run: ${updated?.nextRunAt || 'not scheduled'})`);
    res.json({ success: true, enabled: newEnabled });
  } catch (error) {
    logger.error('Toggle schedule error:', error.message);
    res.status(500).json({ success: false, error: error.message });
  }
});

export default router;
