// routes/upload.js - Company management (API-only, no CSV upload)
import express from "express";
import {
  replaceAllCompanies,
  getAllCompanies,
  getAllCompaniesWithDetails,
  getCompanyCount,
  getCompaniesUploadedAt,
  addCompany,
  removeCompany,
  clearCompanies,
  clearApiUsageRecords,
  getAllSkipCompanies,
  getSkipCompanyCount,
  addSkipCompany,
  removeSkipCompany,
  getDefaultCompany,
  insertJobError,
  clearJobErrors
} from "../services/database.js";
import { getFetchOptionsWithProxy } from "../services/proxy.js";

// Query name used for API import errors
const API_IMPORT_QUERY = 'API_IMPORT';

const router = express.Router();

// GET /upload/status - Check if companies are loaded in database
router.get("/upload/status", (req, res) => {
  try {
    const count = getCompanyCount();
    const uploadedAt = getCompaniesUploadedAt();

    if (count > 0) {
      return res.json({
        ok: true,
        exists: true,
        count,
        uploadedAt
      });
    } else {
      return res.json({ ok: true, exists: false });
    }
  } catch (e) {
    return res.json({ ok: true, exists: false });
  }
});

// GET /upload/companies - List all company IDs
router.get("/upload/companies", (req, res) => {
  try {
    const companies = getAllCompanies();
    return res.json({ ok: true, companies });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// GET /upload/companies/details - List all companies with sage_account_number
router.get("/upload/companies/details", (req, res) => {
  try {
    const companies = getAllCompaniesWithDetails();
    return res.json({ ok: true, companies });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// POST /upload/companies - Add a single company
router.post("/upload/companies", express.json(), (req, res) => {
  const { company_id, sage_account_number } = req.body || {};
  if (!company_id || !String(company_id).trim()) {
    return res.status(400).json({ ok: false, message: "company_id is required" });
  }

  try {
    const id = String(company_id).trim();
    const sageAccNum = sage_account_number ? String(sage_account_number).trim() : null;
    const added = addCompany(id, sageAccNum);
    if (added) {
      return res.json({ ok: true, message: `Company '${id}' added`, count: getCompanyCount() });
    } else {
      return res.status(400).json({ ok: false, message: `Failed to add company '${id}'` });
    }
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// DELETE /upload/companies/:companyId - Remove a single company
router.delete("/upload/companies/:companyId", (req, res) => {
  const companyId = req.params.companyId;
  const defaultCompany = getDefaultCompany();

  // Prevent deletion of default company
  if (companyId === defaultCompany.company_id) {
    return res.status(400).json({ ok: false, message: `Cannot remove default company '${companyId}'` });
  }

  try {
    const removed = removeCompany(companyId);
    if (removed) {
      return res.json({ ok: true, message: `Company '${companyId}' removed`, count: getCompanyCount() });
    } else {
      return res.status(404).json({ ok: false, message: `Company '${companyId}' not found` });
    }
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// DELETE /upload/clear-all - Remove all companies (except default) and all usage data
router.delete("/upload/clear-all", (req, res) => {
  try {
    const usageCleared = clearApiUsageRecords(null);
    const companiesCleared = clearCompanies();
    return res.json({
      ok: true,
      message: `Cleared ${companiesCleared} companies and ${usageCleared} usage records (default company retained)`
    });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// POST /upload/from-api - Load companies from an external API URL
// Expected JSON format: { "companies": [{ "sage_account_number": "XXX", "company_id": "Company Name" }, ...] }
router.post("/upload/from-api", express.json(), async (req, res) => {
  const { apiUrl, apiToken } = req.body || {};

  if (!apiUrl || !String(apiUrl).trim()) {
    return res.status(400).json({ ok: false, message: "apiUrl is required" });
  }

  try {
    // Build request options with optional Authorization header and proxy
    let fetchOptions = {};
    if (apiToken && String(apiToken).trim()) {
      fetchOptions.headers = {
        'Authorization': `Bearer ${apiToken.trim()}`
      };
    }
    fetchOptions = getFetchOptionsWithProxy(fetchOptions);

    const response = await fetch(apiUrl, fetchOptions);

    if (!response.ok) {
      return res.status(400).json({
        ok: false,
        message: `API returned ${response.status}: ${response.statusText}`
      });
    }

    const data = await response.json();

    // Check for companies array in response
    if (!data.companies || !Array.isArray(data.companies)) {
      return res.status(400).json({
        ok: false,
        message: "API response must contain a 'companies' array"
      });
    }

    // Parse companies - expecting objects with sage_account_number and company_id
    const allCompanies = data.companies
      .map(c => {
        // Handle both object format and legacy string format
        if (typeof c === 'object' && c !== null) {
          return {
            sage_account_number: c.sage_account_number || c.SAGEACCNUM || null,
            company_id: c.company_id || c.INTACCT_COMPANY_ID || null
          };
        } else if (typeof c === 'string') {
          // Legacy format: just company ID string
          return {
            sage_account_number: null,
            company_id: c
          };
        }
        return null;
      })
      .filter(c => c !== null);

    // Separate valid companies from those with missing company_id
    const validCompanies = [];
    const missingCompanyId = [];

    for (const c of allCompanies) {
      const companyId = c.company_id ? String(c.company_id).trim() : '';
      const isNullOrBlank = !companyId || companyId.toUpperCase() === 'NULL';

      if (isNullOrBlank) {
        // Track companies with missing company_id (use sage_account_number as identifier)
        if (c.sage_account_number) {
          missingCompanyId.push({
            sage_account_number: String(c.sage_account_number).trim()
          });
        }
      } else {
        validCompanies.push({
          sage_account_number: c.sage_account_number ? String(c.sage_account_number).trim() : null,
          company_id: companyId
        });
      }
    }

    // Sort valid companies by company_id
    validCompanies.sort((a, b) =>
      a.company_id.localeCompare(b.company_id, 'en', { sensitivity: 'base', numeric: true })
    );

    if (validCompanies.length === 0) {
      return res.status(400).json({
        ok: false,
        message: "No valid companies found in API response (company_id is required)"
      });
    }

    const inserted = replaceAllCompanies(validCompanies);

    // Log companies with missing company_id to job_errors
    // Clear previous API_IMPORT errors first
    clearJobErrors(API_IMPORT_QUERY);

    const now = new Date().toISOString();
    for (const company of missingCompanyId) {
      insertJobError({
        company_id: company.sage_account_number,
        query_name: API_IMPORT_QUERY,
        error_type: 'MISSING_COMPANY_ID',
        error_number: null,
        description: 'Missing Intacct company_id in source data',
        description2: 'Add company_id to source system or add to exceptions',
        page: 1,
        recorded_at: now
      });
    }

    const missingCount = missingCompanyId.length;
    let message = `${inserted} companies loaded from API (includes default Datel Group)`;
    if (missingCount > 0) {
      message += `. ${missingCount} entries have missing company_id (see Error Summary)`;
    }

    return res.json({
      ok: true,
      message,
      count: inserted,
      missingCompanyIdCount: missingCount
    });
  } catch (e) {
    return res.status(500).json({
      ok: false,
      message: `Failed to load from API: ${e.message}`
    });
  }
});

// ── Skip Companies (Exceptions) ──

// GET /upload/skip-companies - List all skipped company IDs
router.get("/upload/skip-companies", (req, res) => {
  try {
    const companies = getAllSkipCompanies();
    return res.json({ ok: true, companies, count: companies.length });
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// POST /upload/skip-companies - Add a company to the skip list
router.post("/upload/skip-companies", express.json(), (req, res) => {
  const { company_id } = req.body || {};
  if (!company_id || !String(company_id).trim()) {
    return res.status(400).json({ ok: false, message: "company_id is required" });
  }

  try {
    const id = String(company_id).trim();
    const added = addSkipCompany(id);
    const count = getSkipCompanyCount();
    if (added) {
      return res.json({ ok: true, message: `'${id}' added to exceptions`, count });
    } else {
      return res.json({ ok: true, message: `'${id}' is already in exceptions`, count });
    }
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

// DELETE /upload/skip-companies/:companyId - Remove a company from the skip list
router.delete("/upload/skip-companies/:companyId", (req, res) => {
  const companyId = req.params.companyId;
  try {
    const removed = removeSkipCompany(companyId);
    const count = getSkipCompanyCount();
    if (removed) {
      return res.json({ ok: true, message: `'${companyId}' removed from exceptions`, count });
    } else {
      return res.status(404).json({ ok: false, message: `'${companyId}' not found in exceptions` });
    }
  } catch (e) {
    return res.status(500).json({ ok: false, message: e.message });
  }
});

export default router;
