// server.js v5.0 - Refactored with routes and services
import path from "path";
import fs from "fs";
import { fileURLToPath } from "url";
import dotenv from "dotenv";

// Load .env from project root (parent of source folder) - must be before other imports
const __filename_init = fileURLToPath(import.meta.url);
const __dirname_init = path.dirname(__filename_init);
dotenv.config({ path: path.join(__dirname_init, "..", "..", ".env") });

import express from "express";
import cors from "cors";

import { config, DATA_DIR, setDatabaseModule } from "./services/config.js";
import * as databaseModule from "./services/database.js";
import { initDatabase, closeDatabase, getApiUsageCount } from "./services/database.js";
import logger from "./services/logger.js";

// Import route handlers
import uploadRouter from "./routes/upload.js";
import credentialsRouter from "./routes/credentials.js";
import jobRouter from "./routes/job.js";
import dataRouter from "./routes/data.js";
import configRouter from "./routes/config.js";
import aboutRouter from "./routes/about.js";
import logsRouter from "./routes/logs.js";
import notificationsRouter from "./routes/notifications.js";
import schedulesRouter from "./routes/schedules.js";
import { startScheduler, stopScheduler } from "./services/scheduler.js";
import { testCredentials } from "./services/intacct.js";
import { authMiddleware, isAuthEnabled, login, logout } from "./middleware/auth.js";

// Get version from package.json
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const packageJson = JSON.parse(fs.readFileSync(path.join(__dirname, "..", "package.json"), "utf8"));
const VERSION = packageJson.version;

const app = express();
app.use(cors());
app.use(express.json());

// Auth: login/logout API routes (before auth middleware)
app.post("/api/auth/login", (req, res) => {
  const { key } = req.body;
  if (!key) return res.status(400).json({ ok: false, error: "Access key is required" });
  const token = login(key);
  if (token) {
    res.cookie("intacct_session", token, { httpOnly: true, sameSite: "lax", maxAge: 7 * 24 * 60 * 60 * 1000 });
    return res.json({ ok: true });
  }
  return res.status(401).json({ ok: false, error: "Invalid access key" });
});

app.get("/api/auth/status", (req, res) => {
  res.json({ authEnabled: isAuthEnabled() });
});

app.post("/api/auth/logout", (req, res) => {
  const cookies = req.headers.cookie?.split(";").reduce((acc, c) => {
    const [k, ...v] = c.trim().split("=");
    acc[k] = v.join("=");
    return acc;
  }, {}) || {};
  const token = cookies["intacct_session"];
  if (token) logout(token);
  res.clearCookie("intacct_session");
  res.json({ ok: true });
});

// Auth middleware - must be before static files and routes
app.use(authMiddleware);

app.use(express.static(path.join(process.cwd(), "public")));

logger.info(`Version: ${VERSION}`);
logger.info(`DATA_DIR: ${DATA_DIR}`);
logger.info(`LOG_DIR: ${logger.getLogsDirectory()}`);
if (isAuthEnabled()) {
  logger.info("AUTH_KEY is set - authentication enabled");
} else {
  logger.info("AUTH_KEY not set - authentication disabled");
}

// Version endpoint
app.get("/version", (req, res) => {
  res.json({ version: VERSION });
});

// Mount routes
app.use(uploadRouter);
app.use(credentialsRouter);
app.use(jobRouter);
app.use(dataRouter);
app.use("/config", configRouter);
app.use(aboutRouter);
app.use(logsRouter);
app.use(notificationsRouter);
app.use(schedulesRouter);

// Check Intacct credentials on startup
async function checkCredentialsOnStartup() {
  try {
    const credentials = databaseModule.getAllCredentials();
    if (!credentials || !credentials.senderId || !credentials.senderPassword ||
        !credentials.userId || !credentials.userPassword) {
      logger.info('No Intacct credentials configured yet - skipping validation');
      return;
    }

    // Use Datel Group for credential validation - a known company we always have access to
    const testCompanyId = 'Datel Group';
    logger.info(`Validating Intacct credentials with company "${testCompanyId}"...`);

    const result = await testCredentials({
      senderId: credentials.senderId,
      senderPassword: credentials.senderPassword,
      userId: credentials.userId,
      userPassword: credentials.userPassword,
      testCompanyId
    });

    if (result.ok) {
      logger.info(`Intacct credentials validated: ${result.message}`);
    } else {
      logger.warn(`*** Intacct credential check failed: ${result.message} ***`);
      logger.warn('Scheduled jobs may fail until credentials are corrected in Configuration.');
    }
  } catch (e) {
    logger.error(`Error checking credentials on startup: ${e.message}`);
  }
}

// Initialize database and start server
async function startServer() {
  try {
    await initDatabase();
    logger.info('SQLite database initialized');

    // Connect database module to config service
    setDatabaseModule(databaseModule);

    // Check credentials (non-blocking - just logs warning if invalid)
    await checkCredentialsOnStartup();

    // Start the scheduler service
    startScheduler();
  } catch (e) {
    logger.error(`Failed to initialize database: ${e.message}`);
  }

  const PORT = process.env.PORT || 5050;
  app.listen(PORT, () => {
    console.log(`Web UI ready on http://localhost:${PORT}`);
  });
}

startServer();

// Graceful shutdown - close database and stop scheduler on process exit
function gracefulShutdown(signal) {
  logger.info(`${signal} received, shutting down gracefully...`);
  stopScheduler();
  closeDatabase();
  process.exit(0);
}

process.on('SIGTERM', () => gracefulShutdown('SIGTERM'));
process.on('SIGINT', () => gracefulShutdown('SIGINT'));
