// services/config.js - Configuration management (database-backed)
import path from "path";
import fs from "fs";
import logger from "./logger.js";

// Default settings (key/value pairs)
const DEFAULT_SETTINGS = {
  url: "https://api.intacct.com/ia/xml/xmlgw.phtml",
  pageSize: "1000",
  fields: "PARTNERID,DOCCONTROLID,N_TRANS,CREATED_DATE,CREATED_TIME,STATUS",
  baseStartDate: "2025-08-01",
  companyIdColumn: "Company ID",
  includeCurrentMonth: "false",
  runAllQueries: "true",
  forceFullRefresh: "false"
};

// Default queries (query text is auto-generated at runtime from prefix + baseStartDate)
const DEFAULT_QUERIES = [
  {
    sort_order: 1,
    name: "ExcelConnect",
    prefix: "EC_"
  },
  {
    sort_order: 99,
    name: "FusionUsage",
    prefix: "FUSAPI_"
  }
];

// Legacy DEFAULT_CONFIG for backward compatibility with routes
const DEFAULT_CONFIG = {
  intacct: {
    url: DEFAULT_SETTINGS.url,
    pageSize: parseInt(DEFAULT_SETTINGS.pageSize, 10),
    fields: DEFAULT_SETTINGS.fields,
    baseStartDate: DEFAULT_SETTINGS.baseStartDate,
    companyIdColumn: DEFAULT_SETTINGS.companyIdColumn,
    includeCurrentMonth: DEFAULT_SETTINGS.includeCurrentMonth === "true",
    runAllQueries: DEFAULT_SETTINGS.runAllQueries === "true",
    forceFullRefresh: DEFAULT_SETTINGS.forceFullRefresh === "true",
    queries: {
      "1": { name: "ExcelConnect", prefix: "EC_" },
      "99": { name: "FusionUsage", prefix: "FUSAPI_" }
    }
  }
};

// Database reference - set after database is initialized
let dbModule = null;

/**
 * Set database module reference (called after database is initialized)
 */
export function setDatabaseModule(module) {
  dbModule = module;
}


/**
 * Load config from database (normalized tables)
 */
function loadConfigFromDb() {
  if (!dbModule || !dbModule.isDatabaseReady()) {
    return null;
  }

  // Get all settings
  const settings = dbModule.getAllSettings();
  if (Object.keys(settings).length === 0) {
    return null;
  }

  // Get all queries
  const queriesArray = dbModule.getAllQueries();

  // Convert queries array to object format { "order": { name, prefix } }
  const queriesObj = {};
  for (const q of queriesArray) {
    queriesObj[String(q.sort_order)] = {
      name: q.name,
      prefix: q.prefix || ""
    };
  }

  const baseStartDate = settings.baseStartDate || DEFAULT_SETTINGS.baseStartDate;

  // Build config object in expected format
  return {
    intacct: {
      url: settings.url || DEFAULT_SETTINGS.url,
      pageSize: parseInt(settings.pageSize, 10) || parseInt(DEFAULT_SETTINGS.pageSize, 10),
      fields: settings.fields || DEFAULT_SETTINGS.fields,
      baseStartDate,
      companyIdColumn: settings.companyIdColumn || DEFAULT_SETTINGS.companyIdColumn,
      includeCurrentMonth: settings.includeCurrentMonth === "true",
      runAllQueries: settings.runAllQueries === "true",
      forceFullRefresh: settings.forceFullRefresh === "true",
      companiesSource: settings.companiesSource || "csv",
      companiesApiUrl: settings.companiesApiUrl || "",
      queries: queriesObj
    }
  };
}

/**
 * Save config to database (normalized tables)
 */
function saveConfigToDb(configData) {
  if (!dbModule || !dbModule.isDatabaseReady()) {
    return;
  }

  // Save settings
  const baseStartDate = configData.intacct?.baseStartDate || DEFAULT_SETTINGS.baseStartDate;
  const settingsToSave = {
    url: configData.intacct?.url || DEFAULT_SETTINGS.url,
    pageSize: String(configData.intacct?.pageSize || DEFAULT_SETTINGS.pageSize),
    fields: configData.intacct?.fields || DEFAULT_SETTINGS.fields,
    baseStartDate,
    companyIdColumn: configData.intacct?.companyIdColumn || DEFAULT_SETTINGS.companyIdColumn,
    includeCurrentMonth: String(configData.intacct?.includeCurrentMonth ?? false),
    runAllQueries: String(configData.intacct?.runAllQueries ?? true),
    forceFullRefresh: String(configData.intacct?.forceFullRefresh ?? false),
    companiesSource: configData.intacct?.companiesSource || "csv",
    companiesApiUrl: configData.intacct?.companiesApiUrl || ""
  };

  dbModule.setMultipleSettings(settingsToSave);

  // Save queries - clear and re-insert
  const queries = configData.intacct?.queries || {};
  dbModule.deleteAllQueries();

  for (const [order, q] of Object.entries(queries)) {
    dbModule.saveQuery({
      sort_order: parseInt(order, 10),
      name: q.name,
      prefix: q.prefix || ""
    });
  }

  logger.info("[Config] Saved to database");
}

/**
 * Load config from database, initializes defaults if empty
 */
function loadConfig() {
  // Try database if available
  if (dbModule && dbModule.isDatabaseReady()) {
    const needsSettings = !dbModule.hasSettings();
    const needsQueries = !dbModule.hasQueries();

    if (needsSettings && needsQueries) {
      // Fresh database — seed everything
      saveConfigToDb(DEFAULT_CONFIG);
      logger.info("[Config] Initialized default config in database");
    } else if (needsQueries) {
      // Settings exist but queries missing (e.g., after schema migration) — seed default queries
      const baseStartDate = dbModule.getSetting("baseStartDate") || DEFAULT_SETTINGS.baseStartDate;
      for (const q of DEFAULT_QUERIES) {
        dbModule.saveQuery({
          sort_order: q.sort_order,
          name: q.name,
          prefix: q.prefix
        });
      }
      logger.info("[Config] Seeded default queries in database");
    }

    const dbConfig = loadConfigFromDb();
    if (dbConfig) {
      return dbConfig;
    }
  }

  // Return defaults
  return DEFAULT_CONFIG;
}

/**
 * Save config - saves to database
 */
function saveConfig(configData) {
  // Save to database if available
  if (dbModule && dbModule.isDatabaseReady()) {
    saveConfigToDb(configData);
  }
}

// Initial config load (for backward compatibility)
const config = loadConfig();

// Function to get fresh config (use this when you need current values)
function getConfig() {
  return loadConfig();
}

// Derived paths (use environment variables or default to cwd)
const DATA_DIR = process.env.DATA_DIR || path.join(process.cwd(), "data");

// Ensure data directory exists
if (!fs.existsSync(DATA_DIR)) {
  fs.mkdirSync(DATA_DIR, { recursive: true });
}

export {
  config,
  getConfig,
  saveConfig,
  loadConfig,
  DEFAULT_CONFIG,
  DEFAULT_SETTINGS,
  DEFAULT_QUERIES,
  DATA_DIR
};
