// services/database.js - Barrel re-export for all database modules
//
// All database functionality is split into focused sub-modules under ./db/:
//   core.js           - Schema definitions, encryption, lifecycle, migration
//   credentials.js    - Credential CRUD with encryption
//   settings.js       - Settings key/value CRUD
//   queries.js        - Query definition CRUD
//   api-usage.js      - API usage record operations
//
// This file re-exports everything so downstream imports don't need to change.

// Core - database lifecycle, schema, encryption
export {
  initDatabase,
  reloadDatabase,
  getDatabase,
  closeDatabase,
  isDatabaseReady,
  isEncryptionConfigured,
  saveDatabase
} from './db/core.js';

// Credentials
export {
  setCredential,
  getCredential,
  getAllCredentials,
  deleteCredential,
  deleteAllCredentials,
  hasCredentials
} from './db/credentials.js';

// Settings
export {
  getSetting,
  setSetting,
  getAllSettings,
  setMultipleSettings,
  hasSettings
} from './db/settings.js';

// Queries
export {
  getAllQueries,
  getQueryById,
  saveQuery,
  deleteQuery,
  getQueryCount,
  hasQueries,
  deleteAllQueries
} from './db/queries.js';

// Companies
export {
  replaceAllCompanies,
  getAllCompanies,
  getAllCompaniesWithDetails,
  getCompanyCount,
  hasCompanies,
  getCompaniesUploadedAt,
  clearCompanies,
  addCompany,
  removeCompany,
  ensureDefaultCompany,
  getDefaultCompany
} from './db/companies.js';

// Job Errors
export {
  insertJobError,
  clearJobErrors,
  getJobErrors,
  getJobErrorCount,
  getJobErrorSummary,
  getFilteredJobErrors,
  getFilteredJobErrorSummary,
  getFilteredJobErrorCount
} from './db/job-errors.js';

// API Usage
export {
  clearCompanyQueryRecords,
  clearCompanyQueryRecordsFromDate,
  getLastFetchedDate,
  insertApiUsageRecords,
  getApiUsageRecords,
  getApiUsageMonthly,
  getApiUsageMonthlyByFilter,
  getApiUsageSummary,
  clearApiUsageRecords,
  getApiUsageCompanies,
  getApiUsageDateRange,
  getApiUsageCount,
  getLastFetchedAtByQuery,
  getCompaniesWithNoData
} from './db/api-usage.js';

// Notification Configs
export {
  createNotificationConfig,
  updateNotificationConfig,
  deleteNotificationConfig,
  getNotificationConfig,
  getAllNotificationConfigs,
  getNotificationConfigCount
} from './db/notification-configs.js';

// Schedules
export {
  createSchedule,
  updateSchedule,
  updateScheduleAfterRun,
  deleteSchedule,
  getSchedule,
  getAllSchedules,
  getDueSchedules,
  getScheduleCount,
  calculateNextRun,
  resetStaleRunningSchedules
} from './db/schedules.js';

// Skip Companies (Exceptions)
export {
  getAllSkipCompanies,
  getSkipCompanyCount,
  addSkipCompany,
  removeSkipCompany,
  isSkipCompany
} from './db/skip-companies.js';

// Default export for backward compatibility
import { initDatabase, reloadDatabase, getDatabase, closeDatabase, isDatabaseReady, isEncryptionConfigured } from './db/core.js';
import { setCredential, getCredential, getAllCredentials, deleteCredential, deleteAllCredentials, hasCredentials } from './db/credentials.js';
import { getSetting, setSetting, getAllSettings, setMultipleSettings, hasSettings } from './db/settings.js';
import { getAllQueries, getQueryById, saveQuery, deleteQuery, getQueryCount, hasQueries, deleteAllQueries } from './db/queries.js';
import { clearCompanyQueryRecords, clearCompanyQueryRecordsFromDate, getLastFetchedDate, insertApiUsageRecords, getApiUsageRecords, getApiUsageMonthly, getApiUsageMonthlyByFilter, getApiUsageSummary, clearApiUsageRecords, getApiUsageCompanies, getApiUsageDateRange, getApiUsageCount, getLastFetchedAtByQuery, getCompaniesWithNoData } from './db/api-usage.js';
import { replaceAllCompanies, getAllCompanies, getAllCompaniesWithDetails, getCompanyCount, hasCompanies, getCompaniesUploadedAt, clearCompanies, addCompany, removeCompany, ensureDefaultCompany, getDefaultCompany } from './db/companies.js';
import { insertJobError, clearJobErrors, getJobErrors, getJobErrorCount, getJobErrorSummary, getFilteredJobErrors, getFilteredJobErrorSummary, getFilteredJobErrorCount } from './db/job-errors.js';
import { createNotificationConfig, updateNotificationConfig, deleteNotificationConfig, getNotificationConfig, getAllNotificationConfigs, getNotificationConfigCount } from './db/notification-configs.js';
import { createSchedule, updateSchedule, updateScheduleAfterRun, deleteSchedule, getSchedule, getAllSchedules, getDueSchedules, getScheduleCount, calculateNextRun, resetStaleRunningSchedules } from './db/schedules.js';
import { getAllSkipCompanies, getSkipCompanyCount, addSkipCompany, removeSkipCompany, isSkipCompany } from './db/skip-companies.js';

export default {
  initDatabase,
  reloadDatabase,
  getDatabase,
  closeDatabase,
  isDatabaseReady,
  isEncryptionConfigured,
  // Credentials
  setCredential,
  getCredential,
  getAllCredentials,
  deleteCredential,
  deleteAllCredentials,
  hasCredentials,
  // Settings
  getSetting,
  setSetting,
  getAllSettings,
  setMultipleSettings,
  hasSettings,
  // Queries
  getAllQueries,
  getQueryById,
  saveQuery,
  deleteQuery,
  getQueryCount,
  hasQueries,
  deleteAllQueries,
  // API Usage
  clearCompanyQueryRecords,
  clearCompanyQueryRecordsFromDate,
  getLastFetchedDate,
  insertApiUsageRecords,
  getApiUsageRecords,
  getApiUsageMonthly,
  getApiUsageMonthlyByFilter,
  getApiUsageSummary,
  clearApiUsageRecords,
  getApiUsageCompanies,
  getApiUsageDateRange,
  getApiUsageCount,
  getLastFetchedAtByQuery,
  getCompaniesWithNoData,
  // Companies
  replaceAllCompanies,
  getAllCompanies,
  getAllCompaniesWithDetails,
  getCompanyCount,
  hasCompanies,
  getCompaniesUploadedAt,
  clearCompanies,
  addCompany,
  removeCompany,
  ensureDefaultCompany,
  getDefaultCompany,
  // Job Errors
  insertJobError,
  clearJobErrors,
  getJobErrors,
  getJobErrorCount,
  getJobErrorSummary,
  getFilteredJobErrors,
  getFilteredJobErrorSummary,
  getFilteredJobErrorCount,
  // Notification Configs
  createNotificationConfig,
  updateNotificationConfig,
  deleteNotificationConfig,
  getNotificationConfig,
  getAllNotificationConfigs,
  getNotificationConfigCount,
  // Schedules
  createSchedule,
  updateSchedule,
  updateScheduleAfterRun,
  deleteSchedule,
  getSchedule,
  getAllSchedules,
  getDueSchedules,
  getScheduleCount,
  calculateNextRun,
  resetStaleRunningSchedules,
  // Skip Companies
  getAllSkipCompanies,
  getSkipCompanyCount,
  addSkipCompany,
  removeSkipCompany,
  isSkipCompany
};
