// db/api-usage.js - API usage record operations
import { getDatabase } from './core.js';

/**
 * Clear API usage records for a specific company and query
 * Used before re-fetching data to avoid stale/duplicate records
 */
export function clearCompanyQueryRecords(queryName, companyId) {
  const database = getDatabase();
  const result = database.prepare('DELETE FROM api_usage WHERE query_name = ? AND company_id = ?').run(queryName, companyId);
  return result.changes;
}

/**
 * Get the latest created_date for a specific company and query
 * Returns the date as YYYY-MM-DD string, or null if no records exist
 */
export function getLastFetchedDate(queryName, companyId) {
  const database = getDatabase();
  const row = database.prepare(
    'SELECT MAX(created_date) as max_date FROM api_usage WHERE query_name = ? AND company_id = ?'
  ).get(queryName, companyId);
  return row ? row.max_date : null;
}

/**
 * Clear API usage records for a specific company and query from a given date onward
 * Used in top-up mode to only clear the date range being re-fetched
 * @param {string} queryName
 * @param {string} companyId
 * @param {string} fromDate - ISO date (YYYY-MM-DD) to start clearing from (inclusive)
 */
export function clearCompanyQueryRecordsFromDate(queryName, companyId, fromDate) {
  const database = getDatabase();
  const result = database.prepare(
    'DELETE FROM api_usage WHERE query_name = ? AND company_id = ? AND created_date >= ?'
  ).run(queryName, companyId, fromDate);
  return result.changes;
}

/**
 * Insert multiple API usage records (bulk insert)
 * Returns the number of records actually inserted
 */
export function insertApiUsageRecords(records) {
  if (!records || records.length === 0) return 0;

  const database = getDatabase();

  const insertMany = database.transaction((recs) => {
    const stmt = database.prepare(
      `INSERT INTO api_usage
       (company_id, partner_id, doc_control_id, n_trans, created_date, created_time, status, query_name, fetched_at)
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`
    );
    let inserted = 0;
    for (const r of recs) {
      try {
        const result = stmt.run(
          r.company_id,
          r.partner_id || null,
          r.doc_control_id,
          r.n_trans || 0,
          r.created_date || null,
          r.created_time || null,
          r.status || null,
          r.query_name,
          r.fetched_at || new Date().toISOString()
        );
        inserted += result.changes;
      } catch (e) {
        console.warn(`[insertApiUsageRecords] Failed to insert record:`, e.message);
      }
    }
    return inserted;
  });

  return insertMany(records);
}

/**
 * Get API usage records with optional filters
 * Includes sage_account_number from companies table via LEFT JOIN
 * @param {Object} filters - { query_name, company_id, start_date, end_date, limit, offset }
 */
export function getApiUsageRecords(filters = {}) {
  const database = getDatabase();
  const conditions = [];
  const params = [];

  if (filters.query_name) {
    conditions.push('a.query_name = ?');
    params.push(filters.query_name);
  }
  if (filters.company_id) {
    conditions.push('a.company_id = ?');
    params.push(filters.company_id);
  }
  if (filters.start_date) {
    conditions.push('a.created_date >= ?');
    params.push(filters.start_date);
  }
  if (filters.end_date) {
    conditions.push('a.created_date <= ?');
    params.push(filters.end_date);
  }

  let sql = `
    SELECT a.*, c.sage_account_number
    FROM api_usage a
    LEFT JOIN companies c ON a.company_id = c.company_id
  `;
  if (conditions.length > 0) {
    sql += ' WHERE ' + conditions.join(' AND ');
  }
  sql += ' ORDER BY a.created_date DESC, a.created_time DESC';

  if (filters.limit) {
    sql += ' LIMIT ?';
    params.push(filters.limit);
    if (filters.offset) {
      sql += ' OFFSET ?';
      params.push(filters.offset);
    }
  }

  return database.prepare(sql).all(...params);
}

/**
 * Get monthly aggregation of API usage
 * @param {Object} filters - { query_name, exclude_current_month, start_date, end_date }
 */
export function getApiUsageMonthly(filters = {}) {
  const database = getDatabase();
  const conditions = [];
  const params = [];

  if (filters.query_name) {
    conditions.push('query_name = ?');
    params.push(filters.query_name);
  }

  // Exclude current month if requested
  if (filters.exclude_current_month) {
    const now = new Date();
    const currentMonth = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, '0')}`;
    conditions.push("substr(created_date, 1, 7) != ?");
    params.push(currentMonth);
  }

  // Date range filtering
  if (filters.start_date) {
    conditions.push('created_date >= ?');
    params.push(filters.start_date);
  }
  if (filters.end_date) {
    conditions.push('created_date <= ?');
    params.push(filters.end_date);
  }

  let sql = `
    SELECT
      substr(created_date, 1, 7) as month,
      SUM(n_trans) as total_transactions,
      COUNT(*) as record_count
    FROM api_usage
  `;
  if (conditions.length > 0) {
    sql += ' WHERE ' + conditions.join(' AND ');
  }
  sql += ' GROUP BY substr(created_date, 1, 7) ORDER BY month ASC';

  const rows = database.prepare(sql).all(...params);
  return rows.map(row => ({
    month: row.month,
    total_transactions: row.total_transactions || 0,
    record_count: row.record_count || 0
  }));
}

/**
 * Get monthly aggregation of API usage grouped by filter (query_name)
 * Returns rows like { month, query_name, record_count }
 * @param {Object} filters - { exclude_current_month, start_date, end_date }
 */
export function getApiUsageMonthlyByFilter(filters = {}) {
  const database = getDatabase();
  const conditions = [];
  const params = [];

  if (filters.exclude_current_month) {
    const now = new Date();
    const currentMonth = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, '0')}`;
    conditions.push("substr(created_date, 1, 7) != ?");
    params.push(currentMonth);
  }

  if (filters.start_date) {
    conditions.push('created_date >= ?');
    params.push(filters.start_date);
  }
  if (filters.end_date) {
    conditions.push('created_date <= ?');
    params.push(filters.end_date);
  }

  let sql = `
    SELECT
      substr(created_date, 1, 7) as month,
      query_name,
      COUNT(*) as record_count
    FROM api_usage
  `;
  if (conditions.length > 0) {
    sql += ' WHERE ' + conditions.join(' AND ');
  }
  sql += ' GROUP BY substr(created_date, 1, 7), query_name ORDER BY month ASC, query_name ASC';

  const rows = database.prepare(sql).all(...params);
  return rows.map(row => ({
    month: row.month,
    query_name: row.query_name || 'Other',
    record_count: row.record_count || 0
  }));
}

/**
 * Get summary by company and filter
 * Includes sage_account_number from companies table via LEFT JOIN
 * @param {string} queryName - Optional query name filter
 * @param {string} startDate - Optional start date (YYYY-MM-DD)
 * @param {string} endDate - Optional end date (YYYY-MM-DD)
 */
export function getApiUsageSummary(queryName = null, startDate = null, endDate = null) {
  const database = getDatabase();
  const conditions = [];
  const params = [];

  if (queryName) {
    conditions.push('a.query_name = ?');
    params.push(queryName);
  }
  if (startDate) {
    conditions.push('a.created_date >= ?');
    params.push(startDate);
  }
  if (endDate) {
    conditions.push('a.created_date <= ?');
    params.push(endDate);
  }

  let sql = `
    SELECT
      a.company_id,
      c.sage_account_number,
      a.query_name,
      SUM(a.n_trans) as total_transactions
    FROM api_usage a
    LEFT JOIN companies c ON a.company_id = c.company_id
  `;
  if (conditions.length > 0) {
    sql += ' WHERE ' + conditions.join(' AND ');
  }
  sql += ' GROUP BY a.company_id, c.sage_account_number, a.query_name ORDER BY a.company_id ASC, a.query_name ASC';

  const rows = database.prepare(sql).all(...params);
  return rows.map(row => ({
    company_id: row.company_id,
    sage_account_number: row.sage_account_number || '',
    query_name: row.query_name || '',
    total_trans: row.total_transactions || 0
  }));
}

/**
 * Clear API usage records
 * @param {string} queryName - Optional: only clear records for this query
 */
export function clearApiUsageRecords(queryName = null) {
  const database = getDatabase();
  let result;
  if (queryName) {
    result = database.prepare('DELETE FROM api_usage WHERE query_name = ?').run(queryName);
  } else {
    result = database.prepare('DELETE FROM api_usage').run();
  }
  return result.changes;
}

/**
 * Get distinct companies with usage data
 */
export function getApiUsageCompanies() {
  const database = getDatabase();
  const rows = database.prepare('SELECT DISTINCT company_id FROM api_usage ORDER BY company_id ASC').all();
  return rows.map(row => row.company_id);
}

/**
 * Get date range of stored API usage data
 */
export function getApiUsageDateRange() {
  const database = getDatabase();
  const row = database.prepare(`
    SELECT MIN(created_date) as min_date, MAX(created_date) as max_date
    FROM api_usage
  `).get();

  if (!row || !row.min_date) {
    return { min_date: null, max_date: null };
  }

  return {
    min_date: row.min_date,
    max_date: row.max_date
  };
}

/**
 * Get the last fetched_at timestamp per query name
 * Returns an object like { "ExcelConnect": "2026-01-29T12:18:05Z", ... }
 */
export function getLastFetchedAtByQuery() {
  const database = getDatabase();
  const rows = database.prepare(
    'SELECT query_name, MAX(fetched_at) as last_fetched_at FROM api_usage GROUP BY query_name'
  ).all();

  const map = {};
  for (const row of rows) {
    map[row.query_name] = row.last_fetched_at;
  }
  return map;
}

/**
 * Get companies that have no API usage data
 * Returns objects with company_id and sage_account_number
 * @param {string|null} queryName - Optional: only check for this query
 * @returns {Array} Array of { company_id, sage_account_number } objects with no data
 */
export function getCompaniesWithNoData(queryName = null) {
  const database = getDatabase();
  const params = [];

  let sql = `
    SELECT c.company_id, c.sage_account_number FROM companies c
    WHERE c.company_id NOT IN (
      SELECT DISTINCT company_id FROM api_usage
  `;
  if (queryName) {
    sql += ' WHERE query_name = ?';
    params.push(queryName);
  }
  sql += `) AND c.company_id NOT IN (
      SELECT company_id FROM skip_companies
    ) ORDER BY c.company_id ASC`;

  const rows = database.prepare(sql).all(...params);
  return rows.map(row => ({
    company_id: row.company_id,
    sage_account_number: row.sage_account_number || ''
  }));
}

/**
 * Get total record count
 * @param {string} queryName - Optional query name filter
 * @param {string} startDate - Optional start date (YYYY-MM-DD)
 * @param {string} endDate - Optional end date (YYYY-MM-DD)
 */
export function getApiUsageCount(queryName = null, startDate = null, endDate = null) {
  const database = getDatabase();
  const conditions = [];
  const params = [];

  if (queryName) {
    conditions.push('query_name = ?');
    params.push(queryName);
  }
  if (startDate) {
    conditions.push('created_date >= ?');
    params.push(startDate);
  }
  if (endDate) {
    conditions.push('created_date <= ?');
    params.push(endDate);
  }

  let sql = 'SELECT COUNT(*) as count FROM api_usage';
  if (conditions.length > 0) {
    sql += ' WHERE ' + conditions.join(' AND ');
  }

  const row = database.prepare(sql).get(...params);
  return row.count;
}
