// db/companies.js - Company list CRUD (loaded from API)
import { getDatabase } from './core.js';

// Default company that is always included (used for credential validation)
const DEFAULT_COMPANY = {
  sage_account_number: 'DATEL',
  company_id: 'Datel Group'
};

/**
 * Replace all companies with a new set (clear + bulk insert)
 * Always includes the default Datel Group company
 * @param {Array<{sage_account_number: string, company_id: string}>} companies - Array of company objects
 * @returns {number} Number of companies inserted
 */
export function replaceAllCompanies(companies) {
  const database = getDatabase();
  const now = new Date().toISOString();

  const insertMany = database.transaction((companyList) => {
    database.prepare('DELETE FROM companies').run();
    const stmt = database.prepare(
      'INSERT INTO companies (sage_account_number, company_id, uploaded_at) VALUES (?, ?, ?)'
    );

    let inserted = 0;

    // Always insert the default company first
    const defaultResult = stmt.run(DEFAULT_COMPANY.sage_account_number, DEFAULT_COMPANY.company_id, now);
    inserted += defaultResult.changes;

    // Insert all other companies (skip if company_id matches default)
    for (const company of companyList) {
      if (company.company_id && company.company_id !== DEFAULT_COMPANY.company_id) {
        const result = stmt.run(
          company.sage_account_number || null,
          company.company_id,
          now
        );
        inserted += result.changes;
      }
    }
    return inserted;
  });

  return insertMany(companies);
}

/**
 * Get all company IDs sorted alphabetically
 * @returns {string[]}
 */
export function getAllCompanies() {
  const database = getDatabase();
  const rows = database.prepare(
    'SELECT company_id FROM companies ORDER BY company_id ASC'
  ).all();
  return rows.map(row => row.company_id);
}

/**
 * Get all companies with full details (sage_account_number and company_id)
 * @returns {Array<{sage_account_number: string|null, company_id: string}>}
 */
export function getAllCompaniesWithDetails() {
  const database = getDatabase();
  return database.prepare(
    'SELECT sage_account_number, company_id FROM companies ORDER BY company_id ASC'
  ).all();
}

/**
 * Get the count of stored companies
 * @returns {number}
 */
export function getCompanyCount() {
  const database = getDatabase();
  const row = database.prepare('SELECT COUNT(*) as count FROM companies').get();
  return row.count;
}

/**
 * Check if any companies are stored
 * @returns {boolean}
 */
export function hasCompanies() {
  const database = getDatabase();
  const row = database.prepare('SELECT 1 FROM companies LIMIT 1').get();
  return !!row;
}

/**
 * Get the upload timestamp (all rows share the same batch timestamp)
 * @returns {string|null} ISO timestamp or null
 */
export function getCompaniesUploadedAt() {
  const database = getDatabase();
  const row = database.prepare('SELECT uploaded_at FROM companies LIMIT 1').get();
  return row ? row.uploaded_at : null;
}

/**
 * Add a single company
 * @param {string} companyId
 * @param {string|null} sageAccountNumber
 * @returns {boolean}
 */
export function addCompany(companyId, sageAccountNumber = null) {
  const database = getDatabase();
  const now = new Date().toISOString();
  const result = database.prepare(
    'INSERT INTO companies (sage_account_number, company_id, uploaded_at) VALUES (?, ?, ?)'
  ).run(sageAccountNumber, companyId, now);
  return result.changes > 0;
}

/**
 * Remove a single company by company_id
 * Prevents removal of the default company
 * @param {string} companyId
 * @returns {boolean}
 */
export function removeCompany(companyId) {
  // Prevent removal of default company
  if (companyId === DEFAULT_COMPANY.company_id) {
    return false;
  }
  const database = getDatabase();
  const result = database.prepare('DELETE FROM companies WHERE company_id = ?').run(companyId);
  return result.changes > 0;
}

/**
 * Clear all companies (except default)
 * @returns {number} Number of rows deleted
 */
export function clearCompanies() {
  const database = getDatabase();
  const result = database.prepare('DELETE FROM companies WHERE company_id != ?').run(DEFAULT_COMPANY.company_id);
  return result.changes;
}

/**
 * Ensure default company exists (called on startup)
 */
export function ensureDefaultCompany() {
  const database = getDatabase();
  const existing = database.prepare('SELECT 1 FROM companies WHERE company_id = ?').get(DEFAULT_COMPANY.company_id);
  if (!existing) {
    const now = new Date().toISOString();
    database.prepare(
      'INSERT INTO companies (sage_account_number, company_id, uploaded_at) VALUES (?, ?, ?)'
    ).run(DEFAULT_COMPANY.sage_account_number, DEFAULT_COMPANY.company_id, now);
  }
}

/**
 * Get the default company info
 * @returns {{sage_account_number: string, company_id: string}}
 */
export function getDefaultCompany() {
  return { ...DEFAULT_COMPANY };
}
