// db/core.js - Database lifecycle, schema definitions, encryption, and migration
import Database from 'better-sqlite3';
import path from 'path';
import fs from 'fs';
import crypto from 'crypto';
import logger from '../logger.js';

// Database directory - uses DATA_DIR env var or defaults to ./data
const DATA_DIR = process.env.DATA_DIR || path.join(process.cwd(), 'data');
const DB_FILE = path.join(DATA_DIR, 'intacct.db');

// Encryption configuration
const ALGORITHM = 'aes-256-gcm';
const KEY_LENGTH = 32;
const IV_LENGTH = 16;
const AUTH_TAG_LENGTH = 16;
const SALT_LENGTH = 32;
const PBKDF2_ITERATIONS = 100000;

let db = null;

// ============================================
// Schema Definitions - Single Source of Truth
// ============================================
// Each table defines its expected columns and CREATE SQL.
// On startup, existing tables are validated against these definitions.
// If columns don't match, the table is migrated (data preserved where possible).
// The credentials table is NEVER dropped - only missing columns are added.

const TABLE_SCHEMAS = {
  credentials: {
    create: `CREATE TABLE IF NOT EXISTS credentials (
      key TEXT PRIMARY KEY,
      salt TEXT NOT NULL,
      iv TEXT NOT NULL,
      auth_tag TEXT NOT NULL,
      encrypted_data TEXT NOT NULL
    )`,
    columns: ['key', 'salt', 'iv', 'auth_tag', 'encrypted_data'],
    preserveData: true  // Never drop credentials
  },
  settings: {
    create: `CREATE TABLE IF NOT EXISTS settings (
      key TEXT PRIMARY KEY,
      value TEXT NOT NULL
    )`,
    columns: ['key', 'value']
  },
  filters: {
    create: `CREATE TABLE IF NOT EXISTS filters (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      sort_order INTEGER NOT NULL,
      name TEXT NOT NULL,
      prefix TEXT DEFAULT '',
      created_at TEXT NOT NULL,
      updated_at TEXT NOT NULL
    )`,
    columns: ['id', 'sort_order', 'name', 'prefix', 'created_at', 'updated_at'],
    indexes: [
      `CREATE INDEX IF NOT EXISTS idx_filters_sort_order ON filters(sort_order)`
    ]
  },
  api_usage: {
    create: `CREATE TABLE IF NOT EXISTS api_usage (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      company_id TEXT NOT NULL,
      partner_id TEXT,
      doc_control_id TEXT NOT NULL,
      n_trans INTEGER DEFAULT 0,
      created_date TEXT,
      created_time TEXT,
      status TEXT,
      query_name TEXT NOT NULL,
      fetched_at TEXT NOT NULL
    )`,
    columns: ['id', 'company_id', 'partner_id', 'doc_control_id', 'n_trans', 'created_date', 'created_time', 'status', 'query_name', 'fetched_at'],
    indexes: [
      `CREATE INDEX IF NOT EXISTS idx_api_usage_company ON api_usage(company_id)`,
      `CREATE INDEX IF NOT EXISTS idx_api_usage_date ON api_usage(created_date)`,
      `CREATE INDEX IF NOT EXISTS idx_api_usage_query ON api_usage(query_name)`,
      `CREATE INDEX IF NOT EXISTS idx_api_usage_company_date ON api_usage(company_id, created_date)`
    ]
  },
  companies: {
    create: `CREATE TABLE IF NOT EXISTS companies (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      sage_account_number TEXT,
      company_id TEXT NOT NULL,
      uploaded_at TEXT NOT NULL
    )`,
    columns: ['id', 'sage_account_number', 'company_id', 'uploaded_at'],
    indexes: [
      `CREATE INDEX IF NOT EXISTS idx_companies_company_id ON companies(company_id)`,
      `CREATE INDEX IF NOT EXISTS idx_companies_sage_account ON companies(sage_account_number)`
    ]
  },
  job_errors: {
    create: `CREATE TABLE IF NOT EXISTS job_errors (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      company_id TEXT NOT NULL,
      query_name TEXT NOT NULL,
      error_type TEXT NOT NULL,
      error_number TEXT,
      description TEXT NOT NULL,
      description2 TEXT,
      page INTEGER DEFAULT 1,
      recorded_at TEXT NOT NULL
    )`,
    columns: ['id', 'company_id', 'query_name', 'error_type', 'error_number', 'description', 'description2', 'page', 'recorded_at'],
    indexes: [
      `CREATE INDEX IF NOT EXISTS idx_job_errors_company ON job_errors(company_id)`,
      `CREATE INDEX IF NOT EXISTS idx_job_errors_query ON job_errors(query_name)`
    ]
  },
  notification_configs: {
    create: `CREATE TABLE IF NOT EXISTS notification_configs (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      provider TEXT NOT NULL,
      host TEXT DEFAULT '',
      port INTEGER DEFAULT 0,
      security TEXT DEFAULT '',
      ignore_tls INTEGER DEFAULT 0,
      username TEXT DEFAULT '',
      from_name TEXT DEFAULT '',
      from_email TEXT DEFAULT '',
      tenant_id TEXT DEFAULT '',
      client_id TEXT DEFAULT '',
      user_key TEXT DEFAULT '',
      device TEXT DEFAULT '',
      priority TEXT DEFAULT '0',
      sound TEXT DEFAULT 'pushover',
      secret_salt TEXT DEFAULT '',
      secret_iv TEXT DEFAULT '',
      secret_auth_tag TEXT DEFAULT '',
      secret_data TEXT DEFAULT '',
      recipients TEXT DEFAULT '',
      cc TEXT DEFAULT '',
      bcc TEXT DEFAULT '',
      created_at TEXT NOT NULL,
      updated_at TEXT NOT NULL
    )`,
    columns: ['id', 'name', 'provider', 'host', 'port', 'security', 'ignore_tls', 'username', 'from_name', 'from_email', 'tenant_id', 'client_id', 'user_key', 'device', 'priority', 'sound', 'secret_salt', 'secret_iv', 'secret_auth_tag', 'secret_data', 'recipients', 'cc', 'bcc', 'created_at', 'updated_at']
  },
  schedules: {
    create: `CREATE TABLE IF NOT EXISTS schedules (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      enabled INTEGER DEFAULT 1,
      schedule_type TEXT NOT NULL,
      interval_minutes INTEGER DEFAULT 0,
      time_of_day TEXT DEFAULT '',
      days_of_week TEXT DEFAULT '',
      day_of_month INTEGER DEFAULT 0,
      cron_expression TEXT DEFAULT '',
      oneoff_datetime TEXT DEFAULT '',
      query_filter TEXT DEFAULT 'all',
      attachments TEXT DEFAULT 'detailed,summary,exceptions',
      full_refresh INTEGER DEFAULT 0,
      reports_only INTEGER DEFAULT 0,
      report_start_date TEXT DEFAULT '',
      report_end_date TEXT DEFAULT '',
      complete_months_only INTEGER DEFAULT 0,
      notification_config_id INTEGER,
      last_run_at TEXT,
      next_run_at TEXT,
      last_run_status TEXT,
      last_run_message TEXT,
      created_at TEXT NOT NULL,
      updated_at TEXT NOT NULL
    )`,
    columns: ['id', 'name', 'enabled', 'schedule_type', 'interval_minutes', 'time_of_day', 'days_of_week', 'day_of_month', 'cron_expression', 'oneoff_datetime', 'query_filter', 'attachments', 'full_refresh', 'reports_only', 'report_start_date', 'report_end_date', 'complete_months_only', 'notification_config_id', 'last_run_at', 'next_run_at', 'last_run_status', 'last_run_message', 'created_at', 'updated_at']
  },
  skip_companies: {
    create: `CREATE TABLE IF NOT EXISTS skip_companies (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      company_id TEXT NOT NULL UNIQUE,
      created_at TEXT NOT NULL
    )`,
    columns: ['id', 'company_id', 'created_at'],
    indexes: [
      `CREATE UNIQUE INDEX IF NOT EXISTS idx_skip_companies_company_id ON skip_companies(company_id)`
    ]
  }
};

/**
 * Get encryption key from environment variable
 */
function getEncryptionKey() {
  const key = process.env.ENCRYPTION_KEY;
  if (!key || key.trim() === '') {
    throw new Error('ENCRYPTION_KEY environment variable is not set');
  }
  return key;
}

/**
 * Check if encryption key is configured
 */
export function isEncryptionConfigured() {
  const key = process.env.ENCRYPTION_KEY;
  return key && key.trim() !== '';
}

/**
 * Derive encryption key from secret and salt
 */
function deriveKey(secret, salt) {
  return crypto.pbkdf2Sync(secret, salt, PBKDF2_ITERATIONS, KEY_LENGTH, 'sha256');
}

/**
 * Encrypt a value - returns separate components
 */
export function encrypt(value) {
  const encryptionKey = getEncryptionKey();
  const salt = crypto.randomBytes(SALT_LENGTH);
  const key = deriveKey(encryptionKey, salt);
  const iv = crypto.randomBytes(IV_LENGTH);

  const cipher = crypto.createCipheriv(ALGORITHM, key, iv, { authTagLength: AUTH_TAG_LENGTH });
  const plaintext = typeof value === 'string' ? value : JSON.stringify(value);
  let encrypted = cipher.update(plaintext, 'utf8', 'base64');
  encrypted += cipher.final('base64');

  const authTag = cipher.getAuthTag();

  return {
    salt: salt.toString('base64'),
    iv: iv.toString('base64'),
    authTag: authTag.toString('base64'),
    data: encrypted
  };
}

/**
 * Decrypt a value from separate components
 */
export function decrypt(salt, iv, authTag, encryptedData) {
  try {
    const encryptionKey = getEncryptionKey();
    const saltBuf = Buffer.from(salt, 'base64');
    const ivBuf = Buffer.from(iv, 'base64');
    const authTagBuf = Buffer.from(authTag, 'base64');
    const key = deriveKey(encryptionKey, saltBuf);

    const decipher = crypto.createDecipheriv(ALGORITHM, key, ivBuf, { authTagLength: AUTH_TAG_LENGTH });
    decipher.setAuthTag(authTagBuf);

    let decrypted = decipher.update(encryptedData, 'base64', 'utf8');
    decrypted += decipher.final('utf8');

    try {
      return JSON.parse(decrypted);
    } catch {
      return decrypted;
    }
  } catch (e) {
    logger.error('Failed to decrypt:', e.message);
    return null;
  }
}

/**
 * Save database to file - no-op for better-sqlite3 (writes directly to disk)
 * Kept as export to avoid breaking callers.
 */
export function saveDatabase() {
  // No-op: better-sqlite3 writes directly to the file on disk
}

/**
 * Initialize the database and create/validate tables
 */
export async function initDatabase() {
  // Ensure data directory exists
  if (!fs.existsSync(DATA_DIR)) {
    fs.mkdirSync(DATA_DIR, { recursive: true });
    logger.info(`[Database] Created data directory: ${DATA_DIR}`);
  }

  // Open database file (creates if it doesn't exist)
  db = new Database(DB_FILE);

  // Enable WAL mode for better concurrent read/write performance
  db.pragma('journal_mode = WAL');

  logger.info(`[Database] Opened database at ${DB_FILE}`);

  // Validate and create/migrate tables
  validateAndMigrateTables();

  return db;
}

/**
 * Get the column names for an existing table from sqlite_master
 * Returns an array of column names, or null if the table doesn't exist
 */
function getExistingColumns(tableName) {
  const rows = db.pragma(`table_info('${tableName}')`);
  if (!rows || rows.length === 0) return null;
  return rows.map(row => row.name);
}

/**
 * Check if a table's CREATE SQL contains a UNIQUE constraint (outside of PRIMARY KEY)
 * Used to detect legacy schema issues that need migration
 */
function tableHasUniqueConstraint(tableName) {
  const row = db.prepare(`SELECT sql FROM sqlite_master WHERE name=? AND type='table'`).get(tableName);
  if (!row) return false;
  const createSql = row.sql;
  // Check for table-level UNIQUE constraints (not column-level PRIMARY KEY)
  return /UNIQUE\s*\(/.test(createSql) && tableName === 'api_usage';
}

/**
 * Migrate a table by renaming, recreating with correct schema, copying data, and dropping old
 * Preserves all data that fits the new schema (common columns are copied)
 */
function migrateTable(tableName, schema, existingColumns) {
  const tempName = `${tableName}_old_migration`;
  const commonColumns = existingColumns.filter(col => schema.columns.includes(col));

  if (commonColumns.length === 0) {
    // No overlapping columns - just drop and recreate
    logger.info(`[Database] Table '${tableName}' has no compatible columns, recreating empty`);
    db.exec(`DROP TABLE IF EXISTS "${tableName}"`);
    db.exec(schema.create.replace('IF NOT EXISTS ', ''));
  } else {
    logger.info(`[Database] Migrating table '${tableName}': copying ${commonColumns.length} columns (${commonColumns.join(', ')})`);
    db.exec(`ALTER TABLE "${tableName}" RENAME TO "${tempName}"`);
    db.exec(schema.create.replace('IF NOT EXISTS ', ''));
    const columnList = commonColumns.join(', ');
    db.exec(`INSERT INTO "${tableName}" (${columnList}) SELECT ${columnList} FROM "${tempName}"`);
    db.exec(`DROP TABLE "${tempName}"`);
  }

  // Recreate indexes
  if (schema.indexes) {
    for (const indexSql of schema.indexes) {
      db.exec(indexSql);
    }
  }
}

/**
 * Validate all tables against expected schemas and migrate if needed.
 */
function validateAndMigrateTables() {
  let migrated = false;

  for (const [tableName, schema] of Object.entries(TABLE_SCHEMAS)) {
    const existingColumns = getExistingColumns(tableName);

    if (!existingColumns) {
      // Table doesn't exist - create it
      logger.info(`[Database] Creating missing table: ${tableName}`);
      db.exec(schema.create);
      if (schema.indexes) {
        for (const indexSql of schema.indexes) {
          db.exec(indexSql);
        }
      }
      migrated = true;
      continue;
    }

    // Check if columns match expected schema
    const expectedSet = new Set(schema.columns);
    const existingSet = new Set(existingColumns);

    const missingColumns = schema.columns.filter(col => !existingSet.has(col));
    const extraColumns = existingColumns.filter(col => !expectedSet.has(col));

    // Also check for legacy issues like UNIQUE constraints on api_usage
    const hasLegacyUniqueIssue = tableHasUniqueConstraint(tableName);

    if (missingColumns.length === 0 && extraColumns.length === 0 && !hasLegacyUniqueIssue) {
      // Schema matches - ensure indexes exist
      if (schema.indexes) {
        for (const indexSql of schema.indexes) {
          db.exec(indexSql);
        }
      }
      continue;
    }

    // Schema mismatch detected
    if (hasLegacyUniqueIssue) {
      logger.info(`[Database] Table '${tableName}' has legacy UNIQUE constraint that needs removal`);
    }
    if (missingColumns.length > 0) {
      logger.info(`[Database] Table '${tableName}' missing columns: ${missingColumns.join(', ')}`);
    }
    if (extraColumns.length > 0) {
      logger.info(`[Database] Table '${tableName}' has extra columns: ${extraColumns.join(', ')}`);
    }

    if (schema.preserveData) {
      // For protected tables (credentials): only add missing columns, never drop
      if (missingColumns.length > 0) {
        for (const col of missingColumns) {
          logger.info(`[Database] Adding column '${col}' to protected table '${tableName}'`);
          db.exec(`ALTER TABLE "${tableName}" ADD COLUMN "${col}" TEXT`);
        }
        migrated = true;
      }
    } else {
      // For non-protected tables: full migration (rename, recreate, copy, drop)
      migrateTable(tableName, schema, existingColumns);
      migrated = true;
    }
  }

  if (migrated) {
    logger.info('[Database] Schema migration completed');
  }

  logger.info('[Database] Schema validation complete - all tables verified');
}

/**
 * Get database instance
 */
export function getDatabase() {
  if (!db) {
    throw new Error('Database not initialized. Call initDatabase() first.');
  }
  return db;
}

/**
 * Reload the database from disk (picks up changes written by subprocesses)
 */
export async function reloadDatabase() {
  if (!fs.existsSync(DB_FILE)) {
    logger.warn(`[Database] Reload skipped - file not found: ${DB_FILE}`);
    return;
  }

  const fileStat = fs.statSync(DB_FILE);
  logger.info(`[Database] Reloading from disk: ${DB_FILE} (${fileStat.size} bytes, modified ${fileStat.mtime.toISOString()})`);

  if (db) {
    try { db.close(); } catch (e) {
      logger.warn(`[Database] Error closing previous db instance: ${e.message}`);
    }
    db = null;
  }

  db = new Database(DB_FILE);
  db.pragma('journal_mode = WAL');

  // Checkpoint WAL to ensure we see all changes from subprocesses
  try {
    const result = db.pragma('wal_checkpoint(PASSIVE)');
    logger.info(`[Database] WAL checkpoint: ${JSON.stringify(result)}`);
  } catch (e) {
    logger.warn(`[Database] WAL checkpoint failed: ${e.message}`);
  }

  logger.info('[Database] Reloaded database from disk successfully');
}

/**
 * Close database connection
 */
export function closeDatabase() {
  if (db) {
    // Checkpoint WAL before closing to ensure all writes are visible to other processes
    try {
      db.pragma('wal_checkpoint(FULL)');
    } catch (e) {
      // Ignore checkpoint errors on close
    }
    db.close();
    db = null;
    logger.info('[Database] Connection closed');
  }
}

/**
 * Check if database is initialized
 */
export function isDatabaseReady() {
  return db !== null;
}
