// db/credentials.js - Credential CRUD with encryption
import { getDatabase, encrypt, decrypt } from './core.js';

/**
 * Set a credential value (encrypted)
 */
export function setCredential(key, value) {
  const database = getDatabase();
  const { salt, iv, authTag, data } = encrypt(value);

  // Try update first
  const result = database.prepare(
    'UPDATE credentials SET salt = ?, iv = ?, auth_tag = ?, encrypted_data = ? WHERE key = ?'
  ).run(salt, iv, authTag, data, key);

  if (result.changes === 0) {
    // Insert if not exists
    database.prepare(
      'INSERT INTO credentials (key, salt, iv, auth_tag, encrypted_data) VALUES (?, ?, ?, ?, ?)'
    ).run(key, salt, iv, authTag, data);
  }
}

/**
 * Get a credential value (decrypted)
 */
export function getCredential(key) {
  const database = getDatabase();
  const row = database.prepare(
    'SELECT salt, iv, auth_tag, encrypted_data FROM credentials WHERE key = ?'
  ).get(key);

  if (!row) return null;

  return decrypt(row.salt, row.iv, row.auth_tag, row.encrypted_data);
}

/**
 * Get all credentials as object
 */
export function getAllCredentials() {
  const database = getDatabase();
  const rows = database.prepare('SELECT key, salt, iv, auth_tag, encrypted_data FROM credentials').all();

  if (rows.length === 0) return null;

  const credentials = {};
  for (const row of rows) {
    const decrypted = decrypt(row.salt, row.iv, row.auth_tag, row.encrypted_data);
    if (decrypted !== null) {
      credentials[row.key] = decrypted;
    }
  }

  return Object.keys(credentials).length > 0 ? credentials : null;
}

/**
 * Delete a credential
 */
export function deleteCredential(key) {
  const database = getDatabase();
  const result = database.prepare('DELETE FROM credentials WHERE key = ?').run(key);
  return result.changes > 0;
}

/**
 * Delete all credentials
 */
export function deleteAllCredentials() {
  const database = getDatabase();
  const result = database.prepare('DELETE FROM credentials').run();
  return result.changes > 0;
}

/**
 * Check if any credentials exist
 */
export function hasCredentials() {
  const database = getDatabase();
  const row = database.prepare('SELECT 1 FROM credentials LIMIT 1').get();
  return !!row;
}
