// db/job-errors.js - Job error record operations
import { getDatabase } from './core.js';

/**
 * Insert a job error record
 */
export function insertJobError(record) {
  const database = getDatabase();
  database.prepare(
    `INSERT INTO job_errors
     (company_id, query_name, error_type, error_number, description, description2, page, recorded_at)
     VALUES (?, ?, ?, ?, ?, ?, ?, ?)`
  ).run(
    record.company_id,
    record.query_name,
    record.error_type,
    record.error_number || null,
    record.description,
    record.description2 || null,
    record.page || 1,
    record.recorded_at || new Date().toISOString()
  );
}

/**
 * Clear job errors, optionally filtered by query name
 * @param {string|null} queryName - Optional: only clear errors for this query
 * @returns {number} Number of rows deleted
 */
export function clearJobErrors(queryName = null) {
  const database = getDatabase();
  let result;
  if (queryName) {
    result = database.prepare('DELETE FROM job_errors WHERE query_name = ?').run(queryName);
  } else {
    result = database.prepare('DELETE FROM job_errors').run();
  }
  return result.changes;
}

/**
 * Get all job errors, optionally filtered by query name
 * @param {string|null} queryName - Optional filter
 * @returns {Array} Error records
 */
export function getJobErrors(queryName = null) {
  const database = getDatabase();
  if (queryName) {
    return database.prepare(
      'SELECT * FROM job_errors WHERE query_name = ? ORDER BY company_id ASC, recorded_at DESC'
    ).all(queryName);
  }
  return database.prepare(
    'SELECT * FROM job_errors ORDER BY company_id ASC, recorded_at DESC'
  ).all();
}

/**
 * Get job errors grouped by description with count and list of affected companies
 * @param {string|null} queryName - Optional filter
 * @returns {Array} [{description, company_count, companies: [string]}]
 */
export function getJobErrorSummary(queryName = null) {
  const database = getDatabase();
  let rows;
  if (queryName) {
    rows = database.prepare(
      `SELECT description, GROUP_CONCAT(DISTINCT company_id) as companies,
        COUNT(DISTINCT company_id) as company_count
        FROM job_errors WHERE query_name = ?
        GROUP BY description ORDER BY description ASC`
    ).all(queryName);
  } else {
    rows = database.prepare(
      `SELECT description, GROUP_CONCAT(DISTINCT company_id) as companies,
        COUNT(DISTINCT company_id) as company_count
        FROM job_errors
        GROUP BY description ORDER BY description ASC`
    ).all();
  }

  return rows.map(row => ({
    ...row,
    companies: row.companies ? row.companies.split(',') : []
  }));
}

/**
 * Get count of job errors
 * @param {string|null} queryName - Optional filter
 * @returns {number}
 */
export function getJobErrorCount(queryName = null) {
  const database = getDatabase();
  if (queryName) {
    const row = database.prepare('SELECT COUNT(*) as count FROM job_errors WHERE query_name = ?').get(queryName);
    return row.count;
  }
  const row = database.prepare('SELECT COUNT(*) as count FROM job_errors').get();
  return row.count;
}

/**
 * Get job errors excluding companies in the skip list
 * This is the "single source of truth" for exception reports
 * Includes sage_account_number from companies table via LEFT JOIN
 * @param {string|null} queryName - Optional filter
 * @returns {Array} Error records excluding skip_companies, with sage_account_number
 */
export function getFilteredJobErrors(queryName = null) {
  const database = getDatabase();
  if (queryName) {
    return database.prepare(
      `SELECT j.*, c.sage_account_number
       FROM job_errors j
       LEFT JOIN companies c ON j.company_id = c.company_id
       WHERE j.query_name = ?
       AND j.company_id NOT IN (SELECT company_id FROM skip_companies)
       ORDER BY j.company_id ASC, j.recorded_at DESC`
    ).all(queryName);
  }
  return database.prepare(
    `SELECT j.*, c.sage_account_number
     FROM job_errors j
     LEFT JOIN companies c ON j.company_id = c.company_id
     WHERE j.company_id NOT IN (SELECT company_id FROM skip_companies)
     ORDER BY j.company_id ASC, j.recorded_at DESC`
  ).all();
}

/**
 * Get filtered job errors grouped by description
 * Excludes companies in skip list
 * @param {string|null} queryName - Optional filter
 * @returns {Array} [{description, company_count, companies: [string]}]
 */
export function getFilteredJobErrorSummary(queryName = null) {
  const database = getDatabase();
  let rows;
  if (queryName) {
    rows = database.prepare(
      `SELECT description, GROUP_CONCAT(DISTINCT company_id) as companies,
        COUNT(DISTINCT company_id) as company_count
        FROM job_errors
        WHERE query_name = ?
        AND company_id NOT IN (SELECT company_id FROM skip_companies)
        GROUP BY description ORDER BY description ASC`
    ).all(queryName);
  } else {
    rows = database.prepare(
      `SELECT description, GROUP_CONCAT(DISTINCT company_id) as companies,
        COUNT(DISTINCT company_id) as company_count
        FROM job_errors
        WHERE company_id NOT IN (SELECT company_id FROM skip_companies)
        GROUP BY description ORDER BY description ASC`
    ).all();
  }

  return rows.map(row => ({
    ...row,
    companies: row.companies ? row.companies.split(',') : []
  }));
}

/**
 * Get count of job errors excluding skip list
 * @param {string|null} queryName - Optional filter
 * @returns {number}
 */
export function getFilteredJobErrorCount(queryName = null) {
  const database = getDatabase();
  if (queryName) {
    const row = database.prepare(
      `SELECT COUNT(*) as count FROM job_errors
       WHERE query_name = ?
       AND company_id NOT IN (SELECT company_id FROM skip_companies)`
    ).get(queryName);
    return row.count;
  }
  const row = database.prepare(
    `SELECT COUNT(*) as count FROM job_errors
     WHERE company_id NOT IN (SELECT company_id FROM skip_companies)`
  ).get();
  return row.count;
}
