// db/notification-configs.js - Notification configuration CRUD
import { getDatabase, encrypt, decrypt } from './core.js';

/**
 * Get the secret field name for a given provider
 */
function getSecretField(provider) {
  if (provider === 'smtp') return 'password';
  if (provider === 'msgraph') return 'clientSecret';
  return null;
}

/**
 * Check if encrypted secret columns are populated
 */
function hasSecret(row) {
  return row.secret_salt && row.secret_iv && row.secret_auth_tag && row.secret_data;
}

/**
 * Parse a row from the database into a config object with nested provider data.
 * Secrets are masked (replaced with true) unless decryptSecrets is true.
 */
function parseRow(row, decryptSecrets = false) {
  if (!row) return null;

  const config = {
    id: row.id,
    name: row.name,
    provider: row.provider,
    recipients: row.recipients || '',
    cc: row.cc || '',
    bcc: row.bcc || '',
    created_at: row.created_at,
    updated_at: row.updated_at
  };

  let secretValue;
  if (hasSecret(row)) {
    secretValue = decryptSecrets
      ? decrypt(row.secret_salt, row.secret_iv, row.secret_auth_tag, row.secret_data)
      : true;
  } else {
    secretValue = decryptSecrets ? '' : false;
  }

  if (row.provider === 'smtp') {
    config.smtp = {
      host: row.host || '',
      port: row.port || 25,
      security: row.security || 'none',
      ignoreTLS: row.ignore_tls === 1,
      username: row.username || '',
      password: secretValue,
      from: row.from_email || '',
      fromName: row.from_name || ''
    };
  } else if (row.provider === 'msgraph') {
    config.msgraph = {
      tenantId: row.tenant_id || '',
      clientId: row.client_id || '',
      clientSecret: secretValue,
      from: row.from_email || ''
    };
  }

  return config;
}

/**
 * Encrypt a secret value and return the four column components
 */
function encryptSecret(value) {
  if (!value || value === true) {
    return { salt: '', iv: '', authTag: '', data: '' };
  }
  const encrypted = encrypt(value);
  return {
    salt: encrypted.salt,
    iv: encrypted.iv,
    authTag: encrypted.authTag,
    data: encrypted.data
  };
}

/**
 * Create a new notification configuration
 * @returns {number} The new config ID
 */
export function createNotificationConfig(data) {
  const database = getDatabase();
  const now = new Date().toISOString();

  const provider = data.provider;
  const pc = data[provider] || {};

  const secretField = getSecretField(provider);
  const secret = encryptSecret(secretField ? pc[secretField] : null);

  const result = database.prepare(
    `INSERT INTO notification_configs (
      name, provider, host, port, security, ignore_tls, username, from_name,
      from_email, tenant_id, client_id, user_key, device, priority, sound,
      secret_salt, secret_iv, secret_auth_tag, secret_data,
      recipients, cc, bcc, created_at, updated_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`
  ).run(
    data.name,
    provider,
    pc.host || '',
    pc.port || 0,
    pc.security || '',
    pc.ignoreTLS ? 1 : 0,
    pc.username || '',
    pc.fromName || '',
    pc.from || '',
    pc.tenantId || '',
    pc.clientId || '',
    '',  // user_key (unused)
    '',  // device (unused)
    '',  // priority (unused)
    '',  // sound (unused)
    secret.salt,
    secret.iv,
    secret.authTag,
    secret.data,
    data.recipients || '',
    data.cc || '',
    data.bcc || '',
    now,
    now
  );

  return Number(result.lastInsertRowid);
}

/**
 * Update an existing notification configuration
 */
export function updateNotificationConfig(id, data) {
  const database = getDatabase();
  const now = new Date().toISOString();

  const existing = database.prepare('SELECT * FROM notification_configs WHERE id = ?').get(id);
  if (!existing) throw new Error('Configuration not found');

  const provider = data.provider || existing.provider;
  const pc = data[provider] || {};

  // Handle secret: if the field is undefined, preserve existing encrypted value
  const secretField = getSecretField(provider);
  let secretSalt = existing.secret_salt;
  let secretIv = existing.secret_iv;
  let secretAuthTag = existing.secret_auth_tag;
  let secretData = existing.secret_data;

  if (secretField && pc[secretField] !== undefined) {
    const secret = encryptSecret(pc[secretField]);
    secretSalt = secret.salt;
    secretIv = secret.iv;
    secretAuthTag = secret.authTag;
    secretData = secret.data;
  }

  database.prepare(
    `UPDATE notification_configs SET
      name = ?, provider = ?, host = ?, port = ?, security = ?, ignore_tls = ?,
      username = ?, from_name = ?, from_email = ?, tenant_id = ?, client_id = ?,
      user_key = ?, device = ?, priority = ?, sound = ?,
      secret_salt = ?, secret_iv = ?, secret_auth_tag = ?, secret_data = ?,
      recipients = ?, cc = ?, bcc = ?, updated_at = ?
    WHERE id = ?`
  ).run(
    data.name || existing.name,
    provider,
    pc.host !== undefined ? pc.host : existing.host,
    pc.port !== undefined ? pc.port : existing.port,
    pc.security !== undefined ? pc.security : existing.security,
    pc.ignoreTLS !== undefined ? (pc.ignoreTLS ? 1 : 0) : existing.ignore_tls,
    pc.username !== undefined ? pc.username : existing.username,
    pc.fromName !== undefined ? pc.fromName : existing.from_name,
    pc.from !== undefined ? pc.from : existing.from_email,
    pc.tenantId !== undefined ? pc.tenantId : existing.tenant_id,
    pc.clientId !== undefined ? pc.clientId : existing.client_id,
    '',  // user_key (unused)
    '',  // device (unused)
    '',  // priority (unused)
    '',  // sound (unused)
    secretSalt, secretIv, secretAuthTag, secretData,
    data.recipients !== undefined ? data.recipients : existing.recipients,
    data.cc !== undefined ? data.cc : existing.cc,
    data.bcc !== undefined ? data.bcc : existing.bcc,
    now,
    id
  );
}

/**
 * Delete a notification configuration
 * @returns {boolean}
 */
export function deleteNotificationConfig(id) {
  const database = getDatabase();
  const result = database.prepare('DELETE FROM notification_configs WHERE id = ?').run(id);
  return result.changes > 0;
}

/**
 * Get a single notification configuration
 * @param {number} id
 * @param {boolean} decryptSecretsFlag - If true, decrypt sensitive fields
 */
export function getNotificationConfig(id, decryptSecretsFlag = false) {
  const database = getDatabase();
  const row = database.prepare('SELECT * FROM notification_configs WHERE id = ?').get(id);
  return parseRow(row, decryptSecretsFlag);
}

/**
 * Get all notification configurations (secrets masked)
 */
export function getAllNotificationConfigs() {
  const database = getDatabase();
  const rows = database.prepare('SELECT * FROM notification_configs ORDER BY name ASC').all();
  return rows.map(row => parseRow(row, false));
}

/**
 * Get count of notification configurations
 */
export function getNotificationConfigCount() {
  const database = getDatabase();
  const row = database.prepare('SELECT COUNT(*) as count FROM notification_configs').get();
  return row.count;
}
