// db/queries.js - Query definition CRUD
import { getDatabase } from './core.js';

/**
 * Get all queries, sorted by sort_order
 */
export function getAllQueries() {
  const database = getDatabase();
  const rows = database.prepare(
    'SELECT id, sort_order, name, prefix FROM filters ORDER BY sort_order ASC'
  ).all();

  return rows.map(row => ({
    id: row.id,
    sort_order: row.sort_order,
    name: row.name,
    prefix: row.prefix || ''
  }));
}

/**
 * Get a query by id
 */
export function getQueryById(id) {
  const database = getDatabase();
  const row = database.prepare(
    'SELECT id, sort_order, name, prefix FROM filters WHERE id = ?'
  ).get(id);

  if (!row) return null;

  return {
    id: row.id,
    sort_order: row.sort_order,
    name: row.name,
    prefix: row.prefix || ''
  };
}

/**
 * Create or update a query
 */
export function saveQuery(queryData) {
  const database = getDatabase();
  const now = new Date().toISOString();
  const { id, sort_order, name, prefix } = queryData;

  if (id) {
    // Update existing
    database.prepare(
      'UPDATE filters SET sort_order = ?, name = ?, prefix = ?, updated_at = ? WHERE id = ?'
    ).run(sort_order, name, prefix || '', now, id);
    return id;
  } else {
    // Insert new
    const result = database.prepare(
      'INSERT INTO filters (sort_order, name, prefix, created_at, updated_at) VALUES (?, ?, ?, ?, ?)'
    ).run(sort_order, name, prefix || '', now, now);
    return Number(result.lastInsertRowid);
  }
}

/**
 * Delete a query by id
 */
export function deleteQuery(id) {
  const database = getDatabase();
  const result = database.prepare('DELETE FROM filters WHERE id = ?').run(id);
  return result.changes > 0;
}

/**
 * Get query count
 */
export function getQueryCount() {
  const database = getDatabase();
  const row = database.prepare('SELECT COUNT(*) as count FROM filters').get();
  return row.count;
}

/**
 * Check if any queries exist
 */
export function hasQueries() {
  const database = getDatabase();
  const row = database.prepare('SELECT 1 FROM filters LIMIT 1').get();
  return !!row;
}

/**
 * Delete all queries
 */
export function deleteAllQueries() {
  const database = getDatabase();
  database.prepare('DELETE FROM filters').run();
}
