// db/schedules.js - Schedule configuration CRUD
import { getDatabase } from './core.js';
import cronParser from 'cron-parser';

/**
 * Parse a row from the database into a schedule object
 */
function parseRow(row) {
  if (!row) return null;

  // Parse notification config IDs - support both comma-separated string and legacy single ID
  let notificationConfigIds = [];
  if (row.notification_config_id) {
    const idStr = String(row.notification_config_id);
    if (idStr.includes(',')) {
      notificationConfigIds = idStr.split(',').map(id => parseInt(id.trim(), 10)).filter(id => !isNaN(id));
    } else {
      const singleId = parseInt(idStr, 10);
      if (!isNaN(singleId)) notificationConfigIds = [singleId];
    }
  }

  return {
    id: row.id,
    name: row.name,
    enabled: row.enabled === 1,
    scheduleType: row.schedule_type,
    intervalMinutes: row.interval_minutes || 0,
    timeOfDay: row.time_of_day || '',
    daysOfWeek: row.days_of_week ? row.days_of_week.split(',').filter(d => d) : [],
    dayOfMonth: row.day_of_month || 0,
    cronExpression: row.cron_expression || '',
    oneoffDatetime: row.oneoff_datetime || '',
    queryFilter: row.query_filter || 'all',
    attachments: row.attachments ? row.attachments.split(',').filter(a => a) : ['detailed', 'summary', 'exceptions'],
    fullRefresh: row.full_refresh === 1,
    reportsOnly: row.reports_only === 1,
    reportStartDate: row.report_start_date || '',
    reportEndDate: row.report_end_date || '',
    completeMonthsOnly: row.complete_months_only === 1,
    notificationConfigId: notificationConfigIds[0] || null, // Legacy single ID for backwards compatibility
    notificationConfigIds: notificationConfigIds, // New array of IDs
    lastRunAt: row.last_run_at || null,
    nextRunAt: row.next_run_at || null,
    lastRunStatus: row.last_run_status || null,
    lastRunMessage: row.last_run_message || null,
    createdAt: row.created_at,
    updatedAt: row.updated_at
  };
}

/**
 * Calculate next run time based on schedule configuration
 */
export function calculateNextRun(schedule, fromTime = new Date()) {
  const now = fromTime instanceof Date ? fromTime : new Date(fromTime);

  if (schedule.scheduleType === 'oneoff') {
    const oneoffDatetime = schedule.oneoffDatetime;
    if (!oneoffDatetime) return null;

    const scheduledTime = new Date(oneoffDatetime);
    // Only return the time if it's in the future
    if (scheduledTime > now) {
      return scheduledTime.toISOString();
    }
    return null; // Already passed - no next run
  }

  if (schedule.scheduleType === 'interval') {
    const minutes = schedule.intervalMinutes || 60;
    return new Date(now.getTime() + minutes * 60 * 1000).toISOString();
  }

  if (schedule.scheduleType === 'daily') {
    const [hours, minutes] = (schedule.timeOfDay || '00:00').split(':').map(Number);
    const next = new Date(now);
    next.setHours(hours, minutes, 0, 0);

    if (next <= now) {
      next.setDate(next.getDate() + 1);
    }

    return next.toISOString();
  }

  if (schedule.scheduleType === 'weekly') {
    const [hours, minutes] = (schedule.timeOfDay || '00:00').split(':').map(Number);
    const daysOfWeek = schedule.daysOfWeek || [];

    if (daysOfWeek.length === 0) {
      return null;
    }

    const dayNumbers = daysOfWeek.map(d => parseInt(d, 10)).sort((a, b) => a - b);
    const currentDay = now.getDay();
    const currentTimeMinutes = now.getHours() * 60 + now.getMinutes();
    const targetTimeMinutes = hours * 60 + minutes;

    // Find next scheduled day
    let daysToAdd = null;
    for (const day of dayNumbers) {
      if (day > currentDay || (day === currentDay && targetTimeMinutes > currentTimeMinutes)) {
        daysToAdd = day - currentDay;
        break;
      }
    }

    // Wrap to next week
    if (daysToAdd === null) {
      daysToAdd = 7 - currentDay + dayNumbers[0];
    }

    const next = new Date(now);
    next.setDate(next.getDate() + daysToAdd);
    next.setHours(hours, minutes, 0, 0);

    return next.toISOString();
  }

  if (schedule.scheduleType === 'monthly') {
    const [hours, minutes] = (schedule.timeOfDay || '00:00').split(':').map(Number);
    const dayOfMonth = schedule.dayOfMonth || 1;

    const next = new Date(now);
    next.setHours(hours, minutes, 0, 0);

    // Handle day of month (1-31, or -1 for last day)
    if (dayOfMonth === -1) {
      // Last day of month
      next.setMonth(next.getMonth() + 1, 0); // Day 0 = last day of previous month
      if (next <= now) {
        next.setMonth(next.getMonth() + 2, 0);
      }
    } else {
      // Specific day (1-31)
      const targetDay = Math.min(dayOfMonth, 28); // Cap at 28 to avoid invalid dates
      next.setDate(targetDay);

      if (next <= now) {
        next.setMonth(next.getMonth() + 1);
        // Re-apply day in case month has fewer days
        const daysInMonth = new Date(next.getFullYear(), next.getMonth() + 1, 0).getDate();
        next.setDate(Math.min(dayOfMonth, daysInMonth));
      }
    }

    return next.toISOString();
  }

  if (schedule.scheduleType === 'cron') {
    const cronExpr = schedule.cronExpression;
    if (!cronExpr) return null;

    try {
      const interval = cronParser.parseExpression(cronExpr, { currentDate: now });
      const next = interval.next().toDate();
      return next.toISOString();
    } catch (e) {
      console.error('Invalid cron expression:', cronExpr, e.message);
      return null;
    }
  }

  return null;
}

/**
 * Create a new schedule
 * @returns {number} The new schedule ID
 */
export function createSchedule(data) {
  const database = getDatabase();
  const now = new Date().toISOString();

  const schedule = {
    name: data.name,
    enabled: data.enabled !== false,
    scheduleType: data.scheduleType,
    intervalMinutes: data.intervalMinutes || 0,
    timeOfDay: data.timeOfDay || '',
    daysOfWeek: Array.isArray(data.daysOfWeek) ? data.daysOfWeek : [],
    dayOfMonth: data.dayOfMonth || 0,
    cronExpression: data.cronExpression || '',
    oneoffDatetime: data.oneoffDatetime || ''
  };

  const nextRunAt = schedule.enabled ? calculateNextRun(schedule) : null;

  const attachments = Array.isArray(data.attachments)
    ? data.attachments.join(',')
    : (data.attachments || 'detailed,summary,exceptions');

  const result = database.prepare(
    `INSERT INTO schedules (
      name, enabled, schedule_type, interval_minutes, time_of_day, days_of_week,
      day_of_month, cron_expression, oneoff_datetime, query_filter, attachments, full_refresh, reports_only, report_start_date, report_end_date,
      complete_months_only, notification_config_id, next_run_at, created_at, updated_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`
  ).run(
    data.name,
    schedule.enabled ? 1 : 0,
    data.scheduleType,
    schedule.intervalMinutes,
    schedule.timeOfDay,
    schedule.daysOfWeek.join(','),
    schedule.dayOfMonth,
    schedule.cronExpression,
    schedule.oneoffDatetime,
    data.queryFilter || 'all',
    attachments,
    data.fullRefresh ? 1 : 0,
    data.reportsOnly ? 1 : 0,
    data.reportStartDate || '',
    data.reportEndDate || '',
    data.completeMonthsOnly ? 1 : 0,
    // Support both new array format and legacy single ID format
    Array.isArray(data.notificationConfigIds)
      ? data.notificationConfigIds.join(',')
      : (data.notificationConfigId || null),
    nextRunAt,
    now,
    now
  );

  return Number(result.lastInsertRowid);
}

/**
 * Update an existing schedule
 */
export function updateSchedule(id, data) {
  const database = getDatabase();
  const now = new Date().toISOString();

  const existing = database.prepare('SELECT * FROM schedules WHERE id = ?').get(id);
  if (!existing) throw new Error('Schedule not found');

  const enabled = data.enabled !== undefined ? data.enabled : existing.enabled === 1;
  const scheduleType = data.scheduleType !== undefined ? data.scheduleType : existing.schedule_type;
  const intervalMinutes = data.intervalMinutes !== undefined ? data.intervalMinutes : existing.interval_minutes;
  const timeOfDay = data.timeOfDay !== undefined ? data.timeOfDay : existing.time_of_day;
  const daysOfWeek = data.daysOfWeek !== undefined
    ? (Array.isArray(data.daysOfWeek) ? data.daysOfWeek : [])
    : (existing.days_of_week ? existing.days_of_week.split(',') : []);
  const dayOfMonth = data.dayOfMonth !== undefined ? data.dayOfMonth : existing.day_of_month;
  const cronExpression = data.cronExpression !== undefined ? data.cronExpression : (existing.cron_expression || '');
  const oneoffDatetime = data.oneoffDatetime !== undefined ? data.oneoffDatetime : (existing.oneoff_datetime || '');

  const schedule = { scheduleType, intervalMinutes, timeOfDay, daysOfWeek, dayOfMonth, cronExpression, oneoffDatetime };
  const nextRunAt = enabled ? calculateNextRun(schedule) : null;

  const attachments = data.attachments !== undefined
    ? (Array.isArray(data.attachments) ? data.attachments.join(',') : data.attachments)
    : (existing.attachments || 'detailed,summary,exceptions');
  const fullRefresh = data.fullRefresh !== undefined ? data.fullRefresh : (existing.full_refresh === 1);
  const reportsOnly = data.reportsOnly !== undefined ? data.reportsOnly : (existing.reports_only === 1);
  const reportStartDate = data.reportStartDate !== undefined ? data.reportStartDate : (existing.report_start_date || '');
  const reportEndDate = data.reportEndDate !== undefined ? data.reportEndDate : (existing.report_end_date || '');
  const completeMonthsOnly = data.completeMonthsOnly !== undefined ? data.completeMonthsOnly : (existing.complete_months_only === 1);

  // Reset last_run_status to null (pending) when schedule is updated and enabled
  // This ensures the schedule is ready to run fresh with the new configuration
  const shouldResetStatus = enabled && (existing.last_run_status === 'error' || existing.last_run_status === 'stopped');

  database.prepare(
    `UPDATE schedules SET
      name = ?, enabled = ?, schedule_type = ?, interval_minutes = ?,
      time_of_day = ?, days_of_week = ?, day_of_month = ?, cron_expression = ?, oneoff_datetime = ?,
      query_filter = ?, attachments = ?, full_refresh = ?, reports_only = ?, report_start_date = ?, report_end_date = ?,
      complete_months_only = ?, notification_config_id = ?, next_run_at = ?, updated_at = ?,
      last_run_status = CASE WHEN ? THEN NULL ELSE last_run_status END,
      last_run_message = CASE WHEN ? THEN NULL ELSE last_run_message END
    WHERE id = ?`
  ).run(
    data.name !== undefined ? data.name : existing.name,
    enabled ? 1 : 0,
    scheduleType,
    intervalMinutes,
    timeOfDay,
    daysOfWeek.join(','),
    dayOfMonth,
    cronExpression,
    oneoffDatetime,
    data.queryFilter !== undefined ? data.queryFilter : (existing.query_filter || 'all'),
    attachments,
    fullRefresh ? 1 : 0,
    reportsOnly ? 1 : 0,
    reportStartDate,
    reportEndDate,
    completeMonthsOnly ? 1 : 0,
    // Support both new array format and legacy single ID format
    data.notificationConfigIds !== undefined
      ? (Array.isArray(data.notificationConfigIds) ? data.notificationConfigIds.join(',') : data.notificationConfigIds)
      : (data.notificationConfigId !== undefined ? data.notificationConfigId : existing.notification_config_id),
    nextRunAt,
    now,
    shouldResetStatus ? 1 : 0,
    shouldResetStatus ? 1 : 0,
    id
  );
}

/**
 * Update schedule after a run completes
 */
export function updateScheduleAfterRun(id, status, message) {
  const database = getDatabase();
  const now = new Date().toISOString();

  const existing = database.prepare('SELECT * FROM schedules WHERE id = ?').get(id);
  if (!existing) return;

  const schedule = parseRow(existing);
  const nextRunAt = schedule.enabled ? calculateNextRun(schedule) : null;

  database.prepare(
    `UPDATE schedules SET
      last_run_at = ?, last_run_status = ?, last_run_message = ?,
      next_run_at = ?, updated_at = ?
    WHERE id = ?`
  ).run(now, status, message, nextRunAt, now, id);
}

/**
 * Delete a schedule
 * @returns {boolean}
 */
export function deleteSchedule(id) {
  const database = getDatabase();
  const result = database.prepare('DELETE FROM schedules WHERE id = ?').run(id);
  return result.changes > 0;
}

/**
 * Get a single schedule by ID
 */
export function getSchedule(id) {
  const database = getDatabase();
  const row = database.prepare('SELECT * FROM schedules WHERE id = ?').get(id);
  return parseRow(row);
}

/**
 * Get all schedules
 */
export function getAllSchedules() {
  const database = getDatabase();
  const rows = database.prepare('SELECT * FROM schedules ORDER BY name ASC').all();
  return rows.map(row => parseRow(row));
}

/**
 * Get enabled schedules that are due to run
 */
export function getDueSchedules() {
  const database = getDatabase();
  const now = new Date().toISOString();
  const rows = database.prepare(
    'SELECT * FROM schedules WHERE enabled = 1 AND next_run_at IS NOT NULL AND next_run_at <= ?'
  ).all(now);
  return rows.map(row => parseRow(row));
}

/**
 * Get count of schedules
 */
export function getScheduleCount() {
  const database = getDatabase();
  const row = database.prepare('SELECT COUNT(*) as count FROM schedules').get();
  return row.count;
}

/**
 * Reset any schedules with stale "running" status back to "error"
 * This handles the case where a job crashed or the service restarted while a job was running.
 * @returns {number} Number of schedules reset
 */
export function resetStaleRunningSchedules() {
  const database = getDatabase();
  const now = new Date().toISOString();
  const result = database.prepare(
    `UPDATE schedules SET
      last_run_status = 'error',
      last_run_message = 'Job was interrupted (service restart or crash)',
      updated_at = ?
    WHERE last_run_status = 'running'`
  ).run(now);
  return result.changes;
}
