// db/settings.js - Settings key/value CRUD
import { getDatabase } from './core.js';

/**
 * Get a setting value
 */
export function getSetting(key) {
  const database = getDatabase();
  const row = database.prepare('SELECT value FROM settings WHERE key = ?').get(key);
  return row ? row.value : null;
}

/**
 * Set a setting value
 */
export function setSetting(key, value) {
  const database = getDatabase();
  const strValue = String(value);

  // Try update first
  const result = database.prepare('UPDATE settings SET value = ? WHERE key = ?').run(strValue, key);

  if (result.changes === 0) {
    // Insert if not exists
    database.prepare('INSERT INTO settings (key, value) VALUES (?, ?)').run(key, strValue);
  }
}

/**
 * Get all settings as an object
 */
export function getAllSettings() {
  const database = getDatabase();
  const rows = database.prepare('SELECT key, value FROM settings').all();

  const settings = {};
  for (const row of rows) {
    settings[row.key] = row.value;
  }
  return settings;
}

/**
 * Set multiple settings at once
 */
export function setMultipleSettings(settingsObj) {
  for (const [key, value] of Object.entries(settingsObj)) {
    setSetting(key, value);
  }
}

/**
 * Check if any settings exist
 */
export function hasSettings() {
  const database = getDatabase();
  const row = database.prepare('SELECT 1 FROM settings LIMIT 1').get();
  return !!row;
}
