// db/skip-companies.js - Skip/exceptions company operations
import { getDatabase } from './core.js';

/**
 * Get all skipped company IDs (sorted alphabetically)
 * @returns {string[]}
 */
export function getAllSkipCompanies() {
  const database = getDatabase();
  const rows = database.prepare('SELECT company_id FROM skip_companies ORDER BY company_id ASC').all();
  return rows.map(row => row.company_id);
}

/**
 * Get count of skipped companies
 * @returns {number}
 */
export function getSkipCompanyCount() {
  const database = getDatabase();
  const row = database.prepare('SELECT COUNT(*) as count FROM skip_companies').get();
  return row.count;
}

/**
 * Add a company to the skip list
 * @param {string} companyId
 * @returns {boolean} true if added, false if already exists
 */
export function addSkipCompany(companyId) {
  const database = getDatabase();
  try {
    const result = database.prepare(
      'INSERT OR IGNORE INTO skip_companies (company_id, created_at) VALUES (?, ?)'
    ).run(companyId, new Date().toISOString());
    return result.changes > 0;
  } catch (e) {
    // UNIQUE constraint - already exists
    return false;
  }
}

/**
 * Remove a company from the skip list
 * @param {string} companyId
 * @returns {boolean} true if removed
 */
export function removeSkipCompany(companyId) {
  const database = getDatabase();
  const result = database.prepare('DELETE FROM skip_companies WHERE company_id = ?').run(companyId);
  return result.changes > 0;
}

/**
 * Check if a company is in the skip list
 * @param {string} companyId
 * @returns {boolean}
 */
export function isSkipCompany(companyId) {
  const database = getDatabase();
  const row = database.prepare('SELECT 1 FROM skip_companies WHERE company_id = ?').get(companyId);
  return !!row;
}
