// services/export.js - CSV export and error handling logic
import fs from 'fs';
import path from 'path';
import { Parser as Json2CsvParser } from 'json2csv';
import { isValidPrefixPattern, isValidECPattern } from './filters.js';

/**
 * Convert M/D/YYYY date format to YYYY-MM-DD (ISO format)
 * @param {string} mdyDate - Date in M/D/YYYY format (e.g., "8/15/2025")
 * @returns {string|null} - Date in YYYY-MM-DD format or null if invalid
 */
export function mdyToIso(mdyDate) {
  if (!mdyDate) return null;
  const match = String(mdyDate).match(/^(\d{1,2})\/(\d{1,2})\/(\d{4})$/);
  if (!match) return mdyDate; // Return as-is if not M/D/YYYY format
  const [, m, d, y] = match;
  return `${y}-${m.padStart(2, '0')}-${d.padStart(2, '0')}`;
}

/**
 * Convert YYYY-MM-DD date format to M/D/YYYY format
 * @param {string} isoDate - Date in YYYY-MM-DD format
 * @returns {string|null} - Date in M/D/YYYY format or null if invalid
 */
export function isoToMdy(isoDate) {
  if (!isoDate) return null;
  const match = String(isoDate).match(/^(\d{4})-(\d{2})-(\d{2})$/);
  if (!match) return isoDate; // Return as-is if not YYYY-MM-DD format
  const [, y, m, d] = match;
  return `${parseInt(m, 10)}/${parseInt(d, 10)}/${y}`;
}

/**
 * Map raw Intacct item to output row
 */
export function mapRow(item, companyId) {
  return {
    CompanyID: companyId,
    PARTNERID: item.PARTNERID ?? '',
    DOCCONTROLID: item.DOCCONTROLID ?? '',
    N_TRANS: item.N_TRANS ?? '',
    CREATED_DATE: item.CREATED_DATE ?? '',
    CREATED_TIME: item.CREATED_TIME ?? '',
    STATUS: item.STATUS ?? ''
  };
}

/**
 * Map raw Intacct item to database record format
 * @param {Object} item - Raw item from Intacct API
 * @param {string} companyId - Company ID
 * @param {string} queryName - Name of the query
 * @returns {Object} - Record ready for database insertion
 */
export function mapToDbRecord(item, companyId, queryName) {
  return {
    company_id: companyId,
    partner_id: item.PARTNERID || null,
    doc_control_id: item.DOCCONTROLID || '',
    n_trans: parseInt(item.N_TRANS, 10) || 0,
    created_date: mdyToIso(item.CREATED_DATE),
    created_time: item.CREATED_TIME || null,
    status: item.STATUS || null,
    query_name: queryName,
    fetched_at: new Date().toISOString()
  };
}

/**
 * Filter rows to only include records matching the given prefix
 * @param {Array} rows - Rows to filter
 * @param {string} prefix - Prefix to match (e.g., "EC_" or "FUSAPI_")
 */
export function filterByPrefix(rows, prefix) {
  if (!prefix) return rows; // No prefix means keep all rows
  return rows.filter(row => isValidPrefixPattern(row.DOCCONTROLID, prefix));
}

/**
 * Filter rows to only include valid EC_ pattern (legacy function)
 */
export function filterECRows(rows) {
  return rows.filter(row => isValidECPattern(row.DOCCONTROLID));
}

/**
 * Generate output file path with suffix
 */
export function withSuffix(filePath, suffix) {
  const p = path.parse(filePath);
  const cleanName = p.name.replace(/\.+$/, '');
  const base = cleanName + suffix + (p.ext || '');
  return path.join(p.dir || '.', base);
}

/**
 * Normalize array helper
 */
function normErrArray(errs) {
  return Array.isArray(errs) ? errs : (errs ? [errs] : []);
}

/**
 * Decode HTML entities
 */
function decodeEntities(str) {
  if (str == null) return '';
  return String(str).replace(/&(#x?[0-9a-fA-F]+|[a-zA-Z]+);/g, (m, g1) => {
    if (g1[0] === '#') {
      const hex = g1[1] === 'x' || g1[1] === 'X';
      const num = parseInt(g1.slice(hex ? 2 : 1), hex ? 16 : 10);
      if (!isFinite(num) || num <= 0) return '';
      try { return String.fromCodePoint(num); } catch { return ''; }
    }
    const map = { amp: '&', lt: '<', gt: '>', quot: '"', apos: "'", nbsp: ' ' };
    return Object.prototype.hasOwnProperty.call(map, g1) ? map[g1] : m;
  });
}

/**
 * Flatten mixed text content
 */
function flattenText(v) {
  if (v == null) return '';
  if (typeof v === 'string') return v;
  if (typeof v === 'number' || typeof v === 'boolean') return String(v);
  if (Array.isArray(v)) return v.map(flattenText).join(' ');
  if (v && typeof v === 'object' && v['#cdata-section']) return v['#cdata-section'];
  if (v && typeof v === 'object' && v['#text']) return v['#text'];
  try { return JSON.stringify(v); } catch { return String(v); }
}

/**
 * Clean error text from Intacct response
 */
function cleanErrText(v) {
  let s = decodeEntities(flattenText(v));
  s = s.replace(/\[ *Support ID:[^\]]*]/gi, '');
  s = s.replace(/<[^>]+>/g, '');
  s = s.replace(/\s+/g, ' ').trim();
  return s;
}

/**
 * Error recorder for collecting errors during processing
 */
export class ErrorRecorder {
  constructor() {
    this.errors = [];
  }

  /**
   * Record an error from Intacct response
   */
  recordError(company, stage, errs, page, rawPath) {
    const list = normErrArray(errs);

    if (list.length === 0) {
      // Fallback to raw XML if parser did not surface error
      try {
        if (rawPath && fs.existsSync(rawPath)) {
          const xml = fs.readFileSync(rawPath, 'utf8');
          const enoMatch = xml.match(/<errorno>([\s\S]*?)<\/errorno>/i);
          const d1Match = xml.match(/<description>([\s\S]*?)<\/description>/i);
          const d2Match = xml.match(/<description2>([\s\S]*?)<\/description2>/i);
          const eno = cleanErrText(enoMatch ? enoMatch[1] : '');
          const d1 = cleanErrText(d1Match ? d1Match[1] : '');
          const d2 = cleanErrText(d2Match ? d2Match[1] : '');
          if (eno || d1 || d2) {
            this.errors.push({
              Company: company,
              Stage: stage,
              Page: page,
              ErrorNo: eno,
              Description: d1,
              Description2: d2,
              RawPath: rawPath || ''
            });
            return;
          }
        }
      } catch { /* ignore */ }

      this.errors.push({
        Company: company,
        Stage: stage,
        Page: page,
        ErrorNo: '',
        Description: 'Unknown error',
        Description2: '',
        RawPath: rawPath || ''
      });
    } else {
      for (const e of list) {
        this.errors.push({
          Company: company,
          Stage: stage,
          Page: page,
          ErrorNo: e?.errorno || '',
          Description: cleanErrText(e?.description),
          Description2: cleanErrText(e?.description2),
          RawPath: rawPath || ''
        });
      }
    }
  }

  /**
   * Record an HTTP error
   */
  recordHttpError(company, stage, err, page) {
    let desc = err?.response?.data
      ? String(err.response.data).slice(0, 500)
      : (err?.message || String(err));

    this.errors.push({
      Company: company,
      Stage: stage,
      Page: page,
      ErrorNo: '',
      Description: String(desc).replace(/\s+/g, ' ').trim(),
      Description2: '',
      RawPath: ''
    });
  }

  hasErrors() {
    return this.errors.length > 0;
  }

  getErrors() {
    return this.errors;
  }
}

/**
 * Export detailed CSV
 */
export function exportDetailedCsv(rows, outputPath, fields) {
  const defaultFields = ['CompanyID', 'PARTNERID', 'DOCCONTROLID', 'N_TRANS', 'CREATED_DATE', 'CREATED_TIME', 'STATUS'];
  const csvParser = new Json2CsvParser({ fields: fields || defaultFields });
  const csv = csvParser.parse(rows);
  fs.writeFileSync(outputPath, csv, 'utf8');
  return outputPath;
}

/**
 * Export summary CSV
 */
export function exportSummaryCsv(rows, outputPath) {
  // Calculate API usage per company
  const summaryMap = new Map();
  for (const r of rows) {
    const n = Number(String(r.N_TRANS ?? '').replace(/[,\s]/g, '')) || 0;
    summaryMap.set(r.CompanyID, (summaryMap.get(r.CompanyID) || 0) + n);
  }

  const summaryRows = Array.from(summaryMap.entries()).map(([Company, Sum]) => ({
    Company,
    'API Usage': Sum
  }));

  const csvParser = new Json2CsvParser({ fields: ['Company', 'API Usage'] });
  const csv = csvParser.parse(summaryRows);
  fs.writeFileSync(outputPath, csv, 'utf8');
  return outputPath;
}

/**
 * Export errors CSV
 */
export function exportErrorsCsv(errors, outputPath) {
  if (errors.length === 0) return null;

  const fields = ['Company', 'Stage', 'Page', 'ErrorNo', 'Description', 'Description2', 'RawPath'];
  const csvParser = new Json2CsvParser({ fields });
  const csv = csvParser.parse(errors);
  fs.writeFileSync(outputPath, csv, 'utf8');
  return outputPath;
}

/**
 * Prepare responses directory (clear existing files)
 */
export function prepareResponsesDir(responsesDir) {
  try {
    if (fs.existsSync(responsesDir)) {
      for (const entry of fs.readdirSync(responsesDir)) {
        const target = path.join(responsesDir, entry);
        fs.rmSync(target, { recursive: true, force: true });
      }
    } else {
      fs.mkdirSync(responsesDir, { recursive: true });
    }
    return true;
  } catch (e) {
    console.warn('WARN: Could not clear responses directory:', e.message);
    if (!fs.existsSync(responsesDir)) {
      fs.mkdirSync(responsesDir, { recursive: true });
    }
    return false;
  }
}

/**
 * Generate timestamp string for filenames
 */
export function generateTimestamp() {
  return new Date().toISOString().replace(/[-:T]/g, '').slice(0, 15);
}
