// services/intacct.js - Intacct API client
import axios from 'axios';
import fs from 'fs';
import path from 'path';
import { XMLParser } from 'fast-xml-parser';
import { getAxiosProxyConfig } from './proxy.js';

const DEFAULT_URL = 'https://api.intacct.com/ia/xml/xmlgw.phtml';
const DEFAULT_PAGE_SIZE = 1000;
const DEFAULT_FIELDS = ['PARTNERID', 'DOCCONTROLID', 'N_TRANS', 'CREATED_DATE', 'CREATED_TIME', 'STATUS'];

const parser = new XMLParser({ ignoreAttributes: false, attributeNamePrefix: '@_' });

/**
 * Generate a control ID with FUSAPI prefix and timestamp
 */
function generateControlId() {
  return `FUSAPI_${Date.now()}`;
}

/**
 * XML escape special characters
 */
export function escapeXml(str) {
  if (!str) return '';
  return String(str)
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&apos;');
}

/**
 * Convert ISO date (YYYY-MM-DD) to M/D/YYYY format for Intacct
 */
export function isoToMDY(iso) {
  if (!iso) return '';
  const m = String(iso).match(/^(\d{4})-(\d{2})-(\d{2})$/);
  if (!m) return '';
  return `${m[2]}/${m[3]}/${m[1]}`;
}

/**
 * Build the query string with optional date range
 */
export function buildQuery(baseQuery, startDate, endDate) {
  const rangeFrags = [];
  if (startDate) rangeFrags.push(`CREATED_DATE >= '${isoToMDY(startDate)}'`);
  if (endDate) rangeFrags.push(`CREATED_DATE <= '${isoToMDY(endDate)}'`);
  const range = rangeFrags.join(' and ');

  if (range) {
    // Remove existing CREATED_DATE conditions and use user's range
    const queryWithoutDate = baseQuery.replace(/CREATED_DATE\s*[><=!]+\s*'[^']*'\s*(and\s*)?/gi, '').trim();
    const cleaned = queryWithoutDate.replace(/^and\s+|^\s*and\s*$/gi, '').trim();
    return [range, cleaned].filter(Boolean).join(' and ');
  }

  return baseQuery;
}

/**
 * Build XML for readByQuery request
 */
export function buildReadByQueryXML(options) {
  const {
    senderId,
    senderPassword,
    userId,
    userPassword,
    companyId,
    fields = DEFAULT_FIELDS,
    query,
    pageSize = DEFAULT_PAGE_SIZE
  } = options;

  const controlId = generateControlId();

  return {
    controlId,
    xml: `<?xml version="1.0" encoding="UTF-8"?>
<request>
  <control>
    <senderid>${senderId}</senderid>
    <password>${senderPassword}</password>
    <controlid>${controlId}</controlid>
    <uniqueid>false</uniqueid>
    <dtdversion>3.0</dtdversion>
    <includewhitespace>true</includewhitespace>
  </control>
  <operation>
    <authentication>
      <login>
        <userid>${userId}</userid>
        <companyid>${companyId}</companyid>
        <password>${userPassword}</password>
      </login>
    </authentication>
    <content>
      <function controlid="read-${controlId}">
        <readByQuery>
          <object>APIUSAGEDETAIL</object>
          <fields>${fields.join(',')}</fields>
          <query>${query}</query>
          <returnFormat>xml</returnFormat>
          <pagesize>${pageSize}</pagesize>
        </readByQuery>
      </function>
    </content>
  </operation>
</request>`
  };
}

/**
 * Build XML for readMore request (pagination)
 */
export function buildReadMoreXML(options) {
  const {
    senderId,
    senderPassword,
    userId,
    userPassword,
    companyId,
    controlId,
    resultId
  } = options;

  const cid = `more-${controlId}-${Date.now()}`;

  return `<?xml version="1.0" encoding="UTF-8"?>
<request>
  <control>
    <senderid>${senderId}</senderid>
    <password>${senderPassword}</password>
    <controlid>${cid}</controlid>
    <uniqueid>false</uniqueid>
    <dtdversion>3.0</dtdversion>
    <includewhitespace>true</includewhitespace>
  </control>
  <operation>
    <authentication>
      <login>
        <userid>${userId}</userid>
        <companyid>${companyId}</companyid>
        <password>${userPassword}</password>
      </login>
    </authentication>
    <content>
      <function controlid="more-${cid}">
        <readMore>
          <resultId>${resultId}</resultId>
        </readMore>
      </function>
    </content>
  </operation>
</request>`;
}

/**
 * Post XML to Intacct API
 */
export async function postXml(xml, url = DEFAULT_URL) {
  const proxyConfig = getAxiosProxyConfig();
  try {
    const res = await axios.post(url, xml, {
      headers: {
        'Content-Type': 'application/xml; charset=UTF-8',
        'Accept': 'application/xml',
        'User-Agent': 'Datel-Intacct-Node/2.0'
      },
      timeout: 60000,
      responseType: 'text',
      transformResponse: [d => d],
      ...proxyConfig
    });
    return res.data;
  } catch (error) {
    // If we got a response with XML error body, parse it and throw a better error
    if (error.response && error.response.data) {
      const responseXml = error.response.data;
      try {
        const parsed = parser.parse(responseXml);
        const errMsg = parsed?.response?.errormessage?.error;
        if (errMsg) {
          const errorno = errMsg.errorno || 'UNKNOWN';
          const description = errMsg.description || 'Unknown error';
          const betterError = new Error(`${errorno}: ${description}`);
          betterError.intacctError = { errorno, description };
          betterError.httpStatus = error.response.status;
          throw betterError;
        }
      } catch (parseErr) {
        // If we couldn't parse the error XML, fall through to original error
        if (parseErr.intacctError) throw parseErr;
      }
    }
    throw error;
  }
}

/**
 * Post XML and save raw response to file
 */
export async function postXmlAndSave(xml, responsesDir, label, url = DEFAULT_URL) {
  const rawXml = await postXml(xml, url);
  const timestamp = new Date().toISOString().replace(/[-:T]/g, '').slice(0, 15);
  const rawFile = path.join(responsesDir, `Intacct-Raw-${timestamp}-${label}.xml`);
  fs.writeFileSync(rawFile, rawXml, 'utf8');
  return { rawXml, rawFile };
}

/**
 * Parse XML response
 */
export function parseResponse(xml) {
  return parser.parse(xml);
}

/**
 * Extract data items from parsed response
 */
export function extractItems(data) {
  const keys = Object.keys(data || {}).filter(k => !k.startsWith('@_'));
  for (const k of keys) {
    const val = data[k];
    if (!val) continue;
    const arr = ensureArray(val);
    if (arr.length && (arr[0].PARTNERID || arr[0].DOCCONTROLID || arr[0].CREATED_DATE)) {
      return arr;
    }
  }
  return [];
}

/**
 * Ensure value is an array
 */
export function ensureArray(x) {
  return Array.isArray(x) ? x : (x ? [x] : []);
}

/**
 * Check if control response is successful
 */
export function isControlSuccess(parsed) {
  return parsed?.response?.control?.status === 'success';
}

/**
 * Check if operation result is successful
 */
export function isOperationSuccess(parsed) {
  const result = parsed?.response?.operation?.result;
  return result && result.status === 'success';
}

/**
 * Get operation result from parsed response
 */
export function getOperationResult(parsed) {
  return parsed?.response?.operation?.result;
}

/**
 * Get errors from parsed response
 */
export function getControlErrors(parsed) {
  return parsed?.response?.errormessage?.error;
}

/**
 * Get operation errors from parsed response
 */
export function getOperationErrors(parsed) {
  return parsed?.response?.operation?.result?.errormessage?.error
      || parsed?.response?.operation?.errormessage?.error;
}

/**
 * Check if operation authentication failed (user credentials wrong)
 * Returns true if <operation><authentication><status>failure</status>
 */
export function isAuthenticationFailure(parsed) {
  return parsed?.response?.operation?.authentication?.status === 'failure';
}

/**
 * Check if this is a fatal credential error that should stop the entire job
 * - Control failure (sender credentials wrong) - error XL03000006
 * - Operation auth failure (user credentials wrong)
 */
export function isFatalAuthError(parsed, errors) {
  // Control-level failure (sender credentials)
  if (!isControlSuccess(parsed)) {
    return true;
  }

  // Operation-level auth failure (user credentials)
  if (isAuthenticationFailure(parsed)) {
    return true;
  }

  return false;
}

/**
 * Get pagination info from result data
 */
export function getPaginationInfo(resultData) {
  return {
    count: parseInt((resultData?.['@_count'] ?? resultData?.count ?? '0'), 10),
    totalCount: parseInt((resultData?.['@_totalcount'] ?? resultData?.totalcount ?? '0'), 10),
    numRemaining: parseInt((resultData?.['@_numremaining'] ?? resultData?.numremaining ?? '0'), 10),
    resultId: resultData?.['@_resultId'] ?? resultData?.resultId
  };
}

/**
 * Intacct API Client class for stateful operations
 */
export class IntacctClient {
  constructor(options) {
    this.senderId = options.senderId;
    this.senderPassword = options.senderPassword;
    this.userId = options.userId;
    this.userPassword = options.userPassword;
    this.url = options.url || DEFAULT_URL;
    this.pageSize = options.pageSize || DEFAULT_PAGE_SIZE;
    this.fields = options.fields || DEFAULT_FIELDS;
    this.responsesDir = options.responsesDir;
    this.queryName = options.queryName || 'Query';
    this.queryPrefix = options.queryPrefix || '';
    // New options for database storage
    this.onData = options.onData || null;  // Callback: (items, companyId, queryName, prefix) => void
    this.saveXmlFiles = options.saveXmlFiles ?? false;  // Default to false (no XML file saving)
  }

  /**
   * Query API usage for a company
   */
  async queryCompany(companyId, query, onPage) {
    const companyXML = `${this.senderId}|${companyId}`;
    const escapedQuery = escapeXml(query);
    const allItems = [];
    let page = 1;

    // Initial request
    const built = buildReadByQueryXML({
      senderId: this.senderId,
      senderPassword: this.senderPassword,
      userId: this.userId,
      userPassword: this.userPassword,
      companyId: companyXML,
      fields: this.fields,
      query: escapedQuery,
      pageSize: this.pageSize
    });

    // Post XML - optionally save to file
    let rawXml, rawFile;
    if (this.saveXmlFiles && this.responsesDir) {
      const label = `${this.sanitize(this.queryName)}-${this.sanitize(companyId)}-${page}`;
      const result = await postXmlAndSave(built.xml, this.responsesDir, label, this.url);
      rawXml = result.rawXml;
      rawFile = result.rawFile;
    } else {
      rawXml = await postXml(built.xml, this.url);
      rawFile = null;
    }

    const parsed = parseResponse(rawXml);

    if (!isControlSuccess(parsed)) {
      return { success: false, error: 'CONTROL', errors: getControlErrors(parsed), rawFile, fatalAuth: true };
    }

    if (isAuthenticationFailure(parsed)) {
      return { success: false, error: 'AUTH', errors: getOperationErrors(parsed), rawFile, fatalAuth: true };
    }

    if (!isOperationSuccess(parsed)) {
      return { success: false, error: 'OPERATION', errors: getOperationErrors(parsed), rawFile, fatalAuth: false };
    }

    const result = getOperationResult(parsed);
    let { numRemaining, resultId, count, totalCount } = getPaginationInfo(result.data);
    const items = extractItems(result.data);
    allItems.push(...items);

    // Call onData callback if provided (for database storage)
    if (this.onData && items.length > 0) {
      this.onData(items, companyId, this.queryName, this.queryPrefix);
    }

    if (onPage) {
      onPage({ page, count, totalCount, numRemaining, items });
    }

    // Pagination
    while (numRemaining > 0 && resultId) {
      page++;
      const moreXml = buildReadMoreXML({
        senderId: this.senderId,
        senderPassword: this.senderPassword,
        userId: this.userId,
        userPassword: this.userPassword,
        companyId: companyXML,
        controlId: built.controlId,
        resultId
      });

      // Post XML - optionally save to file
      let moreRawXml, moreRawFile;
      if (this.saveXmlFiles && this.responsesDir) {
        const moreLabel = `${this.sanitize(this.queryName)}-${this.sanitize(companyId)}-${page}`;
        const moreResult = await postXmlAndSave(moreXml, this.responsesDir, moreLabel, this.url);
        moreRawXml = moreResult.rawXml;
        moreRawFile = moreResult.rawFile;
      } else {
        moreRawXml = await postXml(moreXml, this.url);
        moreRawFile = null;
      }

      const moreParsed = parseResponse(moreRawXml);

      if (!isOperationSuccess(moreParsed)) {
        return {
          success: false,
          error: 'READMORE',
          errors: getOperationErrors(moreParsed),
          rawFile: moreRawFile,
          partialItems: allItems
        };
      }

      const moreResult = getOperationResult(moreParsed);
      const pageInfo = getPaginationInfo(moreResult.data);
      numRemaining = pageInfo.numRemaining;
      resultId = pageInfo.resultId;

      const moreItems = extractItems(moreResult.data);
      allItems.push(...moreItems);

      // Call onData callback if provided (for database storage)
      if (this.onData && moreItems.length > 0) {
        this.onData(moreItems, companyId, this.queryName, this.queryPrefix);
      }

      if (onPage) {
        onPage({ page, count: pageInfo.count, totalCount: pageInfo.totalCount, numRemaining, items: moreItems });
      }
    }

    return { success: true, items: allItems };
  }

  sanitize(name) {
    return String(name).toLowerCase().replace(/[^a-z0-9-_]+/g, '_').slice(0, 80);
  }
}

/**
 * Test Intacct credentials by making a simple API call
 * @param {Object} options - Credential options
 * @param {string} options.senderId - Partner/Sender ID
 * @param {string} options.senderPassword - Sender password
 * @param {string} options.userId - User ID
 * @param {string} options.userPassword - User password
 * @param {string} [options.testCompanyId] - Optional company ID for full test
 * @param {string} [options.url] - Optional API URL
 * @returns {Promise<{ok: boolean, error?: string, message: string}>}
 */
export async function testCredentials(options) {
  const { senderId, senderPassword, userId, userPassword, testCompanyId, url } = options;

  if (!senderId || !senderPassword || !userId || !userPassword) {
    return { ok: false, error: 'MISSING_CREDENTIALS', message: 'Missing required credentials' };
  }

  // Use provided company or fall back to 'Datel Group' which is always valid for testing
  const companyId = testCompanyId || 'Datel Group';
  const controlId = `TEST_${Date.now()}`;

  // Escape all credentials for XML
  const escapedSenderId = escapeXml(senderId);
  const escapedSenderPassword = escapeXml(senderPassword);
  const escapedUserId = escapeXml(userId);
  const escapedUserPassword = escapeXml(userPassword);
  const escapedCompanyId = escapeXml(companyId);

  // Full test with company - use getAPISession which is lightweight
  const xml = `<?xml version="1.0" encoding="UTF-8"?>
<request>
  <control>
    <senderid>${escapedSenderId}</senderid>
    <password>${escapedSenderPassword}</password>
    <controlid>${controlId}</controlid>
    <uniqueid>false</uniqueid>
    <dtdversion>3.0</dtdversion>
    <includewhitespace>false</includewhitespace>
  </control>
  <operation>
    <authentication>
      <login>
        <userid>${escapedUserId}</userid>
        <companyid>${escapedSenderId}|${escapedCompanyId}</companyid>
        <password>${escapedUserPassword}</password>
      </login>
    </authentication>
    <content>
      <function controlid="testAuth">
        <getAPISession />
      </function>
    </content>
  </operation>
</request>`;

  try {
    const rawXml = await postXml(xml, url || DEFAULT_URL);
    const parsed = parseResponse(rawXml);

    // Check control (sender credentials)
    if (!isControlSuccess(parsed)) {
      const errors = getControlErrors(parsed);
      const errorList = Array.isArray(errors) ? errors : (errors ? [errors] : []);
      const errorMsg = errorList.map(e => e?.description2 || e?.description || 'Unknown error').join('; ');

      return {
        ok: false,
        error: 'SENDER_CREDENTIALS',
        message: `Sender credentials invalid: ${errorMsg || 'Incorrect Partner ID or password'}`
      };
    }

    // Check authentication (user credentials)
    if (isAuthenticationFailure(parsed)) {
      return {
        ok: false,
        error: 'USER_CREDENTIALS',
        message: companyId
          ? `User credentials invalid for company "${companyId}"`
          : 'User credentials may be invalid (tested without a company)'
      };
    }

    // If we get here with a company, credentials are fully valid
    if (companyId) {
      return { ok: true, message: 'Credentials validated successfully' };
    }

    // Without a company, we can only confirm sender credentials work
    return { ok: true, message: 'Sender credentials validated. Upload companies to fully test user credentials.' };

  } catch (e) {
    // Check if this is a parsed Intacct error (from postXml)
    if (e.intacctError) {
      const { errorno, description } = e.intacctError;
      // GW-0011 typically means invalid sender credentials
      if (errorno === 'GW-0011') {
        return {
          ok: false,
          error: 'SENDER_CREDENTIALS',
          message: `Invalid Sender ID or password`
        };
      }
      return {
        ok: false,
        error: 'API_ERROR',
        message: `${errorno}: ${description}`
      };
    }
    return {
      ok: false,
      error: 'CONNECTION_ERROR',
      message: `Connection error: ${e.message}`
    };
  }
}

export default IntacctClient;
