// services/job.js - Job state management
import logger from "./logger.js";

let procRef = null;
let scheduledJobRunning = false;  // Track if a scheduled job is running (separate from manual jobs)
let scheduledJobStopping = false; // Track if a scheduled job stop has been requested

// Maximum log size in characters (~5MB). Older content is trimmed when exceeded.
const MAX_LOG_SIZE = 5 * 1024 * 1024;

let currentJob = {
  running: false,
  startedAt: null,
  log: "",
  exitCode: null,
  stopping: false,
  stopped: false,
  queryName: null,
  queryPrefix: null
};

export function getJob() {
  return currentJob;
}

export function getProcRef() {
  return procRef;
}

export function setProcRef(ref) {
  procRef = ref;
}

export function resetJob() {
  currentJob = {
    running: false,
    startedAt: null,
    log: "",
    exitCode: null,
    stopping: false,
    stopped: false,
    queryName: null,
    queryPrefix: null
  };
}

export function setJobQueryName(name) {
  currentJob.queryName = name;
}

export function setJobQueryPrefix(prefix) {
  currentJob.queryPrefix = prefix;
}

export function appendLog(data) {
  currentJob.log += data;

  // Cap log size to prevent unbounded memory growth
  if (currentJob.log.length > MAX_LOG_SIZE) {
    // Keep the most recent portion, trim from the front
    const trimPoint = currentJob.log.length - Math.floor(MAX_LOG_SIZE * 0.8);
    const newlineAfterTrim = currentJob.log.indexOf('\n', trimPoint);
    currentJob.log = '[...earlier output trimmed...]\n' + currentJob.log.slice(newlineAfterTrim + 1);
  }

  // Also log important lines to Winston for persistence
  // Parse log data line by line and log significant entries
  const lines = data.toString().split(/\r?\n/).filter(l => l.trim());
  for (const line of lines) {
    const trimmed = line.trim();
    if (!trimmed) continue;

    // Log errors (but not expected skip messages or "no errors" summary)
    if (/error|failed|exception/i.test(trimmed) && !/no errors/i.test(trimmed) && !/SKIPPED/i.test(trimmed)) {
      logger.error(`[Job] ${trimmed}`);
    }
    // Log skipped companies as warnings (expected behaviour)
    else if (/SKIPPED/i.test(trimmed)) {
      logger.warn(`[Job] ${trimmed}`);
    }
    // Log warnings
    else if (/warning|warn/i.test(trimmed)) {
      logger.warn(`[Job] ${trimmed}`);
    }
    // Log important progress messages (skip DEBUG lines)
    else if (/^DEBUG:/i.test(trimmed)) {
      // Skip debug output from CLI
    }
    else if (
      /^={3,}/.test(trimmed) ||  // Section headers (=== Company, ========== Running query)
      /^Mode:/i.test(trimmed) ||  // Fetch mode (Full refresh, Top-up, Initial fetch)
      /^Query for all/i.test(trimmed) ||  // Query being used
      /^Input file:/i.test(trimmed) ||  // Input file info
      /^Companies:/i.test(trimmed) ||  // Company count
      /processing company/i.test(trimmed) ||
      /completed|finished|done/i.test(trimmed) ||
      /starting|began|running query/i.test(trimmed) ||
      /total.*records|rows.*processed/i.test(trimmed) ||
      /records.*stored|stored.*records/i.test(trimmed) ||
      /records.*filtered|filtered.*out/i.test(trimmed) ||
      /cleared.*records|records.*cleared/i.test(trimmed) ||
      /^Recorded fetch date/i.test(trimmed) ||  // Fetch date tracking
      /page \d+/i.test(trimmed) ||  // Page progress (page 1: count=...)
      /fetched \d+/i.test(trimmed) ||
      /prefix filter/i.test(trimmed) ||  // Prefix filtering results
      /monthly totals/i.test(trimmed) ||  // Monthly summary header
      /^\s+\d{4}-\d{2}:/i.test(trimmed) ||  // Monthly data lines (2025-12: ...)
      /^\s+TOTAL:/i.test(trimmed) ||  // Grand total line
      /database.*closed/i.test(trimmed)  // Database lifecycle
    ) {
      logger.info(`[Job] ${trimmed}`);
    }
  }
}

/**
 * Log a message to both the live log and Winston logger
 */
export function logJob(message, level = 'info') {
  const timestamp = new Date().toISOString();
  const formattedMessage = `[${timestamp}] ${message}\n`;
  currentJob.log += formattedMessage;

  // Cap log size (same as appendLog)
  if (currentJob.log.length > MAX_LOG_SIZE) {
    const trimPoint = currentJob.log.length - Math.floor(MAX_LOG_SIZE * 0.8);
    const newlineAfterTrim = currentJob.log.indexOf('\n', trimPoint);
    currentJob.log = '[...earlier output trimmed...]\n' + currentJob.log.slice(newlineAfterTrim + 1);
  }

  // Also log to Winston
  switch (level) {
    case 'error':
      logger.error(`[Job] ${message}`);
      break;
    case 'warn':
      logger.warn(`[Job] ${message}`);
      break;
    default:
      logger.info(`[Job] ${message}`);
  }
}

export function setJobRunning(running) {
  currentJob.running = running;
}

export function setJobStartedAt(time) {
  currentJob.startedAt = time;
}

export function setJobStopping(stopping) {
  currentJob.stopping = stopping;
}

export function setJobExitCode(code) {
  currentJob.exitCode = code;
}

export function setJobStopped(stopped) {
  currentJob.stopped = stopped;
}

export function isScheduledJobRunning() {
  return scheduledJobRunning;
}

export function setScheduledJobRunning(running) {
  scheduledJobRunning = running;
  if (!running) scheduledJobStopping = false; // Reset stopping flag when job finishes
}

export function isScheduledJobStopping() {
  return scheduledJobStopping;
}

export function setScheduledJobStopping(stopping) {
  scheduledJobStopping = stopping;
}
