// services/logger.js - Logging service with monthly rotation
import winston from 'winston';
import path from 'path';
import fs from 'fs';

// Log directory - uses LOG_DIR env var or defaults to ./logs
const logsDir = process.env.LOG_DIR || path.join(process.cwd(), 'logs');

// Ensure logs directory exists
if (!fs.existsSync(logsDir)) {
  fs.mkdirSync(logsDir, { recursive: true });
}

// Get current month string (YYYY-MM)
function getCurrentMonth() {
  const now = new Date();
  const year = now.getFullYear();
  const month = String(now.getMonth() + 1).padStart(2, '0');
  return `${year}-${month}`;
}

// Get log file path for current month
function getLogFilePath() {
  return path.join(logsDir, `app-${getCurrentMonth()}.log`);
}

// Track current month for rotation
let currentMonth = getCurrentMonth();

// Custom format for console output
const consoleFormat = winston.format.combine(
  winston.format.colorize(),
  winston.format.timestamp({ format: 'YYYY-MM-DD HH:mm:ss' }),
  winston.format.printf(({ level, message, timestamp }) => {
    return `${timestamp} ${level}: ${message}`;
  })
);

// JSON format for file output
const fileFormat = winston.format.combine(
  winston.format.timestamp(),
  winston.format.json()
);

// Get log level from environment or default to 'info'
const logLevel = process.env.LOG_LEVEL || 'info';

// Create initial file transport
let fileTransport = new winston.transports.File({
  filename: getLogFilePath(),
  format: fileFormat
});

// Create logger instance
const logger = winston.createLogger({
  level: logLevel,
  transports: [
    new winston.transports.Console({
      format: consoleFormat
    }),
    fileTransport
  ]
});

// Check for month rotation every minute
setInterval(() => {
  const newMonth = getCurrentMonth();
  if (newMonth !== currentMonth) {
    currentMonth = newMonth;
    const newFilePath = getLogFilePath();

    // Remove old file transport
    logger.remove(fileTransport);

    // Create new file transport
    fileTransport = new winston.transports.File({
      filename: newFilePath,
      format: fileFormat
    });

    logger.add(fileTransport);
    logger.info(`Log rotated to new file: ${newFilePath}`);
  }
}, 60 * 1000);

/**
 * Get the logs directory path
 */
function getLogsDirectory() {
  return logsDir;
}

/**
 * List all log files with metadata
 */
function listLogFiles() {
  try {
    if (!fs.existsSync(logsDir)) {
      return [];
    }

    const files = fs.readdirSync(logsDir)
      .filter(file => file.startsWith('app-') && file.endsWith('.log'))
      .map(filename => {
        const filePath = path.join(logsDir, filename);
        const stats = fs.statSync(filePath);
        // Extract date from filename (app-YYYY-MM.log)
        const dateMatch = filename.match(/app-(\d{4}-\d{2})\.log/);
        return {
          filename,
          date: dateMatch ? dateMatch[1] : null,
          size: stats.size,
          modified: stats.mtime
        };
      })
      .sort((a, b) => b.filename.localeCompare(a.filename)); // Newest first

    return files;
  } catch (e) {
    logger.error(`Failed to list log files: ${e.message}`);
    return [];
  }
}

/**
 * Read a specific log file
 */
function readLogFile(filename) {
  try {
    // Security: sanitize filename to prevent directory traversal
    const sanitizedFilename = path.basename(filename);

    // Validate filename format
    if (!sanitizedFilename.match(/^app-\d{4}-\d{2}\.log$/)) {
      throw new Error('Invalid log filename format');
    }

    const filePath = path.join(logsDir, sanitizedFilename);

    if (!fs.existsSync(filePath)) {
      throw new Error('Log file not found');
    }

    return fs.readFileSync(filePath, 'utf8');
  } catch (e) {
    logger.error(`Failed to read log file ${filename}: ${e.message}`);
    throw e;
  }
}

// Export logger with helper functions
logger.getLogsDirectory = getLogsDirectory;
logger.listLogFiles = listLogFiles;
logger.readLogFile = readLogFile;

export default logger;
