// services/notification.js - Notification sending service (SMTP, Microsoft Graph)
import nodemailer from 'nodemailer';
import logger from './logger.js';
import { getNotificationConfig } from './database.js';
import { getNodemailerProxyConfig, getProxyAgent, getProxyUrl } from './proxy.js';

// Cache SMTP transporters by config fingerprint to reuse TCP connections
const transporterCache = new Map();

function getSmtpTransporter(smtp) {
  // Get proxy config
  const proxyConfig = getNodemailerProxyConfig();
  const proxyUrl = proxyConfig.proxy || '';

  // Build a cache key from the connection parameters (including proxy)
  const cacheKey = `${smtp.host}:${smtp.port}:${smtp.username}:${smtp.security}:${proxyUrl}`;

  if (transporterCache.has(cacheKey)) {
    return transporterCache.get(cacheKey);
  }

  const transportConfig = {
    host: smtp.host,
    port: smtp.port,
    pool: true,         // Enable connection pooling
    maxConnections: 3,
    auth: {
      user: smtp.username,
      pass: smtp.password
    },
    ...proxyConfig  // Add proxy configuration if set
  };

  if (smtp.security === 'ssl') {
    transportConfig.secure = true;
  } else if (smtp.security === 'starttls') {
    transportConfig.secure = false;
    transportConfig.requireTLS = true;
  } else {
    transportConfig.secure = false;
    transportConfig.ignoreTLS = true;
  }

  if (smtp.ignoreTLS) {
    transportConfig.tls = { rejectUnauthorized: false };
  }

  const transporter = nodemailer.createTransport(transportConfig);
  transporterCache.set(cacheKey, transporter);
  return transporter;
}

/**
 * Send email via SMTP
 */
async function sendSmtpEmail(config, subject, body, recipients, cc, bcc, attachments = []) {
  const smtp = config.smtp;
  const transporter = getSmtpTransporter(smtp);

  const mailOptions = {
    from: smtp.fromName ? `"${smtp.fromName}" <${smtp.from}>` : smtp.from,
    to: recipients,
    subject: subject,
    html: body
  };

  if (cc) mailOptions.cc = cc;
  if (bcc) mailOptions.bcc = bcc;

  if (attachments.length > 0) {
    mailOptions.attachments = attachments.map(att => ({
      filename: att.filename,
      content: att.content,
      contentType: att.contentType || 'application/octet-stream'
    }));
  }

  const result = await transporter.sendMail(mailOptions);
  logger.info(`Email sent via SMTP: ${result.messageId}`);
  return result;
}

/**
 * Send email via Microsoft Graph API
 */
async function sendGraphEmail(config, subject, body, recipients, cc, bcc, attachments = []) {
  const graph = config.msgraph;

  // Dynamic imports for Azure packages (only loaded when needed)
  const { ClientSecretCredential } = await import('@azure/identity');
  const { Client } = await import('@microsoft/microsoft-graph-client');
  const { TokenCredentialAuthenticationProvider } = await import(
    '@microsoft/microsoft-graph-client/authProviders/azureTokenCredentials/index.js'
  );

  // Build credential options with proxy if configured
  const credentialOptions = {};
  const proxyUrl = getProxyUrl();
  if (proxyUrl) {
    // Azure Identity supports proxyOptions for proxy configuration
    credentialOptions.proxyOptions = {
      host: new URL(proxyUrl).hostname,
      port: parseInt(new URL(proxyUrl).port || '8080', 10)
    };
    // If proxy has auth, add username/password
    const parsedProxy = new URL(proxyUrl);
    if (parsedProxy.username) {
      credentialOptions.proxyOptions.username = parsedProxy.username;
      credentialOptions.proxyOptions.password = parsedProxy.password || '';
    }
  }

  const credential = new ClientSecretCredential(
    graph.tenantId,
    graph.clientId,
    graph.clientSecret,
    credentialOptions
  );

  const authProvider = new TokenCredentialAuthenticationProvider(credential, {
    scopes: ['https://graph.microsoft.com/.default']
  });

  // Build Graph client options with proxy if configured
  const clientOptions = { authProvider };
  const proxyAgent = getProxyAgent();
  if (proxyAgent) {
    // Microsoft Graph client supports custom fetch implementation
    clientOptions.fetchOptions = { agent: proxyAgent };
  }

  const client = Client.initWithMiddleware(clientOptions);

  const toRecipients = recipients.split(',').map(email => ({
    emailAddress: { address: email.trim() }
  }));

  const message = {
    subject: subject,
    body: {
      contentType: 'HTML',
      content: body
    },
    toRecipients: toRecipients
  };

  if (cc) {
    message.ccRecipients = cc.split(',').map(email => ({
      emailAddress: { address: email.trim() }
    }));
  }

  if (bcc) {
    message.bccRecipients = bcc.split(',').map(email => ({
      emailAddress: { address: email.trim() }
    }));
  }

  if (attachments.length > 0) {
    message.attachments = attachments.map(att => ({
      '@odata.type': '#microsoft.graph.fileAttachment',
      name: att.filename,
      contentBytes: att.content.toString('base64'),
      contentType: att.contentType || 'application/octet-stream'
    }));
  }

  await client.api(`/users/${graph.from}/sendMail`).post({ message });
  logger.info(`Email sent via Microsoft Graph from ${graph.from}`);
}

/**
 * Send a test notification using a config object (may be saved or unsaved)
 */
export async function sendTestNotification(config) {
  const appName = 'Intacct API Usage';
  const recipients = config.recipients || 'test@example.com';
  const cc = config.cc || '';
  const bcc = config.bcc || '';
  const subject = `Test Email from ${appName}`;
  const body = `
    <h2>Test Email</h2>
    <p>This is a test email from ${appName}.</p>
    <p>Configuration: <strong>${config.name || 'Unsaved Configuration'}</strong></p>
    <p>Provider: <strong>${config.provider === 'smtp' ? 'SMTP' : 'Microsoft Graph'}</strong></p>
    <p>Sent at: ${new Date().toLocaleDateString('en-GB')} ${new Date().toLocaleTimeString()}</p>
  `;

  if (config.provider === 'smtp') {
    return await sendSmtpEmail(config, subject, body, recipients, cc, bcc);
  } else if (config.provider === 'msgraph') {
    return await sendGraphEmail(config, subject, body, recipients, cc, bcc);
  } else {
    throw new Error('Invalid notification provider');
  }
}

/**
 * Send a notification using a saved config ID
 * @param {number} configId - Notification config ID
 * @param {string} subject - Email subject
 * @param {string} body - HTML body for email
 * @param {Array} attachments - Optional array of {filename, content (Buffer), contentType}
 */
export async function sendNotification(configId, subject, body, attachments = []) {
  const config = getNotificationConfig(configId, true);

  if (!config) {
    throw new Error('Notification configuration not found');
  }

  const recipients = config.recipients;
  const cc = config.cc || '';
  const bcc = config.bcc || '';

  if (!recipients) {
    throw new Error('No recipients configured');
  }

  if (config.provider === 'smtp') {
    return await sendSmtpEmail(config, subject, body, recipients, cc, bcc, attachments);
  } else if (config.provider === 'msgraph') {
    return await sendGraphEmail(config, subject, body, recipients, cc, bcc, attachments);
  } else {
    throw new Error('Invalid notification provider');
  }
}
