// services/proxy.js - Proxy configuration and HTTP agent utilities
import { HttpsProxyAgent } from 'https-proxy-agent';
import { getCredential, getSetting, isEncryptionConfigured } from './database.js';
import logger from './logger.js';

// Cache the proxy agent to reuse connections
let cachedProxyAgent = null;
let cachedProxyUrl = null;

/**
 * Check if proxy is enabled (has URL configured AND is enabled)
 * @returns {boolean}
 */
export function isProxyEnabled() {
  try {
    // URL and enabled flag are in settings (not encrypted)
    const proxyUrl = getSetting('proxyUrl');
    if (!proxyUrl) return false;

    // Check explicit enabled flag (defaults to true if URL exists but flag not set)
    const enabledFlag = getSetting('proxyEnabled');
    return enabledFlag !== '0'; // Enabled unless explicitly disabled
  } catch (e) {
    logger.error(`Error checking proxy enabled state: ${e.message}`);
    return false;
  }
}

/**
 * Get proxy configuration from database
 * @returns {{ url: string, username?: string, password?: string } | null}
 */
export function getProxyConfig() {
  try {
    // Check if proxy is enabled first
    if (!isProxyEnabled()) return null;

    // URL is in settings
    const proxyUrl = getSetting('proxyUrl');
    if (!proxyUrl) return null;

    // Username and password are encrypted in credentials
    const proxyUsername = isEncryptionConfigured() ? getCredential('proxyUsername') : null;
    const proxyPassword = isEncryptionConfigured() ? getCredential('proxyPassword') : null;

    return {
      url: proxyUrl,
      username: proxyUsername || undefined,
      password: proxyPassword || undefined
    };
  } catch (e) {
    logger.error(`Error getting proxy config: ${e.message}`);
    return null;
  }
}

/**
 * Build full proxy URL with authentication if configured
 * @returns {string | null}
 */
export function getProxyUrl() {
  const config = getProxyConfig();
  if (!config || !config.url) return null;

  try {
    const url = new URL(config.url);
    if (config.username) {
      url.username = config.username;
      if (config.password) {
        url.password = config.password;
      }
    }
    return url.toString();
  } catch (e) {
    logger.error(`Invalid proxy URL: ${e.message}`);
    return null;
  }
}

/**
 * Get HTTPS proxy agent for use with fetch, axios, etc.
 * @param {boolean} forceNew - Force creation of new agent (ignore cache)
 * @returns {HttpsProxyAgent | null}
 */
export function getProxyAgent(forceNew = false) {
  const proxyUrl = getProxyUrl();

  if (!proxyUrl) {
    cachedProxyAgent = null;
    cachedProxyUrl = null;
    return null;
  }

  // Return cached agent if URL hasn't changed
  if (!forceNew && cachedProxyAgent && cachedProxyUrl === proxyUrl) {
    return cachedProxyAgent;
  }

  try {
    cachedProxyAgent = new HttpsProxyAgent(proxyUrl);
    cachedProxyUrl = proxyUrl;
    logger.debug(`Proxy agent created for: ${new URL(proxyUrl).host}`);
    return cachedProxyAgent;
  } catch (e) {
    logger.error(`Failed to create proxy agent: ${e.message}`);
    return null;
  }
}

/**
 * Clear cached proxy agent (call when proxy config changes)
 */
export function clearProxyCache() {
  cachedProxyAgent = null;
  cachedProxyUrl = null;
}

/**
 * Check if proxy is configured (has URL, regardless of enabled state)
 * @returns {boolean}
 */
export function isProxyConfigured() {
  try {
    const proxyUrl = getSetting('proxyUrl');
    return !!proxyUrl;
  } catch (e) {
    logger.error(`Error checking proxy configured: ${e.message}`);
    return false;
  }
}

/**
 * Get axios proxy configuration
 * Returns config object suitable for axios request options
 * @returns {{ httpsAgent: HttpsProxyAgent, proxy: false } | { proxy: false }}
 */
export function getAxiosProxyConfig() {
  try {
    const agent = getProxyAgent();
    if (!agent) {
      // Always return proxy: false to prevent axios from using environment proxy settings
      logger.debug('[Proxy] No proxy agent - disabling axios proxy');
      return { proxy: false };
    }

    logger.debug(`[Proxy] Using proxy agent for axios requests`);
    return {
      httpsAgent: agent,
      proxy: false  // Disable axios built-in proxy to use our agent
    };
  } catch (e) {
    logger.error(`Error getting axios proxy config: ${e.message}`);
    return { proxy: false };
  }
}

/**
 * Get fetch options with proxy
 * @param {RequestInit} options - Existing fetch options
 * @returns {RequestInit}
 */
export function getFetchOptionsWithProxy(options = {}) {
  try {
    const agent = getProxyAgent();
    if (!agent) return options;

    return {
      ...options,
      agent
    };
  } catch (e) {
    logger.error(`Error getting fetch proxy options: ${e.message}`);
    return options;
  }
}

/**
 * Get nodemailer proxy configuration for SMTP
 * @returns {{ proxy: string } | {}}
 */
export function getNodemailerProxyConfig() {
  try {
    const proxyUrl = getProxyUrl();
    if (!proxyUrl) return {};

    // Nodemailer supports SOCKS and HTTP proxies
    // Format: protocol://[user:pass@]host:port
    return { proxy: proxyUrl };
  } catch (e) {
    logger.error(`Error getting nodemailer proxy config: ${e.message}`);
    return {};
  }
}
