// services/scheduler.js - Scheduled report generation and email service
import path from 'path';
import { spawn } from 'child_process';
import { fileURLToPath } from 'url';
import ExcelJS from 'exceljs';
import logger from './logger.js';
import { getConfig } from './config.js';
import {
  getJob,
  resetJob,
  appendLog,
  logJob,
  setJobRunning,
  setJobStartedAt,
  setJobExitCode,
  setJobQueryName,
  setJobQueryPrefix,
  setProcRef,
  getProcRef,
  isScheduledJobRunning,
  setScheduledJobRunning,
  isScheduledJobStopping,
  setScheduledJobStopping,
  setJobStopping
} from './job.js';
import {
  getDueSchedules,
  getSchedule,
  updateSchedule,
  updateScheduleAfterRun,
  getApiUsageRecords,
  getApiUsageSummary,
  getApiUsageCount,
  getCompaniesWithNoData,
  getFilteredJobErrors,
  getFilteredJobErrorCount,
  getCompanyCount,
  getAllQueries,
  getLastFetchedAtByQuery,
  getAllCredentials,
  getAllCompanies,
  getCredential,
  clearJobErrors,
  insertJobError,
  reloadDatabase,
  getApiUsageDateRange,
  replaceAllCompanies,
  resetStaleRunningSchedules
} from './database.js';

// Query name used for API import errors
const API_IMPORT_QUERY = 'API_IMPORT';
import { sendNotification } from './notification.js';
import { testCredentials } from './intacct.js';
import { getFetchOptionsWithProxy } from './proxy.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

let schedulerInterval = null;
let isRunning = false;

/**
 * Load companies from an external API URL
 * Expected JSON format: { "companies": [{ "sage_account_number": "XXX", "company_id": "Company Name" }, ...] }
 * Companies with NULL/blank company_id are logged to job_errors for reporting.
 * @param {string} apiUrl - The API URL to fetch companies from
 * @param {string} apiToken - Optional API token for Authorization header
 * @returns {Promise<{success: boolean, count?: number, missingCount?: number, message?: string}>}
 */
async function loadCompaniesFromApi(apiUrl, apiToken = '') {
  if (!apiUrl || !String(apiUrl).trim()) {
    return { success: false, message: 'No API URL configured' };
  }

  try {
    logger.info(`[Scheduler] Loading companies from API: ${apiUrl}`);

    // Build request options with optional Authorization header and proxy
    let fetchOptions = {};
    if (apiToken && String(apiToken).trim()) {
      fetchOptions.headers = {
        'Authorization': `Bearer ${apiToken.trim()}`
      };
    }
    fetchOptions = getFetchOptionsWithProxy(fetchOptions);

    const response = await fetch(apiUrl, fetchOptions);

    if (!response.ok) {
      return { success: false, message: `API returned ${response.status}: ${response.statusText}` };
    }

    const data = await response.json();

    if (!data.companies || !Array.isArray(data.companies)) {
      return { success: false, message: 'API response must contain a "companies" array' };
    }

    // Parse companies - expecting objects with sage_account_number and company_id
    const allCompanies = data.companies
      .map(c => {
        // Handle both object format and legacy string format
        if (typeof c === 'object' && c !== null) {
          return {
            sage_account_number: c.sage_account_number || c.SAGEACCNUM || null,
            company_id: c.company_id || c.INTACCT_COMPANY_ID || null
          };
        } else if (typeof c === 'string') {
          // Legacy format: just company ID string
          return {
            sage_account_number: null,
            company_id: c
          };
        }
        return null;
      })
      .filter(c => c !== null);

    // Separate valid companies from those with missing company_id
    const validCompanies = [];
    const missingCompanyId = [];

    for (const c of allCompanies) {
      const companyId = c.company_id ? String(c.company_id).trim() : '';
      const isNullOrBlank = !companyId || companyId.toUpperCase() === 'NULL';

      if (isNullOrBlank) {
        // Track companies with missing company_id (use sage_account_number as identifier)
        if (c.sage_account_number) {
          missingCompanyId.push({
            sage_account_number: String(c.sage_account_number).trim()
          });
        }
      } else {
        validCompanies.push({
          sage_account_number: c.sage_account_number ? String(c.sage_account_number).trim() : null,
          company_id: companyId
        });
      }
    }

    // Sort valid companies by company_id
    validCompanies.sort((a, b) =>
      a.company_id.localeCompare(b.company_id, 'en', { sensitivity: 'base', numeric: true })
    );

    if (validCompanies.length === 0) {
      return { success: false, message: 'No valid company IDs found in API response' };
    }

    const inserted = replaceAllCompanies(validCompanies);
    logger.info(`[Scheduler] Loaded ${inserted} companies from API`);

    // Log companies with missing company_id to job_errors
    // Clear previous API_IMPORT errors first
    clearJobErrors(API_IMPORT_QUERY);

    const now = new Date().toISOString();
    for (const company of missingCompanyId) {
      insertJobError({
        company_id: company.sage_account_number,
        query_name: API_IMPORT_QUERY,
        error_type: 'MISSING_COMPANY_ID',
        error_number: null,
        description: 'Missing Intacct company_id in source data',
        description2: 'Add company_id to source system or add to exceptions',
        page: 1,
        recorded_at: now
      });
    }

    if (missingCompanyId.length > 0) {
      logger.info(`[Scheduler] ${missingCompanyId.length} companies have missing company_id (logged to job_errors)`);
    }

    return { success: true, count: inserted, missingCount: missingCompanyId.length };
  } catch (e) {
    return { success: false, message: `Failed to load from API: ${e.message}` };
  }
}

/**
 * Run a single query via cli.js
 * @param {Object} queryEntry - Query config
 * @param {Object} envBase - Base environment variables
 * @param {Object} config - App config
 * @param {boolean} useJobService - If true, pipe output to job service for UI visibility
 * @param {boolean} fullRefresh - If true, set FULL_REFRESH env var to clear and re-fetch data
 */
function runQuery(queryEntry, envBase, config, useJobService = false, fullRefresh = false) {
  return new Promise((resolve, reject) => {
    const envVars = { ...envBase };

    if (config.intacct?.url) envVars.IA_URL = config.intacct.url;
    if (queryEntry.prefix) envVars.QUERY_PREFIX = queryEntry.prefix;
    if (queryEntry.name) envVars.QUERY_NAME = queryEntry.name;
    if (config.intacct?.baseStartDate) envVars.BASE_START_DATE = config.intacct.baseStartDate;
    if (config.intacct?.pageSize) envVars.PAGE_SIZE = String(config.intacct.pageSize);
    if (config.intacct?.fields) envVars.FIELDS = config.intacct.fields;
    if (fullRefresh) envVars.FULL_REFRESH = 'true';

    const nodeBin = process.execPath;
    const cliPath = path.join(__dirname, '..', 'cli.js');
    const args = [cliPath];

    const modeStr = fullRefresh ? 'FULL REFRESH' : 'INCREMENTAL';
    logger.info(`[Scheduler] Running query: ${queryEntry.name} (${modeStr})`);
    if (useJobService) {
      appendLog(`\n========== Running query: ${queryEntry.name} (${modeStr}) ==========\n`);
    }

    const proc = spawn(nodeBin, args, { env: envVars, stdio: ['ignore', 'pipe', 'pipe'] });

    // Store process reference so Stop button works
    setProcRef(proc);

    // Pipe output to the job log (for UI visibility) and/or log to Winston
    proc.stdout.on('data', (d) => {
      const output = d.toString().trim();
      if (useJobService) {
        appendLog(d.toString());
      }
      // For scheduled runs, log important progress messages to Winston
      if (!useJobService && output) {
        // Log lines that indicate progress (company processing, errors, completion)
        const lines = output.split('\n').filter(line => line.trim());
        for (const line of lines) {
          const trimmed = line.trim();
          // Log summary lines, company progress, errors, and key status messages
          if (trimmed.startsWith('===') ||
              trimmed.includes('Companies loaded') ||
              trimmed.includes('Mode:') ||
              trimmed.includes('Top-up from') ||
              trimmed.includes('Initial fetch') ||
              trimmed.includes('completed.') ||
              trimmed.includes('Error') ||
              trimmed.includes('records') ||
              trimmed.includes('Stored') ||
              trimmed.includes('Cleared') ||
              trimmed.includes('Skipping') ||
              trimmed.includes('TOTAL')) {
            logger.info(`[Scheduler] [${queryEntry.name}] ${trimmed}`);
          }
        }
      }
    });
    proc.stderr.on('data', (d) => {
      const output = d.toString().trim();
      if (useJobService) {
        appendLog(d.toString());
      }
      // Always log stderr to Winston for debugging
      if (!useJobService && output) {
        logger.warn(`[Scheduler] [${queryEntry.name}] ${output}`);
      }
    });

    proc.on('close', (code) => {
      setProcRef(null);
      if (useJobService) {
        appendLog(`\n========== Query ${queryEntry.name} completed with exit code: ${code} ==========\n`);
      }
      logger.info(`[Scheduler] Query ${queryEntry.name} completed with exit code: ${code}`);
      if (code === 0) {
        resolve({ success: true, queryName: queryEntry.name });
      } else {
        resolve({ success: false, queryName: queryEntry.name, code });
      }
    });

    proc.on('error', (err) => {
      setProcRef(null);
      if (useJobService) {
        appendLog(`\nError spawning query ${queryEntry.name}: ${err.message}\n`);
      }
      logger.error(`[Scheduler] Error spawning query ${queryEntry.name}: ${err.message}`);
      reject(err);
    });
  });
}

/**
 * Run data fetch job for the specified query filter
 * @param {string} queryFilter - 'all' or specific query name
 * @param {string} scheduleName - Name of the schedule (for logging)
 * @param {boolean} useJobService - If true, integrate with job service for UI visibility (manual runs)
 * @param {boolean} fullRefresh - If true, clear and re-fetch all data
 * @returns {Promise<{success: boolean, message: string}>}
 */
async function runDataFetch(queryFilter, scheduleName = 'Scheduled Job', useJobService = false, fullRefresh = false) {
  // Check if a manual job is already running (only matters for manual runs)
  if (useJobService) {
    const currentJob = getJob();
    if (currentJob.running) {
      return { success: false, message: 'Another job is already running. Please wait for it to complete.' };
    }
  }

  // Get credentials
  const credentials = getAllCredentials();
  if (!credentials || !credentials.senderId || !credentials.senderPassword ||
      !credentials.userId || !credentials.userPassword) {
    return { success: false, message: 'Missing credentials. Please configure credentials first.' };
  }

  // Get config
  const config = getConfig();
  const queries = config.intacct?.queries || {};
  const queryEntries = Object.entries(queries)
    .map(([order, q]) => ({ order: parseInt(order, 10), name: q.name, prefix: q.prefix || '' }))
    .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: 'base' }));

  if (queryEntries.length === 0) {
    return { success: false, message: 'No queries configured' };
  }

  // Filter to specific query if not 'all'
  let queriesToRun = queryEntries;
  if (queryFilter && queryFilter !== 'all') {
    queriesToRun = queryEntries.filter(q => q.name === queryFilter);
    if (queriesToRun.length === 0) {
      return { success: false, message: `Query "${queryFilter}" not found in config` };
    }
  }

  const filterLabel = queryFilter === 'all' ? 'All Filters' : queryFilter;

  // Clear job errors for the queries about to run (for all runs, not just manual)
  if (queryFilter && queryFilter !== 'all') {
    clearJobErrors(queryFilter);
  } else {
    clearJobErrors();
  }

  // Set up job state for UI visibility (only for manual runs)
  if (useJobService) {
    resetJob();
    setJobRunning(true);
    setJobStartedAt(new Date().toISOString());
    setJobQueryName(filterLabel);
    setJobQueryPrefix('');
    logJob(`Job started: ${scheduleName} (${filterLabel}, ${queriesToRun.length} queries)`);
  }

  logger.info(`[Scheduler] Starting data fetch: ${scheduleName} (${filterLabel}, ${queriesToRun.length} queries)`);

  // Build base environment
  const envBase = {};
  const systemEnvVars = ['NODE_ENV', 'PATH', 'HOME', 'USER', 'TEMP', 'TMP', 'TMPDIR', 'DATA_DIR', 'ENCRYPTION_KEY', 'LOG_DIR', 'LOG_LEVEL'];
  for (const key of systemEnvVars) {
    if (process.env[key]) {
      envBase[key] = process.env[key];
    }
  }

  envBase.SENDER_ID = credentials.senderId;
  envBase.SENDER_PASSWORD = credentials.senderPassword;
  envBase.USER_ID = credentials.userId;
  envBase.USER_PASSWORD = credentials.userPassword;

  // Run queries sequentially
  let lastError = null;
  let lastExitCode = 0;
  let wasStopped = false;
  for (const entry of queriesToRun) {
    // Check if job was stopped by user
    if (useJobService && getJob().stopping) {
      logJob('Job stopped by user');
      wasStopped = true;
      break;
    }
    if (!useJobService && isScheduledJobStopping()) {
      logger.info('[Scheduler] Scheduled job stopped by user');
      wasStopped = true;
      break;
    }

    try {
      const result = await runQuery(entry, envBase, config, useJobService, fullRefresh);
      if (!result.success) {
        lastError = `Query ${result.queryName} failed with exit code ${result.code}`;
        lastExitCode = result.code;
      }
    } catch (err) {
      lastError = `Query ${entry.name} error: ${err.message}`;
      lastExitCode = 1;
    }
  }

  // Reload database to pick up new data
  try {
    await reloadDatabase();
    const recordCount = getApiUsageCount();
    if (useJobService) {
      logJob(`Data fetch complete. Records in database: ${recordCount}`);
    }
    logger.info(`[Scheduler] Data fetch complete. Records in database: ${recordCount}`);
  } catch (err) {
    logger.error(`[Scheduler] Failed to reload database: ${err.message}`);
  }

  // Update job state (only for manual runs)
  if (useJobService) {
    setJobExitCode(wasStopped ? -1 : lastExitCode);
    setJobRunning(false);
  }

  if (wasStopped) {
    return { success: false, message: 'Job was stopped by user', stopped: true };
  }

  if (lastError) {
    if (useJobService) {
      logJob(`Job completed with errors: ${lastError}`, 'warn');
    }
    logger.warn(`[Scheduler] Job completed with errors: ${lastError}`);
    return { success: false, message: lastError };
  }

  if (useJobService) {
    logJob('Job completed successfully');
  }
  logger.info(`[Scheduler] Job completed successfully: ${scheduleName}`);
  return { success: true, message: 'Data fetch completed successfully' };
}

/**
 * Generate detailed report as Buffer
 * @param {string} queryFilter - 'all' or specific query name
 * @param {string} startDate - Optional start date (YYYY-MM-DD)
 * @param {string} endDate - Optional end date (YYYY-MM-DD)
 */
async function generateDetailedReport(queryFilter = 'all', startDate = '', endDate = '') {
  const filters = queryFilter && queryFilter !== 'all' ? { query_name: queryFilter } : {};
  if (startDate) filters.start_date = startDate;
  if (endDate) filters.end_date = endDate;
  const records = getApiUsageRecords(filters);
  const workbook = new ExcelJS.Workbook();
  const sheet = workbook.addWorksheet('Detailed');

  sheet.columns = [
    { header: 'Sage Account', key: 'SageAccount', width: 15 },
    { header: 'CompanyID', key: 'CompanyID', width: 30 },
    { header: 'PARTNERID', key: 'PARTNERID', width: 15 },
    { header: 'DOCCONTROLID', key: 'DOCCONTROLID', width: 20 },
    { header: 'N_TRANS', key: 'N_TRANS', width: 10 },
    { header: 'CREATED_DATE', key: 'CREATED_DATE', width: 14 },
    { header: 'CREATED_TIME', key: 'CREATED_TIME', width: 14 },
    { header: 'STATUS', key: 'STATUS', width: 12 },
    { header: 'FILTER', key: 'FILTER', width: 18 }
  ];

  for (const r of records) {
    sheet.addRow({
      SageAccount: r.sage_account_number || '',
      CompanyID: r.company_id,
      PARTNERID: r.partner_id || '',
      DOCCONTROLID: r.doc_control_id,
      N_TRANS: r.n_trans,
      CREATED_DATE: r.created_date ? new Date(r.created_date + 'T00:00:00') : null,
      CREATED_TIME: r.created_time || '',
      STATUS: r.status || '',
      FILTER: r.query_name || ''
    });
  }

  sheet.getColumn('CREATED_DATE').numFmt = 'yyyy-mm-dd';

  return await workbook.xlsx.writeBuffer();
}

/**
 * Generate summary report as Buffer
 * @param {string} queryFilter - 'all' or specific query name
 * @param {string} startDate - Optional start date (YYYY-MM-DD)
 * @param {string} endDate - Optional end date (YYYY-MM-DD)
 */
async function generateSummaryReport(queryFilter = 'all', startDate = '', endDate = '') {
  const queryName = queryFilter && queryFilter !== 'all' ? queryFilter : null;
  const summary = getApiUsageSummary(queryName, startDate || null, endDate || null);
  const workbook = new ExcelJS.Workbook();
  const sheet = workbook.addWorksheet('Summary');

  sheet.columns = [
    { header: 'Sage Account', key: 'SageAccount', width: 15 },
    { header: 'Company', key: 'Company', width: 30 },
    { header: 'Filter', key: 'Filter', width: 18 },
    { header: 'API Usage', key: 'ApiUsage', width: 16 }
  ];

  for (const s of summary) {
    sheet.addRow({
      SageAccount: s.sage_account_number || '',
      Company: s.company_id,
      Filter: s.query_name || '',
      ApiUsage: s.total_trans
    });
  }

  return await workbook.xlsx.writeBuffer();
}

/**
 * Get exception rows - SINGLE SOURCE OF TRUTH for all exception reports
 * Returns array of exception objects for use by downloads, scheduled reports, and email body.
 * Uses getFilteredJobErrors which excludes skip_companies.
 * @param {string} queryFilter - 'all' or specific query name
 * @returns {Array} Array of { sageAccountNumber, companyId, queryName, status, errorNumber, description, description2 }
 */
export function getExceptionRows(queryFilter = 'all') {
  const queryName = queryFilter && queryFilter !== 'all' ? queryFilter : null;
  const jobErrors = getFilteredJobErrors(queryName);

  // Track which companies have errors so we don't double-list them
  const companiesWithErrors = new Set(jobErrors.map(err => err.company_id));

  // Collect all rows: one per error record + one per no-data company without errors
  const rows = [];

  for (const err of jobErrors) {
    rows.push({
      sageAccountNumber: err.sage_account_number || '',
      companyId: err.company_id,
      queryName: err.query_name || '',
      status: err.error_type || 'Error',
      errorNumber: err.error_number || '',
      description: err.description || '',
      description2: err.description2 || ''
    });
  }

  // Only check for "companies with no data" when a specific filter is selected
  // When "All Filters" is selected, this check doesn't make sense since we'd need per-filter logic
  if (queryName) {
    const companiesWithNoData = getCompaniesWithNoData(queryName);
    for (const company of companiesWithNoData) {
      if (!companiesWithErrors.has(company.company_id)) {
        rows.push({
          sageAccountNumber: company.sage_account_number || '',
          companyId: company.company_id,
          queryName,
          status: 'No data returned',
          errorNumber: '',
          description: 'Company returned no API usage data',
          description2: ''
        });
      }
    }
  }

  // Sort all rows by company ID
  rows.sort((a, b) => a.companyId.localeCompare(b.companyId));

  return rows;
}

/**
 * Generate exceptions report as Buffer
 * Uses getExceptionRows (single source of truth)
 * @param {string} queryFilter - 'all' or specific query name
 */
async function generateExceptionsReport(queryFilter = 'all') {
  const rows = getExceptionRows(queryFilter);

  const workbook = new ExcelJS.Workbook();
  const sheet = workbook.addWorksheet('Exceptions');

  sheet.columns = [
    { header: 'Sage Account', key: 'SageAccount', width: 15 },
    { header: 'Company ID', key: 'CompanyID', width: 30 },
    { header: 'Query', key: 'Query', width: 20 },
    { header: 'Status', key: 'Status', width: 20 },
    { header: 'Error No', key: 'ErrorNo', width: 15 },
    { header: 'Description', key: 'Description', width: 50 },
    { header: 'Description 2', key: 'Description2', width: 50 }
  ];

  for (const row of rows) {
    sheet.addRow({
      SageAccount: row.sageAccountNumber || '',
      CompanyID: row.companyId,
      Query: row.queryName,
      Status: row.status,
      ErrorNo: row.errorNumber,
      Description: row.description,
      Description2: row.description2
    });
  }

  return await workbook.xlsx.writeBuffer();
}

/**
 * Build email body with data breakdown
 * @param {string} queryFilter - 'all' or specific query name
 * @param {string[]} selectedAttachments - which reports are attached
 * @param {string} startDate - Optional report start date (YYYY-MM-DD)
 * @param {string} endDate - Optional report end date (YYYY-MM-DD)
 */
/**
 * Resolve report date range - if no dates specified, use min/max from api_usage table
 */
export function resolveReportDateRange(startDate, endDate) {
  if (!startDate && !endDate) {
    const range = getApiUsageDateRange();
    return { startDate: range.min_date || '', endDate: range.max_date || '' };
  }
  return { startDate: startDate || '', endDate: endDate || '' };
}

/**
 * Format a YYYY-MM-DD date string as dd/MM/yyyy for display
 */
function formatDateDisplay(isoDate) {
  if (!isoDate) return '';
  const parts = isoDate.split('-');
  if (parts.length !== 3) return isoDate;
  return `${parts[2]}/${parts[1]}/${parts[0]}`;
}

export function buildEmailBody(queryFilter = 'all', selectedAttachments = ['detailed', 'summary', 'exceptions'], startDate = '', endDate = '') {
  const queryName = queryFilter && queryFilter !== 'all' ? queryFilter : null;
  const totalRecords = getApiUsageCount(queryName, startDate || null, endDate || null);
  const companyCount = getCompanyCount();

  // Use getExceptionRows (single source of truth) to get exception count
  const exceptionRows = getExceptionRows(queryFilter);
  const uniqueCompanies = new Set(exceptionRows.map(row => row.companyId));
  const exceptionCount = uniqueCompanies.size;
  const errorCount = getFilteredJobErrorCount(queryName);
  let queries = getAllQueries();
  const lastFetched = getLastFetchedAtByQuery();

  // Filter to only the selected query if not 'all'
  if (queryName) {
    queries = queries.filter(q => q.name === queryName);
  }
  const summary = getApiUsageSummary(queryName, startDate || null, endDate || null);

  // Build date range label (display as dd/MM/yyyy)
  let dateRangeLabel = '';
  if (startDate && endDate) {
    dateRangeLabel = ` (${formatDateDisplay(startDate)} to ${formatDateDisplay(endDate)})`;
  } else if (startDate) {
    dateRangeLabel = ` (from ${formatDateDisplay(startDate)})`;
  } else if (endDate) {
    dateRangeLabel = ` (up to ${formatDateDisplay(endDate)})`;
  }

  // Calculate totals by filter
  const filterTotals = {};
  for (const s of summary) {
    const filter = s.query_name || 'Unknown';
    if (!filterTotals[filter]) {
      filterTotals[filter] = { trans: 0, companies: 0 };
    }
    filterTotals[filter].trans += s.total_trans;
    filterTotals[filter].companies += 1;
  }

  const now = new Date();
  const dateStr = now.toLocaleDateString('en-GB');
  const timeStr = now.toLocaleTimeString();

  let html = `
    <h2>Intacct API Usage Report</h2>
    <p>Generated: ${dateStr} ${timeStr}</p>
    ${dateRangeLabel ? `<p><strong>Date Range:</strong>${dateRangeLabel}</p>` : ''}

    <h3>Summary</h3>
    <table style="border-collapse: collapse; margin-bottom: 20px;">
      <tr>
        <td style="padding: 5px 15px 5px 0;"><strong>Total API Usage${dateRangeLabel}:</strong></td>
        <td style="padding: 5px 0;">${totalRecords.toLocaleString()}</td>
      </tr>
      <tr>
        <td style="padding: 5px 15px 5px 0;"><strong>Companies:</strong></td>
        <td style="padding: 5px 0;">${companyCount.toLocaleString()}</td>
      </tr>
      <tr>
        <td style="padding: 5px 15px 5px 0;"><strong>Exceptions:</strong></td>
        <td style="padding: 5px 0;">${exceptionCount.toLocaleString()}</td>
      </tr>
      <tr>
        <td style="padding: 5px 15px 5px 0;"><strong>Errors:</strong></td>
        <td style="padding: 5px 0;">${errorCount.toLocaleString()}</td>
      </tr>
    </table>

    <h3>Breakdown by Filter</h3>
    <table style="border-collapse: collapse; margin-bottom: 20px;">
      <tr style="background: #00AFAA; color: white;">
        <th style="padding: 8px; text-align: left; border: 1px solid #ddd;">Filter</th>
        <th style="padding: 8px; text-align: right; border: 1px solid #ddd;">Companies/Instances</th>
        <th style="padding: 8px; text-align: right; border: 1px solid #ddd;">API Usage</th>
        <th style="padding: 8px; text-align: left; border: 1px solid #ddd;">Last Fetched</th>
      </tr>
  `;

  for (const q of queries) {
    const totals = filterTotals[q.name] || { trans: 0, companies: 0 };
    const lastFetch = lastFetched[q.name];
    const lastFetchStr = lastFetch ? new Date(lastFetch).toLocaleString('en-GB') : 'Never';

    html += `
      <tr>
        <td style="padding: 8px; border: 1px solid #ddd;">${q.name}</td>
        <td style="padding: 8px; text-align: right; border: 1px solid #ddd;">${totals.companies.toLocaleString()}</td>
        <td style="padding: 8px; text-align: right; border: 1px solid #ddd;">${totals.trans.toLocaleString()}</td>
        <td style="padding: 8px; border: 1px solid #ddd;">${lastFetchStr}</td>
      </tr>
    `;
  }

  html += `
    </table>

    <p style="color: #666; font-size: 12px;">
      This is an automated report from Intacct API Usage.
      ${selectedAttachments.length > 0
        ? `Attached reports: ${selectedAttachments.map(a => a.charAt(0).toUpperCase() + a.slice(1)).join(', ')}.`
        : ''}
    </p>
  `;

  return html;
}

/**
 * Generate report attachments for email notifications
 */
export async function generateReportAttachments(queryFilter, selectedAttachments, startDate = '', endDate = '') {
  const dateStr = new Date().toISOString().slice(0, 10);
  const filterLabel = queryFilter === 'all' ? 'All' : queryFilter;
  const attachments = [];

  if (selectedAttachments.includes('detailed')) {
    const buffer = await generateDetailedReport(queryFilter, startDate, endDate);
    attachments.push({
      filename: `Intacct-Detailed-${filterLabel}-${dateStr}.xlsx`,
      content: Buffer.from(buffer),
      contentType: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    });
  }
  if (selectedAttachments.includes('summary')) {
    const buffer = await generateSummaryReport(queryFilter, startDate, endDate);
    attachments.push({
      filename: `Intacct-Summary-${filterLabel}-${dateStr}.xlsx`,
      content: Buffer.from(buffer),
      contentType: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    });
  }
  if (selectedAttachments.includes('exceptions')) {
    const buffer = await generateExceptionsReport(queryFilter);
    attachments.push({
      filename: `Intacct-Exceptions-${filterLabel}-${dateStr}.xlsx`,
      content: Buffer.from(buffer),
      contentType: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    });
  }

  return attachments;
}

/**
 * Execute a single scheduled report (autonomous - doesn't use job service)
 * @param {boolean} isManualRun - If true, this is a manual "Run Now" trigger that uses job service
 */
async function executeSchedule(schedule, isManualRun = false) {
  logger.info(`Executing scheduled report: ${schedule.name} (ID: ${schedule.id}, manual: ${isManualRun})`);

  // Check if a manual job is already running
  const currentJob = getJob();
  if (currentJob.running && !isManualRun) {
    // A manual job is running - reschedule this for next time
    logger.info(`[Scheduler] Skipping ${schedule.name} - manual job in progress. Will run at next scheduled time.`);
    updateScheduleAfterRun(schedule.id, schedule.lastRunStatus, 'Skipped - manual job in progress');
    return;
  }

  // Mark schedule as running
  setScheduledJobRunning(true);
  updateScheduleAfterRun(schedule.id, 'running', 'Job in progress...');

  try {
    // Support both legacy single ID and new array of IDs
    const notifConfigIds = schedule.notificationConfigIds?.length > 0
      ? schedule.notificationConfigIds
      : (schedule.notificationConfigId ? [schedule.notificationConfigId] : []);

    if (notifConfigIds.length === 0) {
      throw new Error('No notification configuration assigned to schedule');
    }

    const queryFilter = schedule.queryFilter || 'all';

    // Check if this is a "reports only" schedule (skip data fetch)
    if (schedule.reportsOnly) {
      logger.info(`[Scheduler] Reports-only mode: skipping data fetch for ${schedule.name}`);
      if (isManualRun) {
        logJob('Reports-only mode: sending reports from existing data');
      }
    } else {
      // Step 0: Check if API mode is configured and load companies from API
      const config = getConfig();
      const companiesSource = config.intacct?.companiesSource || 'csv';
      const companiesApiUrl = config.intacct?.companiesApiUrl || '';
      const companiesApiToken = getCredential('companiesApiToken') || '';

      logger.info(`[Scheduler] Companies source: ${companiesSource}, API URL: ${companiesApiUrl || '(not set)'}`);

      if (companiesSource === 'api') {
        if (!companiesApiUrl) {
          throw new Error('API mode is configured but no API URL is set. Configure the Companies API URL in Configuration.');
        }

        // Note: loadCompaniesFromApi logs the URL and count
        if (isManualRun) {
          logJob('Loading companies from API...');
        }

        const apiResult = await loadCompaniesFromApi(companiesApiUrl, companiesApiToken);
        if (!apiResult.success) {
          throw new Error(`Failed to load companies from API: ${apiResult.message}`);
        }

        // Note: loadCompaniesFromApi already logs the count
        if (isManualRun) {
          logJob(`Loaded ${apiResult.count} companies from API`);
        }
      }

      // Step 0.5: Pre-check credentials before running the job
      const credentials = getAllCredentials();
      if (!credentials || !credentials.senderId || !credentials.senderPassword ||
          !credentials.userId || !credentials.userPassword) {
        const errorMsg = 'Missing Intacct credentials. Please configure credentials in Configuration.';
        throw new Error(errorMsg);
      }

      // Test credentials with Datel Group - a known company we always have access to
      const testCompanyId = 'Datel Group';

      if (isManualRun) {
        logJob('Validating credentials...');
      }
      logger.info(`[Scheduler] Testing credentials${testCompanyId ? ` with company "${testCompanyId}"` : ''}...`);

      const credResult = await testCredentials({
        senderId: credentials.senderId,
        senderPassword: credentials.senderPassword,
        userId: credentials.userId,
        userPassword: credentials.userPassword,
        testCompanyId
      });

      if (!credResult.ok) {
        // Credential failure - send notification and abort
        logger.error(`[Scheduler] Credential check failed: ${credResult.message}`);

        if (notifConfigIds.length > 0) {
          // Send failure notification to all configured recipients
          const dateStr = new Date().toISOString().slice(0, 10);
          const timeStr = new Date().toLocaleTimeString();
          const subject = `⚠️ Intacct API Usage - Scheduled Job Failed (Credentials) - ${dateStr}`;
          const body = `
            <h2>Scheduled Job Failed</h2>
            <p><strong>Schedule:</strong> ${schedule.name}</p>
            <p><strong>Date:</strong> ${dateStr} ${timeStr}</p>
            <p><strong>Error:</strong> ${credResult.message}</p>
            <h3>Recommended Action</h3>
            <p>Please check and update your Intacct credentials in the Configuration page.</p>
            <p style="color: #666; font-size: 12px;">This is an automated notification from Intacct API Usage.</p>
          `;
          for (const configId of notifConfigIds) {
            try {
              await sendNotification(configId, subject, body, []);
              logger.info(`[Scheduler] Credential failure notification sent to config ${configId} for schedule: ${schedule.name}`);
            } catch (notifError) {
              logger.error(`[Scheduler] Failed to send credential failure notification to config ${configId}: ${notifError.message}`);
            }
          }
        }

        throw new Error(`Credential validation failed: ${credResult.message}`);
      }

      if (isManualRun) {
        logJob('Credentials validated');
      }
      logger.info(`[Scheduler] Credentials validated successfully`);

      // Step 1: Run data fetch for the configured query filter
      logger.info(`[Scheduler] Fetching data for filter: ${queryFilter}`);

      const fetchResult = await runDataFetch(queryFilter, schedule.name, isManualRun, schedule.fullRefresh || false);
      if (!fetchResult.success) {
        if (fetchResult.stopped) {
          // User stopped the job - mark as stopped, not error
          updateScheduleAfterRun(schedule.id, 'stopped', 'Job stopped by user');
          logger.info(`Scheduled job stopped by user: ${schedule.name}`);
          return;
        }
        throw new Error(`Data fetch failed: ${fetchResult.message}`);
      }
    }

    // Step 2: Generate selected reports
    const selectedAttachments = schedule.attachments || ['detailed', 'summary', 'exceptions'];
    let resolvedEndDate = schedule.reportEndDate || '';
    if (schedule.completeMonthsOnly) {
      // Last day of previous month
      const now = new Date();
      resolvedEndDate = new Date(now.getFullYear(), now.getMonth(), 0).toISOString().slice(0, 10);
    }
    const { startDate: reportStartDate, endDate: reportEndDate } = resolveReportDateRange(
      schedule.reportStartDate || '', resolvedEndDate
    );
    logger.info(`[Scheduler] Generating reports for ${schedule.name} (${selectedAttachments.join(', ')})...`);
    if (isManualRun) {
      logJob(`Generating reports (${selectedAttachments.join(', ')})...`);
    }

    const attachments = await generateReportAttachments(queryFilter, selectedAttachments, reportStartDate, reportEndDate);

    logger.info(`[Scheduler] ${attachments.length} report(s) generated. Sending email for ${schedule.name}...`);
    if (isManualRun) {
      logJob(`${attachments.length} report(s) generated. Sending email...`);
    }

    const dateStr = new Date().toISOString().slice(0, 10);
    const filterLabel = queryFilter === 'all' ? 'All' : queryFilter;
    const subject = `Intacct API Usage Report (${filterLabel}) - ${dateStr}`;
    const body = buildEmailBody(queryFilter, selectedAttachments, reportStartDate, reportEndDate);

    // Send to all configured notification configs
    let sentCount = 0;
    let errorMessages = [];
    for (const configId of notifConfigIds) {
      try {
        await sendNotification(configId, subject, body, attachments);
        sentCount++;
        logger.info(`[Scheduler] Email sent to config ${configId} for schedule: ${schedule.name}`);
      } catch (notifError) {
        logger.error(`[Scheduler] Failed to send to config ${configId}: ${notifError.message}`);
        errorMessages.push(notifError.message);
      }
    }

    if (sentCount === 0) {
      throw new Error(`Failed to send to any notification config: ${errorMessages.join(', ')}`);
    }

    if (isManualRun) {
      const plural = sentCount > 1 ? ` to ${sentCount} recipients` : '';
      logJob(`Email sent successfully${plural}!`);
    }

    const successMsg = sentCount === notifConfigIds.length
      ? 'Report sent successfully'
      : `Report sent to ${sentCount} of ${notifConfigIds.length} recipients`;
    updateScheduleAfterRun(schedule.id, 'success', successMsg);
    logger.info(`Scheduled report completed: ${schedule.name}`);

    // Auto-disable one-off schedules after successful run
    if (schedule.scheduleType === 'oneoff') {
      updateSchedule(schedule.id, { enabled: false });
      logger.info(`One-off schedule disabled after run: ${schedule.name}`);
      if (isManualRun) {
        logJob('One-off schedule has been automatically disabled.');
      }
    }

  } catch (error) {
    if (isManualRun) {
      logJob(`Scheduled report failed: ${error.message}`, 'error');
    }
    logger.error(`Scheduled report failed: ${schedule.name} - ${error.message}`);
    updateScheduleAfterRun(schedule.id, 'error', error.message);
  } finally {
    setScheduledJobRunning(false);
  }
}

/**
 * Check for and execute due schedules
 */
async function checkSchedules() {
  if (isRunning) {
    return;
  }

  isRunning = true;

  try {
    const dueSchedules = getDueSchedules();

    for (const schedule of dueSchedules) {
      await executeSchedule(schedule);
    }
  } catch (error) {
    logger.error(`Scheduler check error: ${error.message}`);
  } finally {
    isRunning = false;
  }
}

/**
 * Start the scheduler (runs every minute)
 */
export function startScheduler() {
  if (schedulerInterval) {
    return;
  }

  logger.info('Starting scheduler service');
  schedulerInterval = setInterval(checkSchedules, 60 * 1000);

  // Also run immediately on startup
  checkSchedules();
}

/**
 * Stop the scheduler
 */
export function stopScheduler() {
  if (schedulerInterval) {
    clearInterval(schedulerInterval);
    schedulerInterval = null;
    logger.info('Scheduler service stopped');
  }
}

/**
 * Stop a currently running scheduled job
 * If no job is actually running but the database has stale "running" status, reset it.
 */
export function stopScheduledJob() {
  if (!isScheduledJobRunning()) {
    // No job is actually running in memory - check for stale "running" status in DB
    const resetCount = resetStaleRunningSchedules();
    if (resetCount > 0) {
      logger.info(`[Scheduler] Reset ${resetCount} stale running schedule(s) to error status`);
      return { reset: true, message: `Reset ${resetCount} stale schedule(s). The job was not actually running (likely interrupted by service restart).` };
    }
    throw new Error('No scheduled job is currently running.');
  }

  const currentJob = getJob();

  // If it's a manual run (uses job service), use the job stopping mechanism
  if (currentJob.running) {
    setJobStopping(true);
  } else {
    // Autonomous scheduled run - use the scheduled stopping flag
    setScheduledJobStopping(true);
  }

  // Kill the current process
  const procRef = getProcRef();
  if (procRef) {
    logger.warn('[Scheduler] Stop requested - sending SIGINT');
    procRef.kill('SIGINT');
    setTimeout(() => {
      const ref = getProcRef();
      if (ref && !ref.killed) {
        try { process.kill(ref.pid, 'SIGKILL'); } catch {}
      }
    }, 2000);
  }

  return { reset: false, message: 'Stop signal sent.' };
}
