// about.js - About tab module

/**
 * Initialize the about tab
 */
export async function initAbout() {
  loadAboutInfo();
}

/**
 * Cleanup when leaving the about tab
 */
export function cleanupAbout() {
  // Nothing to cleanup
}

async function loadAboutInfo() {
  const infoEl = document.getElementById("about-info");
  if (!infoEl) return;

  try {
    const response = await fetch("/api/about");
    const info = await response.json();

    let html = `
      <div class="about-section">
        <p><strong>Version:</strong> ${info.version}</p>
        <p><strong>Description:</strong> ${info.description || "Intacct API Usage Export"}</p>
        <p><strong>Node.js:</strong> ${info.nodeVersion || "Unknown"}</p>
      </div>

      <div class="about-section">
        <h3>Environment</h3>
        <p><strong>Hostname:</strong> ${info.environment?.hostname || "Unknown"}</p>
        <p><strong>Server Address:</strong> ${info.environment?.ipAddresses?.length ? info.environment.ipAddresses.map(ip => ip + ":" + info.environment.port).join(", ") : "localhost:" + (info.environment?.port || 5050)}</p>
        <p><strong>Operating System:</strong> ${info.environment?.os || "Unknown"}</p>
        ${info.environment?.distro ? "<p><strong>Distribution:</strong> " + info.environment.distro + "</p>" : ""}
        <p><strong>Platform:</strong> ${info.environment?.platform || "Unknown"}</p>
        <p><strong>Architecture:</strong> ${info.environment?.architecture || "Unknown"}</p>
        <p><strong>Running in Docker:</strong> ${info.environment?.isDocker ? "Yes" : "No"}</p>
      </div>
    `;

    if (info.dependencies && Object.keys(info.dependencies).length > 0) {
      html += `
        <div class="about-section">
          <h3>Dependencies</h3>
          <div class="tablewrap">
            <table>
              <thead>
                <tr>
                  <th>Package</th>
                  <th>Version</th>
                </tr>
              </thead>
              <tbody>
      `;

      Object.entries(info.dependencies).sort().forEach(([pkg, version]) => {
        html += `
          <tr>
            <td>${pkg}</td>
            <td>${version}</td>
          </tr>
        `;
      });

      html += `
              </tbody>
            </table>
          </div>
        </div>
      `;
    }

    // Add Third-Party Licenses section
    html += `
      <div class="about-section">
        <h3>Third-Party Licenses</h3>
        <p class="hint">This application uses open source packages. Notable license attributions:</p>
        <div class="tablewrap">
          <table>
            <thead>
              <tr>
                <th>Package</th>
                <th>License</th>
                <th>Copyright</th>
              </tr>
            </thead>
            <tbody>
              <tr>
                <td>express</td>
                <td>MIT</td>
                <td>TJ Holowaychuk</td>
              </tr>
              <tr>
                <td>axios</td>
                <td>MIT</td>
                <td>Matt Zabriskie</td>
              </tr>
              <tr>
                <td>fast-xml-parser</td>
                <td>MIT</td>
                <td>Amit Gupta</td>
              </tr>
              <tr>
                <td>json2csv</td>
                <td>MIT</td>
                <td>Mirco Zeiss</td>
              </tr>
              <tr>
                <td>multer</td>
                <td>MIT</td>
                <td>Hage Yaapa</td>
              </tr>
              <tr>
                <td>cors</td>
                <td>MIT</td>
                <td>Troy Goode</td>
              </tr>
              <tr>
                <td>dotenv</td>
                <td>BSD-2-Clause</td>
                <td>Scott Motte</td>
              </tr>
              <tr>
                <td>sql.js</td>
                <td>MIT</td>
                <td>SQL.js Contributors</td>
              </tr>
              <tr>
                <td>winston</td>
                <td>MIT</td>
                <td>Charlie Robbins</td>
              </tr>
            </tbody>
          </table>
        </div>
        <p class="muted">All dependencies are licensed under MIT, ISC, BSD, or similarly permissive licenses.</p>
      </div>
    `;

    // Add Copyright section
    const currentYear = new Date().getFullYear();
    html += `
      <div class="about-section copyright">
        <p>Copyright ${currentYear} Datel Computing Limited</p>
      </div>
    `;

    infoEl.innerHTML = html;
  } catch (error) {
    infoEl.innerHTML = '<p class="err">Error loading application information: ' + error.message + '</p>';
  }
}
