// config.js - Configuration tab module

const $ = (s) => document.querySelector(s);

let editingFilterOrder = null;
let hasCompaniesFile = false;
let hasCredentials = false;
let hasSettings = false;
let hasFilters = false;
let hasProxy = false;

/**
 * Initialize the config tab
 */
export async function initConfig() {
  loadSavedCredentials();
  loadProxy();
  loadSettings();
  loadFilters();
  loadCompaniesApiUrl();
  checkExistingCompanies();
  loadExceptionsCount();
  setupEventListeners();
}

/**
 * Cleanup when leaving the config tab
 */
export function cleanupConfig() {
  // Nothing to cleanup
}

function setupEventListeners() {
  const saveCredsBtn = $("#saveCredsBtn");
  const clearCredsBtn = $("#clearCredsBtn");
  const saveSettingsBtn = $("#saveSettingsBtn");
  const resetSettingsBtn = $("#resetSettingsBtn");
  const addFilterBtn = $("#addFilterBtn");
  const resetFiltersBtn = $("#resetFiltersBtn");
  const saveFilterBtn = $("#saveFilterBtn");
  const cancelFilterBtn = $("#cancelFilterBtn");

  if (saveCredsBtn) saveCredsBtn.onclick = saveCredentials;
  if (clearCredsBtn) clearCredsBtn.onclick = clearSavedCredentials;

  const editCredsBtn = $("#editCredsBtn");
  const cancelCredsBtn = $("#cancelCredsBtn");
  if (editCredsBtn) editCredsBtn.onclick = expandCredentials;
  if (cancelCredsBtn) cancelCredsBtn.onclick = cancelEditCredentials;
  if (saveSettingsBtn) saveSettingsBtn.onclick = saveSettings;
  if (resetSettingsBtn) resetSettingsBtn.onclick = resetSettings;

  const editSettingsBtn = $("#editSettingsBtn");
  const cancelSettingsBtn = $("#cancelSettingsBtn");
  if (editSettingsBtn) editSettingsBtn.onclick = expandSettings;
  if (cancelSettingsBtn) cancelSettingsBtn.onclick = cancelEditSettings;

  const editFiltersBtn = $("#editFiltersBtn");
  const cancelFiltersBtn = $("#cancelFiltersBtn");
  if (editFiltersBtn) editFiltersBtn.onclick = expandFilters;
  if (cancelFiltersBtn) cancelFiltersBtn.onclick = cancelEditFilters;
  if (addFilterBtn) addFilterBtn.onclick = openAddModal;
  if (resetFiltersBtn) resetFiltersBtn.onclick = resetFilters;
  if (saveFilterBtn) saveFilterBtn.onclick = saveFilter;
  if (cancelFilterBtn) cancelFilterBtn.onclick = closeFilterModal;

  // Companies / exceptions
  const manageBtn = $("#manageCompaniesBtn");
  const closeModalBtn = $("#closeCompaniesModalBtn");
  const addCompanyBtn = $("#addCompanyBtn");
  const addCompanyInput = $("#addCompanyInput");
  const clearAllBtn = $("#clearAllBtn");
  const confirmClearAllBtn = $("#confirmClearAllBtn");
  const cancelClearAllBtn = $("#cancelClearAllBtn");

  if (manageBtn) manageBtn.onclick = openCompaniesModal;
  if (closeModalBtn) closeModalBtn.onclick = closeCompaniesModal;
  if (addCompanyBtn) addCompanyBtn.onclick = addCompanyFromModal;
  if (addCompanyInput) {
    addCompanyInput.onkeydown = (e) => {
      if (e.key === 'Enter') addCompanyFromModal();
    };
  }
  const companySearchInput = $("#companySearchInput");
  if (companySearchInput) companySearchInput.oninput = filterCompanies;
  if (clearAllBtn) clearAllBtn.onclick = openClearAllModal;
  if (confirmClearAllBtn) confirmClearAllBtn.onclick = handleClearAll;
  if (cancelClearAllBtn) cancelClearAllBtn.onclick = closeClearAllModal;

  const exceptionsBtn = $("#manageExceptionsBtn");
  const closeExceptionsBtn = $("#closeExceptionsModalBtn");
  if (exceptionsBtn) exceptionsBtn.onclick = openExceptionsModal;
  if (closeExceptionsBtn) closeExceptionsBtn.onclick = closeExceptionsModal;

  // API companies source
  const loadFromApiBtn = $("#loadFromApiBtn");
  const companiesApiUrlInput = $("#companiesApiUrl");
  const companiesApiTokenInput = $("#companiesApiToken");
  if (loadFromApiBtn) loadFromApiBtn.onclick = loadCompaniesFromApi;
  if (companiesApiUrlInput) companiesApiUrlInput.onblur = saveCompaniesApiUrl;
  if (companiesApiTokenInput) companiesApiTokenInput.onblur = saveCompaniesApiUrl;

  // Proxy configuration
  const saveProxyBtn = $("#saveProxyBtn");
  const clearProxyBtn = $("#clearProxyBtn");
  const editProxyBtn = $("#editProxyBtn");
  const cancelProxyBtn = $("#cancelProxyBtn");
  const proxyEnabledToggle = $("#proxyEnabledToggle");
  if (saveProxyBtn) saveProxyBtn.onclick = saveProxy;
  if (clearProxyBtn) clearProxyBtn.onclick = clearProxy;
  if (editProxyBtn) editProxyBtn.onclick = expandProxy;
  if (cancelProxyBtn) cancelProxyBtn.onclick = cancelEditProxy;
  if (proxyEnabledToggle) proxyEnabledToggle.onchange = toggleProxyEnabled;
}

function showMessage(_el, text, type) {
  // Skip empty messages
  if (!text) return;

  // Use global status modal
  if (window.showStatusModal) {
    window.showStatusModal(text, type);
  }
}

async function loadSavedCredentials() {
  const credsMsg = $("#credsMsg");
  const saveCredsBtn = $("#saveCredsBtn");

  try {
    const r = await fetch("/credentials");
    const j = await r.json();

    if (j.encryptionConfigured === false) {
      showMessage(credsMsg, "ENCRYPTION_KEY not configured. Set it in .env or Docker environment to save credentials.", "error");
      if (saveCredsBtn) saveCredsBtn.disabled = true;
      return;
    }

    if (j.ok && j.credentials) {
      const senderIdInput = $("#senderId");
      const senderPasswordInput = $("#senderPassword");
      const userIdInput = $("#userId");
      const userPasswordInput = $("#userPassword");

      if (senderIdInput) senderIdInput.value = j.credentials.senderId || "";
      if (senderPasswordInput) senderPasswordInput.value = j.credentials.senderPassword || "";
      if (userIdInput) userIdInput.value = j.credentials.userId || "";
      if (userPasswordInput) userPasswordInput.value = j.credentials.userPassword || "";

      // Check if all credentials are filled - if so, collapse the section
      if (j.credentials.senderId && j.credentials.senderPassword &&
          j.credentials.userId && j.credentials.userPassword) {
        hasCredentials = true;
        collapseCredentials();
      }
    }
  } catch (e) {
    console.warn("Failed to load saved credentials:", e);
  }
}

function collapseCredentials() {
  const credsCollapsed = $("#credsCollapsed");
  const credsExpanded = $("#credsExpanded");
  const editCredsBtn = $("#editCredsBtn");
  const cancelCredsBtn = $("#cancelCredsBtn");

  if (credsCollapsed) credsCollapsed.classList.remove("hidden");
  if (credsExpanded) credsExpanded.classList.add("hidden");
  if (editCredsBtn) editCredsBtn.classList.remove("hidden");
  if (cancelCredsBtn) cancelCredsBtn.classList.add("hidden");
}

function expandCredentials() {
  const credsCollapsed = $("#credsCollapsed");
  const credsExpanded = $("#credsExpanded");
  const editCredsBtn = $("#editCredsBtn");
  const cancelCredsBtn = $("#cancelCredsBtn");

  if (credsCollapsed) credsCollapsed.classList.add("hidden");
  if (credsExpanded) credsExpanded.classList.remove("hidden");
  if (editCredsBtn) editCredsBtn.classList.add("hidden");
  // Show cancel button only if credentials already exist
  if (cancelCredsBtn && hasCredentials) cancelCredsBtn.classList.remove("hidden");
}

async function cancelEditCredentials() {
  // Reload credentials from server to restore original values
  await loadSavedCredentials();
}

function collapseSettings() {
  const settingsCollapsed = $("#settingsCollapsed");
  const settingsExpanded = $("#settingsExpanded");
  const editSettingsBtn = $("#editSettingsBtn");
  const cancelSettingsBtn = $("#cancelSettingsBtn");

  if (settingsCollapsed) settingsCollapsed.classList.remove("hidden");
  if (settingsExpanded) settingsExpanded.classList.add("hidden");
  if (editSettingsBtn) editSettingsBtn.classList.remove("hidden");
  if (cancelSettingsBtn) cancelSettingsBtn.classList.add("hidden");
}

function expandSettings() {
  const settingsCollapsed = $("#settingsCollapsed");
  const settingsExpanded = $("#settingsExpanded");
  const editSettingsBtn = $("#editSettingsBtn");
  const cancelSettingsBtn = $("#cancelSettingsBtn");

  if (settingsCollapsed) settingsCollapsed.classList.add("hidden");
  if (settingsExpanded) settingsExpanded.classList.remove("hidden");
  if (editSettingsBtn) editSettingsBtn.classList.add("hidden");
  if (cancelSettingsBtn && hasSettings) cancelSettingsBtn.classList.remove("hidden");
}

async function cancelEditSettings() {
  // Reload settings from server to restore original values
  await loadSettings();
}

function collapseFilters() {
  const filtersCollapsed = $("#filtersCollapsed");
  const filtersExpanded = $("#filtersExpanded");
  const editFiltersBtn = $("#editFiltersBtn");
  const cancelFiltersBtn = $("#cancelFiltersBtn");

  if (filtersCollapsed) filtersCollapsed.classList.remove("hidden");
  if (filtersExpanded) filtersExpanded.classList.add("hidden");
  if (editFiltersBtn) editFiltersBtn.classList.remove("hidden");
  if (cancelFiltersBtn) cancelFiltersBtn.classList.add("hidden");
}

function expandFilters() {
  const filtersCollapsed = $("#filtersCollapsed");
  const filtersExpanded = $("#filtersExpanded");
  const editFiltersBtn = $("#editFiltersBtn");
  const cancelFiltersBtn = $("#cancelFiltersBtn");

  if (filtersCollapsed) filtersCollapsed.classList.add("hidden");
  if (filtersExpanded) filtersExpanded.classList.remove("hidden");
  if (editFiltersBtn) editFiltersBtn.classList.add("hidden");
  if (cancelFiltersBtn && hasFilters) cancelFiltersBtn.classList.remove("hidden");
}

async function cancelEditFilters() {
  // Reload filters from server to restore original values
  await loadFilters();
}

async function saveCredentials() {
  const credsMsg = $("#credsMsg");
  const saveCredsBtn = $("#saveCredsBtn");
  const senderIdInput = $("#senderId");
  const senderPasswordInput = $("#senderPassword");
  const userIdInput = $("#userId");
  const userPasswordInput = $("#userPassword");

  const credsToSave = {
    senderId: senderIdInput?.value.trim(),
    senderPassword: senderPasswordInput?.value.trim(),
    userId: userIdInput?.value.trim(),
    userPassword: userPasswordInput?.value.trim()
  };

  if (!credsToSave.senderId || !credsToSave.senderPassword ||
      !credsToSave.userId || !credsToSave.userPassword) {
    showMessage(credsMsg, "Please fill in all credential fields", "error");
    return;
  }

  if (saveCredsBtn) saveCredsBtn.disabled = true;
  showMessage(credsMsg, "Testing credentials...", "info");

  try {
    // Test credentials first
    const testResult = await fetch("/test-credentials", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(credsToSave)
    });
    const testJson = await testResult.json();

    if (!testJson.ok) {
      showMessage(credsMsg, testJson.message || "Invalid credentials", "error");
      if (saveCredsBtn) saveCredsBtn.disabled = false;
      return;
    }

    // Credentials valid - now save them
    showMessage(credsMsg, "Saving credentials...", "info");
    const r = await fetch("/credentials", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(credsToSave)
    });
    const j = await r.json();

    if (j.ok) {
      showMessage(credsMsg, "Credentials validated and saved", "success");
      hasCredentials = true;
      collapseCredentials();
    } else {
      showMessage(credsMsg, j.message || "Failed to save credentials", "error");
    }
  } catch (e) {
    showMessage(credsMsg, "Failed to save credentials: " + e.message, "error");
  } finally {
    if (saveCredsBtn) saveCredsBtn.disabled = false;
  }
}

async function clearSavedCredentials() {
  const credsMsg = $("#credsMsg");
  const clearCredsBtn = $("#clearCredsBtn");

  if (clearCredsBtn) clearCredsBtn.disabled = true;

  try {
    const r = await fetch("/credentials", { method: "DELETE" });
    const j = await r.json();

    if (j.ok) {
      const senderIdInput = $("#senderId");
      const senderPasswordInput = $("#senderPassword");
      const userIdInput = $("#userId");
      const userPasswordInput = $("#userPassword");

      if (senderIdInput) senderIdInput.value = "";
      if (senderPasswordInput) senderPasswordInput.value = "";
      if (userIdInput) userIdInput.value = "";
      if (userPasswordInput) userPasswordInput.value = "";
      showMessage(credsMsg, "Saved credentials cleared", "success");
      hasCredentials = false;
      expandCredentials();
    } else {
      showMessage(credsMsg, j.message || "Failed to clear credentials", "error");
    }
  } catch (e) {
    showMessage(credsMsg, "Failed to clear credentials: " + e.message, "error");
  } finally {
    if (clearCredsBtn) clearCredsBtn.disabled = false;
  }
}

// ── Proxy Configuration ──────────────────────────────────────────

async function loadProxy() {
  try {
    const r = await fetch("/proxy");
    const j = await r.json();

    const proxyStatusConfigured = $("#proxyStatusConfigured");
    const proxyStatusNone = $("#proxyStatusNone");
    const proxyToggleContainer = $("#proxyToggleContainer");

    if (j.ok && j.proxy && j.proxy.url) {
      const proxyUrlInput = $("#proxyUrl");
      const proxyUsernameInput = $("#proxyUsername");
      const proxyPasswordInput = $("#proxyPassword");
      const proxyStatusHost = $("#proxyStatusHost");
      const proxyStatusState = $("#proxyStatusState");
      const proxyEnabledToggle = $("#proxyEnabledToggle");

      if (proxyUrlInput) proxyUrlInput.value = j.proxy.url || "";
      if (proxyUsernameInput) proxyUsernameInput.value = j.proxy.username || "";
      // Don't populate password - just show placeholder if set
      if (proxyPasswordInput && j.proxy.hasPassword) {
        proxyPasswordInput.placeholder = "••••••••";
      }

      // Set enabled state on header toggle
      const isEnabled = j.proxy.enabled !== false;
      if (proxyEnabledToggle) proxyEnabledToggle.checked = isEnabled;

      // Update status display
      if (proxyStatusHost) {
        try {
          const url = new URL(j.proxy.url);
          proxyStatusHost.textContent = url.host;
        } catch {
          proxyStatusHost.textContent = j.proxy.url;
        }
      }

      // Show enabled/disabled state in collapsed view
      if (proxyStatusState) {
        proxyStatusState.textContent = isEnabled ? "(enabled)" : "(disabled)";
        proxyStatusState.className = isEnabled ? "proxy-state enabled" : "proxy-state disabled";
      }

      hasProxy = true;
      // Show configured status, hide "no proxy" message, show toggle
      if (proxyStatusConfigured) proxyStatusConfigured.classList.remove("hidden");
      if (proxyStatusNone) proxyStatusNone.classList.add("hidden");
      if (proxyToggleContainer) proxyToggleContainer.classList.remove("hidden");
    } else {
      hasProxy = false;
      // Show "no proxy" message, hide configured status, hide toggle
      if (proxyStatusConfigured) proxyStatusConfigured.classList.add("hidden");
      if (proxyStatusNone) proxyStatusNone.classList.remove("hidden");
      if (proxyToggleContainer) proxyToggleContainer.classList.add("hidden");
    }
  } catch (e) {
    console.warn("Failed to load proxy config:", e);
  }
}

function collapseProxy() {
  const proxyCollapsed = $("#proxyCollapsed");
  const proxyExpanded = $("#proxyExpanded");
  const editProxyBtn = $("#editProxyBtn");
  const cancelProxyBtn = $("#cancelProxyBtn");
  const proxyToggleContainer = $("#proxyToggleContainer");
  const proxyStatusConfigured = $("#proxyStatusConfigured");
  const proxyStatusNone = $("#proxyStatusNone");

  if (proxyCollapsed) proxyCollapsed.classList.remove("hidden");
  if (proxyExpanded) proxyExpanded.classList.add("hidden");
  if (editProxyBtn) editProxyBtn.classList.remove("hidden");
  if (cancelProxyBtn) cancelProxyBtn.classList.add("hidden");

  // Show appropriate status message and toggle based on whether proxy is configured
  if (hasProxy) {
    if (proxyStatusConfigured) proxyStatusConfigured.classList.remove("hidden");
    if (proxyStatusNone) proxyStatusNone.classList.add("hidden");
    if (proxyToggleContainer) proxyToggleContainer.classList.remove("hidden");
  } else {
    if (proxyStatusConfigured) proxyStatusConfigured.classList.add("hidden");
    if (proxyStatusNone) proxyStatusNone.classList.remove("hidden");
    if (proxyToggleContainer) proxyToggleContainer.classList.add("hidden");
  }
}

function expandProxy() {
  const proxyCollapsed = $("#proxyCollapsed");
  const proxyExpanded = $("#proxyExpanded");
  const editProxyBtn = $("#editProxyBtn");
  const cancelProxyBtn = $("#cancelProxyBtn");
  const proxyToggleContainer = $("#proxyToggleContainer");

  if (proxyCollapsed) proxyCollapsed.classList.add("hidden");
  if (proxyExpanded) proxyExpanded.classList.remove("hidden");
  if (editProxyBtn) editProxyBtn.classList.add("hidden");
  if (cancelProxyBtn) cancelProxyBtn.classList.remove("hidden");
  if (proxyToggleContainer) proxyToggleContainer.classList.add("hidden");
}

async function cancelEditProxy() {
  await loadProxy();
  collapseProxy();
}

async function toggleProxyEnabled(event) {
  const proxyEnabledToggle = event?.target || $("#proxyEnabledToggle");
  const proxyStatusState = $("#proxyStatusState");

  // Get the current checked state from the event target
  const enabled = proxyEnabledToggle?.checked ?? false;

  try {
    const r = await fetch("/proxy/toggle", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ enabled: enabled })
    });
    const j = await r.json();

    if (j.ok) {
      // Update status state display
      if (proxyStatusState) {
        proxyStatusState.textContent = enabled ? "(enabled)" : "(disabled)";
        proxyStatusState.className = enabled ? "proxy-state enabled" : "proxy-state disabled";
      }
    } else {
      // Revert toggle on error
      if (proxyEnabledToggle) proxyEnabledToggle.checked = !enabled;
      window.showStatusModal(j.message || "Failed to toggle proxy", "error");
    }
  } catch (e) {
    // Revert toggle on error
    if (proxyEnabledToggle) proxyEnabledToggle.checked = !enabled;
    window.showStatusModal("Failed to toggle proxy: " + e.message, "error");
  }
}

async function saveProxy() {
  const proxyMsg = $("#proxyMsg");
  const saveProxyBtn = $("#saveProxyBtn");
  const proxyUrlInput = $("#proxyUrl");
  const proxyUsernameInput = $("#proxyUsername");
  const proxyPasswordInput = $("#proxyPassword");

  const proxyUrl = proxyUrlInput?.value.trim() || '';
  const proxyUsername = proxyUsernameInput?.value.trim() || '';
  const proxyPassword = proxyPasswordInput?.value.trim() || '';
  const proxyEnabled = true; // New proxies are enabled by default; use toggle to disable

  // Require URL
  if (!proxyUrl) {
    showMessage(proxyMsg, "Proxy URL is required", "error");
    return;
  }

  // Validate URL format
  try {
    new URL(proxyUrl);
  } catch {
    showMessage(proxyMsg, "Invalid proxy URL format. Use http://host:port or https://host:port", "error");
    return;
  }

  if (saveProxyBtn) saveProxyBtn.disabled = true;

  try {
    const r = await fetch("/proxy", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({
        url: proxyUrl,
        username: proxyUsername,
        password: proxyPassword,
        enabled: proxyEnabled
      })
    });
    const j = await r.json();

    if (j.ok) {
      showMessage(proxyMsg, "Proxy configuration saved", "success");
      hasProxy = true;

      // Update status display
      const proxyStatusHost = $("#proxyStatusHost");
      const proxyStatusState = $("#proxyStatusState");
      const proxyEnabledToggle = $("#proxyEnabledToggle");

      if (proxyStatusHost) {
        try {
          const url = new URL(proxyUrl);
          proxyStatusHost.textContent = url.host;
        } catch {
          proxyStatusHost.textContent = proxyUrl;
        }
      }

      // Update enabled state in collapsed view
      if (proxyStatusState) {
        proxyStatusState.textContent = proxyEnabled ? "(enabled)" : "(disabled)";
        proxyStatusState.className = proxyEnabled ? "proxy-state enabled" : "proxy-state disabled";
      }
      if (proxyEnabledToggle) proxyEnabledToggle.checked = proxyEnabled;

      collapseProxy();
    } else {
      showMessage(proxyMsg, j.message || "Failed to save proxy", "error");
    }
  } catch (e) {
    showMessage(proxyMsg, "Failed to save proxy: " + e.message, "error");
  } finally {
    if (saveProxyBtn) saveProxyBtn.disabled = false;
  }
}

async function clearProxy() {
  const proxyMsg = $("#proxyMsg");
  const clearProxyBtn = $("#clearProxyBtn");

  if (clearProxyBtn) clearProxyBtn.disabled = true;

  try {
    const r = await fetch("/proxy", { method: "DELETE" });
    const j = await r.json();

    if (j.ok) {
      const proxyUrlInput = $("#proxyUrl");
      const proxyUsernameInput = $("#proxyUsername");
      const proxyPasswordInput = $("#proxyPassword");

      if (proxyUrlInput) proxyUrlInput.value = "";
      if (proxyUsernameInput) proxyUsernameInput.value = "";
      if (proxyPasswordInput) {
        proxyPasswordInput.value = "";
        proxyPasswordInput.placeholder = "Proxy password";
      }

      showMessage(proxyMsg, "Proxy configuration cleared", "success");
      hasProxy = false;
      collapseProxy();
    } else {
      showMessage(proxyMsg, j.message || "Failed to clear proxy", "error");
    }
  } catch (e) {
    showMessage(proxyMsg, "Failed to clear proxy: " + e.message, "error");
  } finally {
    if (clearProxyBtn) clearProxyBtn.disabled = false;
  }
}

async function loadSettings() {
  try {
    const r = await fetch("/config/settings");
    const j = await r.json();
    if (j.ok && j.settings) {
      const apiUrlInput = $("#apiUrl");
      const pageSizeInput = $("#pageSize");
      const fieldsInput = $("#fields");
      const baseStartDateInput = $("#baseStartDate");

      if (apiUrlInput) apiUrlInput.value = j.settings.url || "";
      if (pageSizeInput) pageSizeInput.value = j.settings.pageSize || "";
      if (fieldsInput) fieldsInput.value = j.settings.fields || "";
      if (baseStartDateInput) baseStartDateInput.value = j.settings.baseStartDate || "2025-08-01";

      // If settings exist, collapse the section
      if (j.settings.url) {
        hasSettings = true;
        collapseSettings();
      }
    }
  } catch (e) {
    console.warn("Failed to load settings:", e);
  }
}


async function saveSettings() {
  const settingsMsg = $("#settingsMsg");
  const saveSettingsBtn = $("#saveSettingsBtn");
  const apiUrlInput = $("#apiUrl");
  const pageSizeInput = $("#pageSize");
  const fieldsInput = $("#fields");
  const baseStartDateInput = $("#baseStartDate");

  const settings = {
    url: apiUrlInput?.value.trim(),
    pageSize: parseInt(pageSizeInput?.value, 10) || 1000,
    fields: fieldsInput?.value.trim(),
    baseStartDate: baseStartDateInput?.value.trim()
  };

  if (!settings.url) {
    showMessage(settingsMsg, "API URL is required", "error");
    return;
  }

  if (saveSettingsBtn) saveSettingsBtn.disabled = true;

  try {
    const r = await fetch("/config/settings", {
      method: "PUT",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(settings)
    });
    const j = await r.json();

    if (j.ok) {
      showMessage(settingsMsg, "Settings saved", "success");
      hasSettings = true;
      collapseSettings();
    } else {
      showMessage(settingsMsg, j.message || "Failed to save settings", "error");
    }
  } catch (e) {
    showMessage(settingsMsg, "Failed to save settings: " + e.message, "error");
  } finally {
    if (saveSettingsBtn) saveSettingsBtn.disabled = false;
  }
}

async function resetSettings() {
  const confirmed = await window.showConfirmModal("Reset API settings to defaults?", "Reset Settings");
  if (!confirmed) return;

  const settingsMsg = $("#settingsMsg");
  const resetSettingsBtn = $("#resetSettingsBtn");

  if (resetSettingsBtn) resetSettingsBtn.disabled = true;

  try {
    const r = await fetch("/config/settings/reset", { method: "POST" });
    const j = await r.json();

    if (j.ok) {
      await loadSettings();
      showMessage(settingsMsg, "Settings reset to defaults", "success");
    } else {
      showMessage(settingsMsg, j.message || "Failed to reset settings", "error");
    }
  } catch (e) {
    showMessage(settingsMsg, "Failed to reset settings: " + e.message, "error");
  } finally {
    if (resetSettingsBtn) resetSettingsBtn.disabled = false;
  }
}

async function loadFilters() {
  const filtersTableBody = $("#filtersTableBody");

  try {
    const r = await fetch("/queries");
    const j = await r.json();
    if (j.ok && j.queries && j.queries.length > 0) {
      renderFilters(j.queries);
      hasFilters = true;
      collapseFilters();
    } else {
      hasFilters = false;
      if (filtersTableBody) filtersTableBody.innerHTML = '<tr><td colspan="4" class="hint">No filters configured</td></tr>';
    }
  } catch (e) {
    console.warn("Failed to load filters:", e);
    if (filtersTableBody) filtersTableBody.innerHTML = '<tr><td colspan="4" class="err">Failed to load filters</td></tr>';
  }
}

function escapeHtml(text) {
  const div = document.createElement("div");
  div.textContent = text;
  return div.innerHTML;
}

function renderFilters(queries) {
  const filtersTableBody = $("#filtersTableBody");
  if (!filtersTableBody) return;

  filtersTableBody.innerHTML = "";

  if (!queries || queries.length === 0) {
    filtersTableBody.innerHTML = '<tr><td colspan="4" class="hint">No filters configured. Add one to get started or Click Reset to Defaults.</td></tr>';
    return;
  }

  const isLastFilter = queries.length === 1;

  for (const q of queries) {
    const tr = document.createElement("tr");

    tr.innerHTML = `
      <td class="filter-order-cell">${q.order}</td>
      <td class="filter-name-cell">${escapeHtml(q.name)}</td>
      <td class="filter-prefix-cell">${escapeHtml(q.prefix || "")}</td>
      <td class="filter-actions-cell">
        <button type="button" class="btn-secondary btn-small edit-filter-btn">Edit</button>
        <button type="button" class="btn-danger btn-small delete-filter-btn" ${isLastFilter ? 'disabled title="Cannot delete the last filter"' : ''}>Delete</button>
      </td>
    `;
    filtersTableBody.appendChild(tr);

    const editBtn = tr.querySelector(".edit-filter-btn");
    const deleteBtn = tr.querySelector(".delete-filter-btn");

    editBtn.onclick = () => openEditModal(q.order, q.name, q.prefix);

    deleteBtn.onclick = async () => {
      const confirmed = await window.showConfirmModal(`Delete filter "${q.name}"?`, "Delete Filter");
      if (!confirmed) return;

      deleteBtn.disabled = true;

      try {
        const r = await fetch(`/config/queries/${encodeURIComponent(q.order)}`, { method: "DELETE" });
        const j = await r.json();

        if (j.ok) {
          loadFilters();
        } else {
          window.showStatusModal(j.message || "Failed to delete", "error");
        }
      } catch (e) {
        window.showStatusModal("Failed: " + e.message, "error");
      } finally {
        deleteBtn.disabled = false;
      }
    };
  }
}

async function openAddModal() {
  editingFilterOrder = null;
  const filterModalTitle = $("#filterModalTitle");
  const filterModalOrderInput = $("#filterModalOrder");
  const filterModalNameInput = $("#filterModalName");
  const filterModalPrefixInput = $("#filterModalPrefix");
  const filterModalMsg = $("#filterModalMsg");
  const filterModal = $("#filterModal");

  if (filterModalTitle) filterModalTitle.textContent = "Add Filter";

  try {
    const r = await fetch("/queries");
    const j = await r.json();
    const maxOrder = j.queries && j.queries.length > 0
      ? Math.max(...j.queries.map(q => q.order))
      : 0;
    if (filterModalOrderInput) filterModalOrderInput.value = maxOrder + 1;
  } catch {
    if (filterModalOrderInput) filterModalOrderInput.value = 1;
  }

  if (filterModalNameInput) filterModalNameInput.value = "";
  if (filterModalPrefixInput) filterModalPrefixInput.value = "";
  if (filterModalMsg) {
    filterModalMsg.className = "message";
    filterModalMsg.textContent = "";
  }
  filterModal?.classList.remove("hidden");
  filterModalNameInput?.focus();
}

function openEditModal(order, name, prefix) {
  editingFilterOrder = order;
  const filterModalTitle = $("#filterModalTitle");
  const filterModalOrderInput = $("#filterModalOrder");
  const filterModalNameInput = $("#filterModalName");
  const filterModalPrefixInput = $("#filterModalPrefix");
  const filterModalMsg = $("#filterModalMsg");
  const filterModal = $("#filterModal");

  if (filterModalTitle) filterModalTitle.textContent = "Edit Filter";
  if (filterModalOrderInput) filterModalOrderInput.value = order;
  if (filterModalNameInput) filterModalNameInput.value = name;
  if (filterModalPrefixInput) filterModalPrefixInput.value = prefix || "";
  if (filterModalMsg) {
    filterModalMsg.className = "message";
    filterModalMsg.textContent = "";
  }
  filterModal?.classList.remove("hidden");
  filterModalNameInput?.focus();
}

function closeFilterModal() {
  const filterModal = $("#filterModal");
  const filterModalOrderInput = $("#filterModalOrder");
  const filterModalNameInput = $("#filterModalName");
  const filterModalPrefixInput = $("#filterModalPrefix");
  const filterModalMsg = $("#filterModalMsg");

  filterModal?.classList.add("hidden");
  editingFilterOrder = null;
  if (filterModalOrderInput) filterModalOrderInput.value = "";
  if (filterModalNameInput) filterModalNameInput.value = "";
  if (filterModalPrefixInput) filterModalPrefixInput.value = "";
  if (filterModalMsg) {
    filterModalMsg.className = "message";
    filterModalMsg.textContent = "";
  }
}

async function saveFilter() {
  const filterModalMsg = $("#filterModalMsg");
  const saveFilterBtn = $("#saveFilterBtn");
  const filterModalOrderInput = $("#filterModalOrder");
  const filterModalNameInput = $("#filterModalName");
  const filterModalPrefixInput = $("#filterModalPrefix");

  const newOrder = parseInt(filterModalOrderInput?.value, 10);
  const newName = filterModalNameInput?.value.trim();
  const newPrefix = filterModalPrefixInput?.value.trim();

  if (!newOrder || newOrder < 1) {
    showMessage(filterModalMsg, "Order must be a positive number", "error");
    return;
  }

  if (!newName || !newPrefix) {
    showMessage(filterModalMsg, "Name and prefix are required", "error");
    return;
  }

  if (saveFilterBtn) saveFilterBtn.disabled = true;

  try {
    if (editingFilterOrder !== null && newOrder !== editingFilterOrder) {
      await fetch(`/config/queries/${encodeURIComponent(editingFilterOrder)}`, { method: "DELETE" });
    }

    const r = await fetch("/config/queries", {
      method: "PUT",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ order: newOrder, name: newName, prefix: newPrefix })
    });
    const j = await r.json();

    if (j.ok) {
      closeFilterModal();
      loadFilters();
    } else {
      showMessage(filterModalMsg, j.message || "Failed to save", "error");
    }
  } catch (e) {
    showMessage(filterModalMsg, "Failed: " + e.message, "error");
  } finally {
    if (saveFilterBtn) saveFilterBtn.disabled = false;
  }
}

async function resetFilters() {
  const confirmed = await window.showConfirmModal("Reset all filters to defaults? This will remove all custom filters.", "Reset Filters");
  if (!confirmed) return;

  const resetFiltersBtn = $("#resetFiltersBtn");

  if (resetFiltersBtn) resetFiltersBtn.disabled = true;

  try {
    const r = await fetch("/config/queries/reset", { method: "POST" });
    const j = await r.json();

    if (j.ok) {
      loadFilters();
    } else {
      alert(j.message || "Failed to reset filters");
    }
  } catch (e) {
    alert("Failed to reset filters: " + e.message);
  } finally {
    if (resetFiltersBtn) resetFiltersBtn.disabled = false;
  }
}

// ── Companies (API-only) ─────────────────────────────────────────

async function checkExistingCompanies() {
  const currentFileDisplay = $("#currentFileDisplay");
  const currentFileName = $("#currentFileName");
  const currentFileSize = $("#currentFileSize");

  try {
    const r = await fetch("/upload/status");
    const j = await r.json();
    if (j.ok && j.exists) {
      hasCompaniesFile = true;
      if (currentFileName) currentFileName.textContent = `${j.count} companies/instances loaded`;
      if (currentFileSize) {
        const uploaded = j.uploadedAt ? new Date(j.uploadedAt).toLocaleString() : '';
        currentFileSize.textContent = uploaded ? `(loaded ${uploaded})` : '';
      }
      currentFileDisplay?.classList.remove("hidden", "no-file");
      currentFileDisplay?.classList.add("has-file");
      setApiHint();
    } else {
      hasCompaniesFile = false;
      if (currentFileName) currentFileName.textContent = "No companies loaded";
      if (currentFileSize) currentFileSize.textContent = "";
      currentFileDisplay?.classList.remove("has-file", "no-file");
      currentFileDisplay?.classList.add("hidden");
      setApiHint();
    }
  } catch (e) {
    console.warn("Failed to check existing companies:", e);
  }
}

function setApiHint() {
  const uploadMsg = $("#uploadMsg");
  if (!uploadMsg) return;
  uploadMsg.innerHTML = `Enter API URL returning <code>{"companies": [{"sage_account_number": "XXX", "company_id": "Name"}, ...]}</code>. Datel Group is always included.`;
  uploadMsg.className = "hint";
}

async function loadCompaniesApiUrl() {
  try {
    const r = await fetch("/config/settings");
    const j = await r.json();
    if (j.ok && j.settings) {
      const companiesApiUrl = j.settings.companiesApiUrl || '';
      const companiesApiToken = j.settings.companiesApiToken || '';

      const companiesApiUrlInput = $("#companiesApiUrl");
      const companiesApiTokenInput = $("#companiesApiToken");

      if (companiesApiUrlInput) companiesApiUrlInput.value = companiesApiUrl;
      if (companiesApiTokenInput) companiesApiTokenInput.value = companiesApiToken;
    }
  } catch (e) {
    console.warn("Failed to load companies API URL:", e);
  }
}

async function saveCompaniesApiUrl() {
  const companiesApiUrlInput = $("#companiesApiUrl");
  const companiesApiTokenInput = $("#companiesApiToken");
  try {
    await fetch("/config/settings/companies-source", {
      method: "PUT",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({
        companiesSource: 'api',
        companiesApiUrl: companiesApiUrlInput?.value.trim() || '',
        companiesApiToken: companiesApiTokenInput?.value.trim() || ''
      })
    });
  } catch (e) {
    console.warn("Failed to save companies API URL:", e);
  }
}

async function loadCompaniesFromApi() {
  const loadBtn = $("#loadFromApiBtn");
  const uploadMsg = $("#uploadMsg");
  const companiesApiUrlInput = $("#companiesApiUrl");
  const companiesApiTokenInput = $("#companiesApiToken");

  const apiUrl = companiesApiUrlInput?.value.trim();
  const apiToken = companiesApiTokenInput?.value.trim() || '';

  if (!apiUrl) {
    if (uploadMsg) {
      uploadMsg.textContent = "Enter an API URL first.";
      uploadMsg.className = "err";
    }
    return;
  }

  if (loadBtn) loadBtn.disabled = true;
  if (uploadMsg) {
    uploadMsg.textContent = "Loading companies from API...";
    uploadMsg.className = "hint";
  }

  // Save the URL and token
  await saveCompaniesApiUrl();

  try {
    const r = await fetch("/upload/from-api", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ apiUrl, apiToken })
    });
    const j = await r.json();

    if (j.ok) {
      hasCompaniesFile = true;
      const currentFileName = $("#currentFileName");
      const currentFileSize = $("#currentFileSize");
      const currentFileDisplay = $("#currentFileDisplay");

      if (currentFileName) currentFileName.textContent = `${j.count} companies/instances loaded`;
      if (currentFileSize) currentFileSize.textContent = `(loaded from API ${new Date().toLocaleString()})`;
      if (currentFileDisplay) {
        currentFileDisplay.classList.remove("hidden", "no-file");
        currentFileDisplay.classList.add("has-file");
      }
      setApiHint();
      window.showStatusModal(`Loaded ${j.count} companies from API`, "success");
    } else {
      if (uploadMsg) {
        uploadMsg.textContent = j.message || "Failed to load from API.";
        uploadMsg.className = "err";
      }
    }
  } catch (e) {
    if (uploadMsg) {
      uploadMsg.textContent = "Failed to load: " + e.message;
      uploadMsg.className = "err";
    }
  } finally {
    if (loadBtn) loadBtn.disabled = false;
  }
}

// ── Companies Modal ──────────────────────────────────────────────

// Store companies data for filtering
let cachedCompanies = [];
let cachedSkipSet = new Set();

async function openCompaniesModal() {
  const modal = $("#companiesModal");
  if (!modal) return;
  modal.classList.remove("hidden");
  // Clear search when opening
  const searchInput = $("#companySearchInput");
  if (searchInput) searchInput.value = "";
  await refreshCompanyList();
  const input = $("#addCompanyInput");
  if (input) input.focus();
}

function closeCompaniesModal() {
  const modal = $("#companiesModal");
  if (modal) modal.classList.add("hidden");
  const msg = $("#companyModalMsg");
  if (msg) msg.textContent = "";
}

async function refreshCompanyList() {
  const container = $("#companyListContainer");
  if (!container) return;
  container.innerHTML = '<p class="loading">Loading...</p>';

  try {
    const [companiesRes, skipRes] = await Promise.all([
      fetch("/upload/companies"),
      fetch("/upload/skip-companies")
    ]);
    const companiesData = await companiesRes.json();
    const skipData = await skipRes.json();
    cachedSkipSet = new Set(skipData.ok ? (skipData.companies || []) : []);
    if (companiesData.ok) {
      cachedCompanies = companiesData.companies || [];
      renderCompanyList();
    } else {
      container.innerHTML = '<p class="err">Failed to load companies.</p>';
    }
  } catch (e) {
    container.innerHTML = '<p class="err">Failed to load companies.</p>';
  }
}

function filterCompanies() {
  renderCompanyList();
}

function renderCompanyList() {
  const container = $("#companyListContainer");
  const searchInput = $("#companySearchInput");
  const countSpan = $("#companySearchCount");
  if (!container) return;

  const searchTerm = (searchInput?.value || "").toLowerCase().trim();
  const filtered = searchTerm
    ? cachedCompanies.filter(id => id.toLowerCase().includes(searchTerm))
    : cachedCompanies;

  // Update count display
  if (countSpan) {
    if (searchTerm && cachedCompanies.length > 0) {
      countSpan.textContent = `${filtered.length} of ${cachedCompanies.length}`;
    } else if (cachedCompanies.length > 0) {
      countSpan.textContent = `${cachedCompanies.length} total`;
    } else {
      countSpan.textContent = "";
    }
  }

  if (cachedCompanies.length === 0) {
    container.innerHTML = '<p class="empty-state">No companies/instances loaded.</p>';
    return;
  }

  if (filtered.length === 0) {
    container.innerHTML = '<p class="empty-state">No matches found.</p>';
    return;
  }

  container.innerHTML = "";
  for (const id of filtered) {
    const item = document.createElement("div");
    item.className = "company-item";
    const nameSpan = document.createElement("span");
    nameSpan.className = "company-name";
    nameSpan.textContent = id;
    const actionsDiv = document.createElement("div");
    actionsDiv.className = "company-item-actions";
    const isSkipped = cachedSkipSet && cachedSkipSet.has(id);
    const skipBtn = document.createElement("button");
    skipBtn.className = isSkipped ? "btn-unskip" : "btn-skip";
    skipBtn.textContent = isSkipped ? "Unskip" : "Skip";
    skipBtn.title = isSkipped ? `Remove ${id} from exceptions` : `Add ${id} to exceptions`;
    skipBtn.onclick = isSkipped ? () => removeFromExceptionsInModal(id) : () => addToExceptions(id);
    const removeBtn = document.createElement("button");
    removeBtn.className = "btn-danger btn-small btn-remove";
    removeBtn.textContent = "\u00D7";
    removeBtn.title = `Remove ${id}`;
    removeBtn.onclick = () => removeCompanyFromModal(id);
    actionsDiv.appendChild(skipBtn);
    actionsDiv.appendChild(removeBtn);
    item.appendChild(nameSpan);
    item.appendChild(actionsDiv);
    container.appendChild(item);
  }
}

async function addCompanyFromModal() {
  const input = $("#addCompanyInput");
  const msg = $("#companyModalMsg");
  const id = (input?.value || '').trim();
  if (!id) {
    if (msg) { msg.textContent = "Enter a company ID."; msg.className = "err"; }
    return;
  }

  try {
    const r = await fetch("/upload/companies", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ company_id: id })
    });
    const j = await r.json();
    if (j.ok) {
      if (input) input.value = "";
      if (msg) { msg.textContent = j.message; msg.className = "ok"; }
      await refreshCompanyList();
      updateBannerCount(j.count);
    } else {
      if (msg) { msg.textContent = j.message || "Failed to add."; msg.className = "err"; }
    }
  } catch (e) {
    if (msg) { msg.textContent = e.message; msg.className = "err"; }
  }
}

async function removeCompanyFromModal(companyId) {
  const msg = $("#companyModalMsg");
  try {
    const r = await fetch(`/upload/companies/${encodeURIComponent(companyId)}`, { method: "DELETE" });
    const j = await r.json();
    if (j.ok) {
      if (msg) { msg.textContent = j.message; msg.className = "ok"; }
      await refreshCompanyList();
      updateBannerCount(j.count);
    } else {
      if (msg) { msg.textContent = j.message || "Failed to remove."; msg.className = "err"; }
    }
  } catch (e) {
    if (msg) { msg.textContent = e.message; msg.className = "err"; }
  }
}

function updateBannerCount(count) {
  const currentFileName = $("#currentFileName");
  const currentFileDisplay = $("#currentFileDisplay");
  if (count > 0) {
    hasCompaniesFile = true;
    if (currentFileName) currentFileName.textContent = `${count} companies/instances loaded`;
    currentFileDisplay?.classList.remove("hidden", "no-file");
    currentFileDisplay?.classList.add("has-file");
  } else {
    hasCompaniesFile = false;
    if (currentFileName) currentFileName.textContent = "No companies uploaded";
    currentFileDisplay?.classList.remove("has-file", "no-file");
    currentFileDisplay?.classList.add("hidden");
  }
  setUploadHint();
}

// ── Clear All Modal ──────────────────────────────────────────────

function openClearAllModal() {
  const modal = $("#clearAllModal");
  if (modal) modal.classList.remove("hidden");
}

function closeClearAllModal() {
  const modal = $("#clearAllModal");
  if (modal) modal.classList.add("hidden");
}

async function handleClearAll() {
  const confirmBtn = $("#confirmClearAllBtn");
  if (confirmBtn) confirmBtn.disabled = true;

  try {
    const r = await fetch("/upload/clear-all", { method: "DELETE" });
    const j = await r.json();
    if (j.ok) {
      closeClearAllModal();
      updateBannerCount(0);
    }
  } catch (e) {
    console.warn("Failed to clear all:", e);
  } finally {
    if (confirmBtn) confirmBtn.disabled = false;
  }
}

// ── Exceptions (Skip Companies) ─────────────────────────────────

async function loadExceptionsCount() {
  try {
    const r = await fetch("/upload/skip-companies");
    const j = await r.json();
    if (j.ok) {
      updateExceptionsBadge(j.count || 0);
    }
  } catch {
    // Silently fail
  }
}

function updateExceptionsBadge(count) {
  const badge = $("#exceptionsCount");
  if (!badge) return;
  if (count > 0) {
    badge.textContent = count;
    badge.classList.remove("hidden");
  } else {
    badge.classList.add("hidden");
  }
}

async function addToExceptions(companyId) {
  const msg = $("#companyModalMsg");
  try {
    const r = await fetch("/upload/skip-companies", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ company_id: companyId })
    });
    const j = await r.json();
    if (j.ok) {
      if (msg) { msg.textContent = j.message; msg.className = "ok"; }
      updateExceptionsBadge(j.count || 0);
      await refreshCompanyList();
    } else {
      if (msg) { msg.textContent = j.message || "Failed to add to exceptions."; msg.className = "err"; }
    }
  } catch (e) {
    if (msg) { msg.textContent = e.message; msg.className = "err"; }
  }
}

async function removeFromExceptionsInModal(companyId) {
  const msg = $("#companyModalMsg");
  try {
    const r = await fetch(`/upload/skip-companies/${encodeURIComponent(companyId)}`, { method: "DELETE" });
    const j = await r.json();
    if (j.ok) {
      if (msg) { msg.textContent = j.message; msg.className = "ok"; }
      updateExceptionsBadge(j.count || 0);
      await refreshCompanyList();
    } else {
      if (msg) { msg.textContent = j.message || "Failed to remove from exceptions."; msg.className = "err"; }
    }
  } catch (e) {
    if (msg) { msg.textContent = e.message; msg.className = "err"; }
  }
}

async function openExceptionsModal() {
  const modal = $("#exceptionsModal");
  if (!modal) return;
  modal.classList.remove("hidden");
  await refreshExceptionsList();
}

function closeExceptionsModal() {
  const modal = $("#exceptionsModal");
  if (modal) modal.classList.add("hidden");
  const msg = $("#exceptionsModalMsg");
  if (msg) msg.textContent = "";
}

async function refreshExceptionsList() {
  const container = $("#exceptionsListContainer");
  if (!container) return;
  container.innerHTML = '<p class="loading">Loading...</p>';

  try {
    const r = await fetch("/upload/skip-companies");
    const j = await r.json();
    if (j.ok) {
      renderExceptionsList(j.companies || []);
      updateExceptionsBadge(j.count || 0);
    } else {
      container.innerHTML = '<p class="err">Failed to load exceptions.</p>';
    }
  } catch (e) {
    container.innerHTML = '<p class="err">Failed to load exceptions.</p>';
  }
}

function renderExceptionsList(companies) {
  const container = $("#exceptionsListContainer");
  if (!container) return;

  if (companies.length === 0) {
    container.innerHTML = '<p class="empty-state">No exceptions configured.</p>';
    return;
  }

  container.innerHTML = "";
  for (const id of companies) {
    const item = document.createElement("div");
    item.className = "company-item";
    const nameSpan = document.createElement("span");
    nameSpan.className = "company-name";
    nameSpan.textContent = id;
    const removeBtn = document.createElement("button");
    removeBtn.className = "btn-danger btn-small btn-remove";
    removeBtn.textContent = "\u00D7";
    removeBtn.title = `Remove ${id} from exceptions`;
    removeBtn.onclick = () => removeException(id);
    item.appendChild(nameSpan);
    item.appendChild(removeBtn);
    container.appendChild(item);
  }
}

async function removeException(companyId) {
  const msg = $("#exceptionsModalMsg");
  try {
    const r = await fetch(`/upload/skip-companies/${encodeURIComponent(companyId)}`, { method: "DELETE" });
    const j = await r.json();
    if (j.ok) {
      if (msg) { msg.textContent = j.message; msg.className = "ok"; }
      await refreshExceptionsList();
    } else {
      if (msg) { msg.textContent = j.message || "Failed to remove."; msg.className = "err"; }
    }
  } catch (e) {
    if (msg) { msg.textContent = e.message; msg.className = "err"; }
  }
}
