// logs.js - Logs tab module

const $ = (s) => document.querySelector(s);
const $$ = (s) => document.querySelectorAll(s);

// State
let currentView = 'parsed';
let reverseOrder = true;
let autoRefreshEnabled = true;
let currentFilter = 'all';
let autoRefreshInterval = null;
let currentLogContent = '';

/**
 * Initialize the logs tab
 */
export async function initLogs() {
  loadLogFiles();
  setupAutoRefresh();
  setupEventListeners();
}

/**
 * Cleanup when leaving the logs tab
 */
export function cleanupLogs() {
  if (autoRefreshInterval) {
    clearInterval(autoRefreshInterval);
    autoRefreshInterval = null;
  }
}

function setupEventListeners() {
  const logFileSelect = $("#logFileSelect");
  const btnParsed = $("#btnParsed");
  const btnRaw = $("#btnRaw");
  const btnSortOrder = $("#btnSortOrder");
  const btnRefresh = $("#btnRefresh");
  const chkAutoRefresh = $("#chkAutoRefresh");

  if (logFileSelect) {
    logFileSelect.onchange = (e) => {
      loadLogContent(e.target.value);
    };
  }

  if (btnParsed) {
    btnParsed.onclick = () => {
      currentView = 'parsed';
      btnParsed.classList.add('active');
      btnRaw?.classList.remove('active');
      renderLogContent();
    };
  }

  if (btnRaw) {
    btnRaw.onclick = () => {
      currentView = 'raw';
      btnRaw.classList.add('active');
      btnParsed?.classList.remove('active');
      renderLogContent();
    };
  }

  if (btnSortOrder) {
    btnSortOrder.onclick = () => {
      reverseOrder = !reverseOrder;
      btnSortOrder.textContent = reverseOrder ? '↑ Newest First' : '↓ Oldest First';
      renderLogContent();
    };
  }

  if (btnRefresh) {
    btnRefresh.onclick = () => {
      const logFileSelect = $("#logFileSelect");
      if (logFileSelect?.value) {
        loadLogContent(logFileSelect.value);
      }
    };
  }

  if (chkAutoRefresh) {
    chkAutoRefresh.onchange = (e) => {
      autoRefreshEnabled = e.target.checked;
      setupAutoRefresh();
    };
  }

  // Filter buttons
  $$('.filter-btn').forEach(btn => {
    btn.onclick = () => {
      $$('.filter-btn').forEach(b => b.classList.remove('active'));
      btn.classList.add('active');
      currentFilter = btn.dataset.filter;
      renderLogContent();
    };
  });
}

async function loadLogFiles() {
  try {
    const response = await fetch('/api/logs');
    const data = await response.json();

    const select = $('#logFileSelect');
    if (!select) return;

    select.innerHTML = '<option value="">Select a log file...</option>';

    if (data.success && data.logs.length > 0) {
      data.logs.forEach(log => {
        const option = document.createElement('option');
        option.value = log.filename;
        const sizeKB = (log.size / 1024).toFixed(1);
        option.textContent = `${log.filename} (${sizeKB} KB)`;
        select.appendChild(option);
      });

      // Auto-select the first (most recent) log file
      if (data.logs.length > 0) {
        select.value = data.logs[0].filename;
        loadLogContent(data.logs[0].filename);
      }
    }
  } catch (e) {
    console.error('Failed to load log files:', e);
  }
}

async function loadLogContent(filename) {
  const logViewer = $('#logViewer');
  const logInfo = $('#logInfo');

  if (!filename) {
    if (logViewer) logViewer.innerHTML = '<div class="empty-state">Select a log file to view its contents</div>';
    if (logInfo) logInfo.textContent = '';
    return;
  }

  try {
    const response = await fetch(`/api/logs/${encodeURIComponent(filename)}`);
    const data = await response.json();

    if (data.success) {
      currentLogContent = data.content;
      renderLogContent();

      // Update info
      const lines = data.content.trim().split('\n').filter(l => l.trim());
      if (logInfo) logInfo.textContent = `${lines.length} log entries`;
    } else {
      if (logViewer) logViewer.innerHTML = `<div class="empty-state">Error: ${data.message}</div>`;
    }
  } catch (e) {
    if (logViewer) logViewer.innerHTML = `<div class="empty-state">Failed to load log: ${e.message}</div>`;
  }
}

function renderLogContent() {
  const viewer = $('#logViewer');
  if (!viewer) return;

  if (!currentLogContent.trim()) {
    viewer.innerHTML = '<div class="empty-state">Log file is empty</div>';
    return;
  }

  if (currentView === 'raw') {
    viewer.innerHTML = escapeHtml(currentLogContent);
    return;
  }

  // Parse and render
  let lines = currentLogContent.trim().split('\n').filter(l => l.trim());

  // Apply filter
  if (currentFilter !== 'all') {
    lines = lines.filter(line => {
      try {
        const entry = JSON.parse(line);
        return entry.level === currentFilter;
      } catch {
        return currentFilter === 'info';
      }
    });
  }

  // Apply sort order
  if (reverseOrder) {
    lines = lines.reverse();
  }

  if (lines.length === 0) {
    viewer.innerHTML = '<div class="empty-state">No log entries match the current filter</div>';
    return;
  }

  let html = '';
  lines.forEach(line => {
    try {
      const entry = JSON.parse(line);
      const timestamp = entry.timestamp ? new Date(entry.timestamp).toLocaleString() : '';
      const level = entry.level || 'info';
      const message = entry.message || '';

      html += `<div class="log-entry">
        <span class="log-timestamp">${escapeHtml(timestamp)}</span>
        <span class="log-level ${level}">${level}</span>
        <span class="log-message">${escapeHtml(message)}</span>
      </div>`;
    } catch {
      // Plain text line
      html += `<div class="log-entry">
        <span class="log-message">${escapeHtml(line)}</span>
      </div>`;
    }
  });

  viewer.innerHTML = html;
}

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

function setupAutoRefresh() {
  if (autoRefreshInterval) {
    clearInterval(autoRefreshInterval);
    autoRefreshInterval = null;
  }

  if (autoRefreshEnabled) {
    autoRefreshInterval = setInterval(() => {
      const logFileSelect = $('#logFileSelect');
      if (logFileSelect?.value) {
        loadLogContent(logFileSelect.value);
      }
    }, 10000); // 10 seconds
  }
}
