// main.js - Main entry point that wires up all tab modules

import {
  registerTabInit,
  registerTabCleanup,
  switchTab,
  toggleSettingsDropdown,
  toggleResourcesDropdown,
  toggleMobileMore,
  toggleMobileResources,
  loadInitialTab
} from './tabs.js';

import { initReports, cleanupReports } from './reports.js';
import { initConfig, cleanupConfig } from './config.js';
import { initLogs, cleanupLogs } from './logs.js';
import { initAbout, cleanupAbout } from './about.js';
import { initNotifications, cleanupNotifications } from './notifications.js';
import { initSchedule, cleanupSchedule } from './schedule.js';
import { initGuide, cleanupGuide } from './guide.js';

// Register tab initializers
registerTabInit('schedule', initSchedule);
registerTabInit('reports', initReports);
registerTabInit('config', initConfig);
registerTabInit('notifications', initNotifications);
registerTabInit('logs', initLogs);
registerTabInit('about', initAbout);
registerTabInit('guide', initGuide);

// Register tab cleanups
registerTabCleanup('schedule', cleanupSchedule);
registerTabCleanup('reports', cleanupReports);
registerTabCleanup('config', cleanupConfig);
registerTabCleanup('notifications', cleanupNotifications);
registerTabCleanup('logs', cleanupLogs);
registerTabCleanup('about', cleanupAbout);
registerTabCleanup('guide', cleanupGuide);

// Expose functions to window for onclick handlers in HTML
window.switchTab = switchTab;
window.toggleSettingsDropdown = toggleSettingsDropdown;
window.toggleResourcesDropdown = toggleResourcesDropdown;
window.toggleMobileMore = toggleMobileMore;
window.toggleMobileResources = toggleMobileResources;

// ========== Auth / Logout ==========

async function handleLogout() {
  await fetch('/api/auth/logout', { method: 'POST' });
  window.location.href = '/login.html';
}

async function checkAuthStatus() {
  try {
    const res = await fetch('/api/auth/status');
    const data = await res.json();
    if (data.authEnabled) {
      const btn = document.getElementById('logoutBtn');
      if (btn) btn.style.display = '';
    }
  } catch { /* auth status unavailable, hide button */ }
}

window.handleLogout = handleLogout;
checkAuthStatus();

// ========== Global Status Modal ==========

/**
 * Show a status modal with a message
 * @param {string} message - The message to display
 * @param {string} type - 'success', 'error', 'warning', or 'info'
 * @param {string} title - Optional custom title (defaults based on type)
 */
function showStatusModal(message, type = 'success', title = null) {
  const modal = document.getElementById('statusModal');
  const modalContent = modal?.querySelector('.status-modal-content');
  const titleEl = document.getElementById('statusModalTitle');
  const messageEl = document.getElementById('statusModalMessage');

  if (!modal || !modalContent || !titleEl || !messageEl) return;

  // Set default titles based on type
  const defaultTitles = {
    success: 'Success',
    error: 'Error',
    warning: 'Warning',
    info: 'Information'
  };

  // Remove previous status classes
  modalContent.classList.remove('status-success', 'status-error', 'status-warning', 'status-info');
  modalContent.classList.add('status-' + type);

  titleEl.textContent = title || defaultTitles[type] || 'Status';
  messageEl.textContent = message;

  modal.classList.remove('hidden');
  modal.classList.add('show');
}

function closeStatusModal() {
  const modal = document.getElementById('statusModal');
  if (modal) {
    modal.classList.add('hidden');
    modal.classList.remove('show');
  }
}

// Expose status modal functions globally
window.showStatusModal = showStatusModal;
window.closeStatusModal = closeStatusModal;

// ========== Global Confirm Modal ==========

/**
 * Show a confirmation modal and return a promise
 * @param {string} message - The message to display
 * @param {string} title - Optional custom title (defaults to "Confirm")
 * @param {object} options - Optional settings { destructive: boolean }
 * @returns {Promise<boolean>} - Resolves to true if OK clicked, false if Cancel
 */
function showConfirmModal(message, title = 'Confirm', options = {}) {
  return new Promise((resolve) => {
    const modal = document.getElementById('confirmModal');
    const titleEl = document.getElementById('confirmModalTitle');
    const messageEl = document.getElementById('confirmModalMessage');
    const okBtn = document.getElementById('confirmModalOk');
    const cancelBtn = document.getElementById('confirmModalCancel');

    if (!modal || !titleEl || !messageEl || !okBtn || !cancelBtn) {
      // Fallback to browser confirm if modal not available
      resolve(confirm(message));
      return;
    }

    titleEl.textContent = title;
    messageEl.textContent = message;

    // Clean up previous handlers
    const newOkBtn = okBtn.cloneNode(true);
    const newCancelBtn = cancelBtn.cloneNode(true);
    okBtn.parentNode.replaceChild(newOkBtn, okBtn);
    cancelBtn.parentNode.replaceChild(newCancelBtn, cancelBtn);

    // Apply destructive styling if specified (warning color for OK button)
    if (options.destructive) {
      newOkBtn.classList.remove('btn-primary');
      newOkBtn.classList.add('btn-warning');
    } else {
      newOkBtn.classList.remove('btn-warning');
      newOkBtn.classList.add('btn-primary');
    }

    // Set up new handlers
    newOkBtn.onclick = () => {
      modal.classList.add('hidden');
      modal.classList.remove('show');
      resolve(true);
    };

    newCancelBtn.onclick = () => {
      modal.classList.add('hidden');
      modal.classList.remove('show');
      resolve(false);
    };

    modal.classList.remove('hidden');
    modal.classList.add('show');
  });
}

// Expose confirm modal function globally
window.showConfirmModal = showConfirmModal;

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', async () => {
  // Load the initial tab (checks URL hash)
  await loadInitialTab('schedule');
});

// Handle browser back/forward navigation
window.addEventListener('hashchange', () => {
  let hash = window.location.hash.substring(1);
  // Migrate old hashes
  if (hash === 'usage' || hash === 'manual') {
    hash = 'schedule';
    window.location.hash = 'schedule';
    return;
  }
  const validTabs = ['schedule', 'reports', 'config', 'notifications', 'logs', 'guide', 'about'];
  if (hash && validTabs.includes(hash)) {
    switchTab(hash);
  }
});
