// notifications.js - Notifications configuration tab module

const $ = (s) => document.querySelector(s);

let editingConfigId = null;
let originalConfigName = null;
let messageTimeout = null;
let initialFormState = null;

// ---- Form state tracking ----

function captureFormState() {
  return {
    name: $('#notif-name')?.value || '',
    provider: $('#notif-provider')?.value || '',
    // SMTP
    smtpHost: $('#smtp-host')?.value || '',
    smtpPort: $('#smtp-port')?.value || '',
    smtpSecurity: $('#smtp-security')?.value || '',
    smtpIgnoreTls: $('#smtp-ignore-tls')?.checked || false,
    smtpUsername: $('#smtp-username')?.value || '',
    smtpPassword: $('#smtp-password')?.value || '',
    smtpFromName: $('#smtp-from-name')?.value || '',
    smtpFrom: $('#smtp-from')?.value || '',
    // MS Graph
    msgraphTenant: $('#msgraph-tenant')?.value || '',
    msgraphClientId: $('#msgraph-client-id')?.value || '',
    msgraphClientSecret: $('#msgraph-client-secret')?.value || '',
    msgraphFrom: $('#msgraph-from')?.value || '',
    // Recipients
    recipients: $('#notif-recipients')?.value || '',
    cc: $('#notif-cc')?.value || '',
    bcc: $('#notif-bcc')?.value || ''
  };
}

function isFormDirty() {
  if (!initialFormState) return false;
  const currentState = captureFormState();
  return JSON.stringify(currentState) !== JSON.stringify(initialFormState);
}

// ---- View switching ----

function showListView() {
  const listView = $('#notif-list-view');
  const formView = $('#notif-form-view');
  if (listView) listView.classList.remove('hidden');
  if (formView) formView.classList.add('hidden');
}

function showFormView(mode = 'add') {
  const listView = $('#notif-list-view');
  const formView = $('#notif-form-view');
  const formTitle = $('#notif-form-title');

  if (listView) listView.classList.add('hidden');
  if (formView) formView.classList.remove('hidden');

  if (formTitle) {
    formTitle.textContent = mode === 'edit' ? 'Edit Configuration' : 'Add Configuration';
  }
}

function showMessage(text, type = 'success') {
  // Skip empty messages
  if (!text) return;

  // Use global status modal
  if (window.showStatusModal) {
    window.showStatusModal(text, type);
  }
}

// ---- Provider toggle ----

function toggleNotificationProvider() {
  const provider = $('#notif-provider').value;

  const sections = {
    smtp: $('#smtp-config'),
    msgraph: $('#msgraph-config')
  };

  for (const [key, el] of Object.entries(sections)) {
    if (el) el.classList.toggle('hidden', key !== provider);
  }

  const recipientsSection = $('#notif-recipients-section');
  if (recipientsSection) {
    recipientsSection.classList.toggle('hidden', !provider);
  }

  const actionsSection = $('#notif-actions');
  if (actionsSection) {
    actionsSection.classList.toggle('hidden', !provider);
  }
}

function updateSmtpPort() {
  const security = $('#smtp-security').value;
  const portField = $('#smtp-port');

  const defaultPorts = { starttls: 587, ssl: 465, none: 25 };
  const currentPort = parseInt(portField.value);
  const defaults = Object.values(defaultPorts);

  if (!currentPort || defaults.includes(currentPort)) {
    portField.value = defaultPorts[security] || 25;
  }
}

// ---- Form clear / cancel ----

function clearNotificationForm() {
  editingConfigId = null;
  originalConfigName = null;
  initialFormState = null;

  if ($('#notif-name')) $('#notif-name').value = '';
  if ($('#notif-provider')) $('#notif-provider').value = '';

  // SMTP
  if ($('#smtp-host')) $('#smtp-host').value = '';
  if ($('#smtp-port')) $('#smtp-port').value = '25';
  if ($('#smtp-security')) $('#smtp-security').value = 'none';
  if ($('#smtp-ignore-tls')) $('#smtp-ignore-tls').checked = false;
  if ($('#smtp-username')) $('#smtp-username').value = '';
  if ($('#smtp-password')) $('#smtp-password').value = '';
  if ($('#smtp-from-name')) $('#smtp-from-name').value = '';
  if ($('#smtp-from')) $('#smtp-from').value = '';

  // Microsoft Graph
  if ($('#msgraph-tenant')) $('#msgraph-tenant').value = '';
  if ($('#msgraph-client-id')) $('#msgraph-client-id').value = '';
  if ($('#msgraph-client-secret')) $('#msgraph-client-secret').value = '';
  if ($('#msgraph-from')) $('#msgraph-from').value = '';

  // Recipients
  if ($('#notif-recipients')) $('#notif-recipients').value = '';
  if ($('#notif-cc')) $('#notif-cc').value = '';
  if ($('#notif-bcc')) $('#notif-bcc').value = '';

  // Reset buttons
  showButtons('create');
  toggleNotificationProvider();
}

function showButtons(mode) {
  const saveBtn = $('#notif-save-btn');
  const updateBtn = $('#notif-update-btn');
  const saveNewBtn = $('#notif-save-new-btn');
  const cancelBtn = $('#notif-cancel-btn');

  if (mode === 'edit') {
    if (saveBtn) saveBtn.classList.add('hidden');
    if (updateBtn) updateBtn.classList.remove('hidden');
    if (saveNewBtn) saveNewBtn.classList.remove('hidden');
    if (cancelBtn) cancelBtn.classList.remove('hidden');
  } else {
    if (saveBtn) saveBtn.classList.remove('hidden');
    if (updateBtn) updateBtn.classList.add('hidden');
    if (saveNewBtn) saveNewBtn.classList.add('hidden');
    if (cancelBtn) cancelBtn.classList.add('hidden');
  }
}

// ---- Load configs table ----

async function loadNotificationConfigs() {
  const tbody = document.querySelector('#notifConfigsTable tbody');
  const noConfigs = $('#notifNoConfigs');

  try {
    const [configsRes, schedulesRes] = await Promise.all([
      fetch('/api/notifications/configs'),
      fetch('/api/schedules')
    ]);
    const j = await configsRes.json();
    const schedData = await schedulesRes.json();

    // Build a set of notification config IDs that are linked to schedules
    const linkedConfigIds = new Set();
    if (schedData.success && schedData.schedules) {
      for (const s of schedData.schedules) {
        if (s.notificationConfigId) linkedConfigIds.add(s.notificationConfigId);
      }
    }

    if (j.success && j.configs && j.configs.length > 0) {
      tbody.innerHTML = '';
      noConfigs.classList.add('hidden');
      document.querySelector('#notifConfigsTable').classList.remove('hidden');

      for (const cfg of j.configs) {
        let providerLabel = cfg.provider;
        if (cfg.provider === 'smtp') providerLabel = 'SMTP';
        else if (cfg.provider === 'msgraph') providerLabel = 'Microsoft Graph';

        let from = '';
        if (cfg.smtp) from = cfg.smtp.from || '';
        else if (cfg.msgraph) from = cfg.msgraph.from || '';

        const recipients = cfg.recipients || '';
        const count = recipients ? recipients.split(',').length : 0;
        const recipientDisplay = count > 1 ? `${count} recipients` : (recipients || 'None');
        const isLinked = linkedConfigIds.has(cfg.id);

        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td data-label="Name">${cfg.name}</td>
          <td data-label="Provider">${providerLabel}</td>
          <td data-label="From">${from}</td>
          <td data-label="Recipients" title="${recipients}">${recipientDisplay}</td>
          <td class="col-actions" data-label="Actions">
            <button type="button" class="btn-small btn-secondary edit-btn">Edit</button>
            <button type="button" class="btn-small btn-secondary test-btn">Test</button>
            <button type="button" class="btn-small btn-danger delete-btn" ${isLinked ? 'disabled title="Linked to a schedule"' : ''}>Delete</button>
          </td>
        `;
        tbody.appendChild(tr);

        tr.querySelector('.edit-btn').onclick = () => editNotificationConfig(cfg.id);
        tr.querySelector('.test-btn').onclick = () => testNotificationConfig(cfg.id);
        if (!isLinked) tr.querySelector('.delete-btn').onclick = () => deleteNotificationConfig(cfg.id);
      }
    } else {
      tbody.innerHTML = '';
      document.querySelector('#notifConfigsTable').classList.add('hidden');
      noConfigs.classList.remove('hidden');
    }
  } catch (e) {
    tbody.innerHTML = '';
    noConfigs.textContent = 'Error loading configurations: ' + e.message;
    noConfigs.classList.remove('hidden');
  }
}

// ---- Edit ----

async function editNotificationConfig(id) {
  try {
    const r = await fetch(`/api/notifications/configs/${id}`);
    const j = await r.json();

    if (!j.success || !j.config) {
      showMessage('Failed to load configuration', 'error');
      return;
    }

    const cfg = j.config;
    editingConfigId = id;
    originalConfigName = cfg.name || '';

    $('#notif-name').value = cfg.name || '';
    $('#notif-provider').value = cfg.provider || '';
    toggleNotificationProvider();

    if (cfg.smtp) {
      $('#smtp-host').value = cfg.smtp.host || '';
      $('#smtp-port').value = cfg.smtp.port || 25;
      $('#smtp-security').value = cfg.smtp.security || 'none';
      $('#smtp-ignore-tls').checked = cfg.smtp.ignoreTLS === true;
      $('#smtp-username').value = cfg.smtp.username || '';
      $('#smtp-password').value = cfg.smtp.password === true ? '********' : '';
      $('#smtp-from-name').value = cfg.smtp.fromName || '';
      $('#smtp-from').value = cfg.smtp.from || '';
    }

    if (cfg.msgraph) {
      $('#msgraph-tenant').value = cfg.msgraph.tenantId || '';
      $('#msgraph-client-id').value = cfg.msgraph.clientId || '';
      $('#msgraph-client-secret').value = cfg.msgraph.clientSecret === true ? '********' : '';
      $('#msgraph-from').value = cfg.msgraph.from || '';
    }

    $('#notif-recipients').value = cfg.recipients || '';
    $('#notif-cc').value = cfg.cc || '';
    $('#notif-bcc').value = cfg.bcc || '';

    showButtons('edit');
    showFormView('edit');

    // Capture initial state after loading for dirty check
    initialFormState = captureFormState();
  } catch (e) {
    showMessage('Error loading configuration: ' + e.message, 'error');
  }
}

// ---- Save ----

async function saveNotificationConfig(saveAsNew = false) {
  const name = $('#notif-name').value.trim();
  const provider = $('#notif-provider').value;

  if (!name) {
    showMessage('Please enter a configuration name', 'error');
    return;
  }
  if (!provider) {
    showMessage('Please select a notification provider', 'error');
    return;
  }

  // Check duplicate names
  try {
    const existingR = await fetch('/api/notifications/configs');
    const existingJ = await existingR.json();
    if (existingJ.success && existingJ.configs) {
      const existingNames = existingJ.configs
        .filter(c => c.id !== editingConfigId)
        .map(c => c.name.toLowerCase());
      if (existingNames.includes(name.toLowerCase())) {
        showMessage(`A configuration named "${name}" already exists`, 'error');
        return;
      }
    }
  } catch (e) {
    console.warn('Error checking duplicates:', e);
  }

  const config = { name, provider };

  config.recipients = $('#notif-recipients').value;
  config.cc = $('#notif-cc').value;
  config.bcc = $('#notif-bcc').value;

  if (provider === 'smtp') {
    const password = $('#smtp-password').value;
    config.smtp = {
      host: $('#smtp-host').value,
      port: parseInt($('#smtp-port').value) || 25,
      security: $('#smtp-security').value,
      ignoreTLS: $('#smtp-ignore-tls').checked,
      username: $('#smtp-username').value,
      password: password === '********' ? undefined : password,
      from: $('#smtp-from').value,
      fromName: $('#smtp-from-name').value
    };
    if (!config.smtp.host || !config.smtp.username || !config.smtp.from) {
      showMessage('Please fill in all required SMTP fields', 'error');
      return;
    }
  } else if (provider === 'msgraph') {
    const clientSecret = $('#msgraph-client-secret').value;
    config.msgraph = {
      tenantId: $('#msgraph-tenant').value,
      clientId: $('#msgraph-client-id').value,
      clientSecret: clientSecret === '********' ? undefined : clientSecret,
      from: $('#msgraph-from').value
    };
    if (!config.msgraph.tenantId || !config.msgraph.clientId || !config.msgraph.from) {
      showMessage('Please fill in all required Microsoft Graph fields', 'error');
      return;
    }
  }

  // Build test config with full credentials for testing
  const testConfig = { ...config };
  if (provider === 'smtp') {
    const smtpPassword = $('#smtp-password').value;
    testConfig.smtp = {
      ...config.smtp,
      password: smtpPassword === '********' ? undefined : smtpPassword
    };
  } else if (provider === 'msgraph') {
    const clientSecret = $('#msgraph-client-secret').value;
    testConfig.msgraph = {
      ...config.msgraph,
      clientSecret: clientSecret === '********' ? undefined : clientSecret
    };
  }

  // Check if editing existing config with masked password - use stored config test
  const smtpPassword = $('#smtp-password').value;
  const msgraphSecret = $('#msgraph-client-secret').value;
  const secretIsMasked = (provider === 'smtp' && smtpPassword === '********') ||
                         (provider === 'msgraph' && msgraphSecret === '********');

  // Show loading state on buttons
  const saveBtn = $('#notif-save-btn');
  const updateBtn = $('#notif-update-btn');
  const saveNewBtn = $('#notif-save-new-btn');
  const activeBtn = saveAsNew ? saveNewBtn : (editingConfigId ? updateBtn : saveBtn);

  if (activeBtn) {
    activeBtn.disabled = true;
    activeBtn.classList.add('btn-loading');
  }
  // Disable the other button in edit mode
  if (saveAsNew && updateBtn) updateBtn.disabled = true;
  else if (!saveAsNew && saveNewBtn) saveNewBtn.disabled = true;

  try {
    let testSuccess = false;

    if (editingConfigId && secretIsMasked) {
      // Use saved config endpoint for testing, but pass updated form values
      // The backend will merge with stored credentials
      const testR = await fetch(`/api/notifications/configs/${editingConfigId}/test`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(config)
      });
      const testJ = await testR.json();
      testSuccess = testJ.success;
      if (!testSuccess) {
        showMessage(testJ.error || 'Test email failed. Configuration not saved.', 'error');
        return;
      }
    } else {
      // Test with provided credentials
      const testR = await fetch('/api/notifications/test', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(testConfig)
      });
      const testJ = await testR.json();
      testSuccess = testJ.success;
      if (!testSuccess) {
        showMessage(testJ.error || 'Test email failed. Configuration not saved.', 'error');
        return;
      }
    }

    // Test succeeded, now save

    const url = editingConfigId
      ? `/api/notifications/configs/${editingConfigId}`
      : '/api/notifications/configs';
    const method = editingConfigId ? 'PUT' : 'POST';

    const r = await fetch(url, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(config)
    });
    const j = await r.json();

    if (j.success) {
      showMessage(editingConfigId
        ? 'Test email sent. Configuration updated successfully.'
        : 'Test email sent. Configuration saved successfully.');
      clearNotificationForm();
      showListView();
      loadNotificationConfigs();
    } else {
      showMessage(j.error || 'Failed to save configuration', 'error');
    }
  } catch (e) {
    showMessage('Error: ' + e.message, 'error');
  } finally {
    // Restore button states
    if (activeBtn) {
      activeBtn.disabled = false;
      activeBtn.classList.remove('btn-loading');
    }
    // Re-enable the other button
    if (saveAsNew && updateBtn) updateBtn.disabled = false;
    else if (!saveAsNew && saveNewBtn) saveNewBtn.disabled = false;
  }
}

async function saveNotificationConfigAsNew() {
  const currentName = $('#notif-name').value.trim();
  if (originalConfigName && currentName.toLowerCase() === originalConfigName.toLowerCase()) {
    showMessage('Please change the configuration name before using "Save As New"', 'error');
    return;
  }

  // Check if the secret is still masked - user needs to enter a new password for the new config
  const provider = $('#notif-provider').value;
  const smtpPassword = $('#smtp-password').value;
  const msgraphSecret = $('#msgraph-client-secret').value;

  if (provider === 'smtp' && smtpPassword === '********') {
    showMessage('Please enter a password for the new configuration', 'error');
    $('#smtp-password').focus();
    return;
  }
  if (provider === 'msgraph' && msgraphSecret === '********') {
    showMessage('Please enter a client secret for the new configuration', 'error');
    $('#msgraph-client-secret').focus();
    return;
  }

  editingConfigId = null;
  await saveNotificationConfig(true);
}

// ---- Delete ----

async function deleteNotificationConfig(id) {
  const confirmed = await window.showConfirmModal(
    'Are you sure you want to delete this notification configuration? This cannot be undone.',
    'Delete Configuration'
  );
  if (!confirmed) return;

  try {
    const r = await fetch(`/api/notifications/configs/${id}`, { method: 'DELETE' });
    const j = await r.json();

    if (j.success) {
      showMessage('Configuration deleted successfully');
      loadNotificationConfigs();
    } else {
      showMessage(j.error || 'Failed to delete configuration', 'error');
    }
  } catch (e) {
    showMessage('Error: ' + e.message, 'error');
  }
}

// ---- Test ----

async function testNotificationConfig(id) {
  try {
    showMessage('Sending test email...');

    const r = await fetch(`/api/notifications/configs/${id}/test`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' }
    });
    const j = await r.json();

    if (j.success) {
      showMessage('Test email sent successfully! Check your inbox.');
    } else {
      showMessage(j.error || 'Failed to send test email', 'error');
    }
  } catch (e) {
    showMessage('Error: ' + e.message, 'error');
  }
}

async function testNotification() {
  const provider = $('#notif-provider').value;
  if (!provider) {
    showMessage('Please select a notification provider first', 'error');
    return;
  }

  // Check if we're editing an existing config and the secret is masked
  // In that case, use the stored credentials by calling the saved config test endpoint
  const smtpPassword = $('#smtp-password').value;
  const msgraphSecret = $('#msgraph-client-secret').value;
  const secretIsMasked = (provider === 'smtp' && smtpPassword === '********') ||
                         (provider === 'msgraph' && msgraphSecret === '********');

  if (editingConfigId && secretIsMasked) {
    // Use the saved config endpoint which has access to stored credentials
    return testNotificationConfig(editingConfigId);
  }

  const config = {
    name: $('#notif-name').value.trim() || 'Unsaved Configuration',
    provider,
    recipients: $('#notif-recipients').value,
    cc: $('#notif-cc').value,
    bcc: $('#notif-bcc').value
  };

  if (provider === 'smtp') {
    config.smtp = {
      host: $('#smtp-host').value,
      port: parseInt($('#smtp-port').value) || 25,
      security: $('#smtp-security').value,
      ignoreTLS: $('#smtp-ignore-tls').checked,
      username: $('#smtp-username').value,
      password: smtpPassword,
      from: $('#smtp-from').value,
      fromName: $('#smtp-from-name').value
    };
  } else if (provider === 'msgraph') {
    config.msgraph = {
      tenantId: $('#msgraph-tenant').value,
      clientId: $('#msgraph-client-id').value,
      clientSecret: msgraphSecret,
      from: $('#msgraph-from').value
    };
  }

  try {
    showMessage('Sending test email...');

    const r = await fetch('/api/notifications/test', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(config)
    });
    const j = await r.json();

    if (j.success) {
      showMessage('Test email sent successfully! Check your inbox. Saving configuration...');
      await saveNotificationConfig();
    } else {
      showMessage(j.error || 'Failed to send test email', 'error');
    }
  } catch (e) {
    showMessage('Error: ' + e.message, 'error');
  }
}

// ---- Tab init / cleanup ----

export async function initNotifications() {
  // Wire up event handlers
  const providerSelect = $('#notif-provider');
  if (providerSelect) providerSelect.onchange = toggleNotificationProvider;

  const securitySelect = $('#smtp-security');
  if (securitySelect) securitySelect.onchange = updateSmtpPort;

  const saveBtn = $('#notif-save-btn');
  if (saveBtn) saveBtn.onclick = saveNotificationConfig;

  const updateBtn = $('#notif-update-btn');
  if (updateBtn) updateBtn.onclick = saveNotificationConfig;

  const saveNewBtn = $('#notif-save-new-btn');
  if (saveNewBtn) saveNewBtn.onclick = saveNotificationConfigAsNew;

  // Add button - show form for adding new configuration
  const addBtn = $('#notif-add-btn');
  if (addBtn) addBtn.onclick = () => {
    clearNotificationForm();
    showFormView('add');
    // Capture initial state for dirty check
    initialFormState = captureFormState();
  };

  // Back button - return to list view with unsaved changes check
  const backBtn = $('#notif-back-btn');
  if (backBtn) backBtn.onclick = async () => {
    if (isFormDirty()) {
      const confirmed = await window.showConfirmModal(
        'You have unsaved changes. Are you sure you want to go back?',
        'Unsaved Changes',
        { destructive: true }
      );
      if (!confirmed) return;
    }
    clearNotificationForm();
    showListView();
  };

  await loadNotificationConfigs();
}

export function cleanupNotifications() {
  editingConfigId = null;
}
