// reports.js - Reports tab module

const $ = (s) => document.querySelector(s);

let hasDataAvailable = false;
let reportingChart = null;
let lastOutputs = null;

// ── Lifecycle ──

export async function initReports() {
  await loadFilters();
  await loadNotificationConfigs();
  await checkForData();
  setupEventListeners();
}

export function cleanupReports() {
  if (reportingChart) {
    reportingChart.destroy();
    reportingChart = null;
  }
}

// ── Event Listeners ──

function setupEventListeners() {
  const sendReportsBtn = $("#sendReportsBtn");
  const refreshReportingBtn = $("#refreshReportingBtn");
  const clearDatesBtn = $("#clearDatesBtn");
  const reportStartDate = $("#reportStartDate");
  const reportEndDate = $("#reportEndDate");
  const downloadFormat = $("#downloadFormat");
  const reportFilter = $("#reportFilter");

  if (sendReportsBtn) sendReportsBtn.onclick = handleSendReports;
  if (refreshReportingBtn) refreshReportingBtn.onclick = handleRefreshReporting;
  if (clearDatesBtn) clearDatesBtn.onclick = clearDates;
  if (reportStartDate) reportStartDate.onchange = () => renderReporting();
  if (reportEndDate) reportEndDate.onchange = () => renderReporting();
  if (downloadFormat) downloadFormat.onchange = () => setDownloadButtons();
  if (reportFilter) reportFilter.onchange = () => setDownloadButtons();
}

// ── Filters ──

async function loadFilters() {
  const filterSelect = $("#reportFilter");
  if (!filterSelect) return;

  try {
    const r = await fetch("/queries");
    const j = await r.json();
    if (j.ok && j.queries && j.queries.length > 0) {
      filterSelect.innerHTML = '<option value="all">All Filters</option>';
      for (const q of j.queries) {
        const opt = document.createElement("option");
        opt.value = q.name;
        opt.textContent = q.name;
        opt.title = q.prefix ? `Prefix: ${q.prefix}` : q.name;
        filterSelect.appendChild(opt);
      }
    } else {
      filterSelect.innerHTML = '<option value="all">All Filters</option>';
    }
  } catch (e) {
    console.warn("Failed to load filters:", e);
  }
}

// ── Notification Configs ──

async function loadNotificationConfigs() {
  const list = $("#reportNotifList");
  const noConfigsMsg = $("#reportNoNotifConfigs");
  if (!list) return;

  try {
    const r = await fetch("/api/notifications/configs");
    const j = await r.json();
    list.innerHTML = '';

    if (j.success && j.configs && j.configs.length > 0) {
      noConfigsMsg?.classList.add("hidden");
      list.classList.remove("hidden");

      for (const cfg of j.configs) {
        const row = document.createElement("div");
        row.className = "toggle-row";
        row.innerHTML = `
          <span class="toggle-label">${cfg.name}</span>
          <label class="toggle-switch">
            <input type="checkbox" class="report-notif-toggle" data-id="${cfg.id}" value="${cfg.id}" title="Send to ${cfg.name}">
            <span class="toggle-slider"></span>
          </label>
        `;
        // Add change handler for each toggle
        const checkbox = row.querySelector('input[type="checkbox"]');
        if (checkbox) checkbox.onchange = updateSendButton;
        list.appendChild(row);
      }
    } else {
      noConfigsMsg?.classList.remove("hidden");
      list.classList.add("hidden");
    }
  } catch (e) {
    console.warn("Failed to load notification configs:", e);
    noConfigsMsg?.classList.remove("hidden");
    list.classList.add("hidden");
  }
  updateSendButton();
}

function getSelectedNotificationConfigs() {
  const checkboxes = document.querySelectorAll(".report-notif-toggle:checked");
  return Array.from(checkboxes).map(cb => parseInt(cb.dataset.id, 10));
}

function updateSendButton() {
  const btn = $("#sendReportsBtn");
  if (!btn) return;

  const selectedConfigs = getSelectedNotificationConfigs();
  const hasConfig = selectedConfigs.length > 0;

  btn.disabled = !(hasConfig && hasDataAvailable);
}

// ── Send Reports ──

function showMessage(text, type = 'info') {
  const msgEl = $("#sendReportsMsg");
  if (!msgEl) return;
  msgEl.textContent = text;
  msgEl.className = 'message';
  if (type === 'error') msgEl.classList.add('error');
  else if (type === 'success') msgEl.classList.add('success');
}

function clearMessage() {
  const msgEl = $("#sendReportsMsg");
  if (msgEl) {
    msgEl.textContent = '';
    msgEl.className = 'message';
  }
}

async function handleSendReports() {
  const sendBtn = $("#sendReportsBtn");
  const selectedConfigs = getSelectedNotificationConfigs();

  if (selectedConfigs.length === 0) {
    showMessage("Select at least one email configuration.", "error");
    return;
  }

  // Show loading spinner on button
  if (sendBtn) {
    sendBtn.disabled = true;
    sendBtn.classList.add("btn-loading");
  }
  showMessage("Generating and sending reports...");

  const attachments = [];
  if ($("#reportAttachDetailed")?.checked) attachments.push("detailed");
  if ($("#reportAttachSummary")?.checked) attachments.push("summary");
  if ($("#reportAttachExceptions")?.checked) attachments.push("exceptions");

  const filterValue = $("#reportFilter")?.value || "all";
  const reportStartDate = $("#reportEmailStart")?.value || "";
  const reportEndDate = $("#reportEmailEnd")?.value || "";

  let successCount = 0;
  let errorMessages = [];

  // Send to each selected notification config
  for (const notifConfigId of selectedConfigs) {
    try {
      const r = await fetch("/send-reports", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          notificationConfigId: notifConfigId,
          attachments,
          reportStartDate,
          reportEndDate,
          queryFilter: filterValue
        })
      });
      const j = await r.json();
      if (j.ok) {
        successCount++;
      } else {
        errorMessages.push(j.message || "Failed to send");
      }
    } catch (e) {
      errorMessages.push(e.message);
    }
  }

  clearMessage();
  if (successCount === selectedConfigs.length) {
    const plural = successCount > 1 ? `to ${successCount} configurations` : "";
    window.showStatusModal(`Reports have been sent successfully${plural}!`, "success", "Reports Sent");
  } else if (successCount > 0) {
    window.showStatusModal(`Sent to ${successCount} of ${selectedConfigs.length} configurations. Errors: ${errorMessages.join(", ")}`, "warning", "Partial Success");
  } else {
    showMessage("Failed to send reports: " + errorMessages.join(", "), "error");
  }

  // Remove loading spinner
  if (sendBtn) {
    sendBtn.disabled = false;
    sendBtn.classList.remove("btn-loading");
  }
}

// ── Data Check ──

async function checkForData() {
  const noDataBanner = $("#noDataBanner");
  const sendReportsSection = $("#sendReportsSection");
  const downloadsCard = $("#downloadsCard");

  try {
    const r = await fetch(`/status?_t=${Date.now()}`);
    const j = await r.json();
    if (j.dataAvailable) {
      hasDataAvailable = true;
      noDataBanner?.classList.add("hidden");
      sendReportsSection?.classList.remove("hidden");
      downloadsCard?.classList.remove("hidden");
      setDownloadButtons({ detailed: true, summary: true, errors: j.exceptionCount > 0 || j.errorCount > 0 });
      renderErrorSummary(j.errorSummary);
      await renderReporting();
      updateSendButton();
    } else {
      hasDataAvailable = false;
      updateSendButton();
      noDataBanner?.classList.remove("hidden");
      sendReportsSection?.classList.add("hidden");
      downloadsCard?.classList.add("hidden");
      $("#errorSummaryCard")?.classList.add("hidden");
      $("#reportingCard")?.classList.add("hidden");
    }
  } catch {
    noDataBanner?.classList.remove("hidden");
    sendReportsSection?.classList.add("hidden");
    downloadsCard?.classList.add("hidden");
    $("#errorSummaryCard")?.classList.add("hidden");
    $("#reportingCard")?.classList.add("hidden");
  }
}

function setDownloadButtons(outputs) {
  if (outputs !== undefined) lastOutputs = outputs;
  const current = lastOutputs;

  const btnDetailed = $("#btnDetailed");
  const btnSummary = $("#btnSummary");
  const btnErrors = $("#btnErrors");
  const fmt = $("#downloadFormat")?.value || "csv";
  const filter = $("#reportFilter")?.value || "all";

  const setBtn = (btn, available, path) => {
    if (!btn) return;
    btn.disabled = !available;
    btn.onclick = available ? () => window.location = `${path}?format=${fmt}&filter=${encodeURIComponent(filter)}` : null;
  };

  setBtn(btnDetailed, current && current.detailed, "/download/detailed");
  setBtn(btnSummary, current && current.summary, "/download/summary");
  setBtn(btnErrors, current && current.errors, "/download/exceptions");
}

function clearDates() {
  const startDateInput = $("#reportStartDate");
  const endDateInput = $("#reportEndDate");
  if (startDateInput) startDateInput.value = "";
  if (endDateInput) endDateInput.value = "";
  renderReporting();
}

async function handleRefreshReporting() {
  const refreshBtn = $("#refreshReportingBtn");
  if (!refreshBtn) return;

  refreshBtn.disabled = true;
  refreshBtn.textContent = "Refreshing...";
  try {
    await checkForData();
    refreshBtn.textContent = "Refreshed!";
    setTimeout(() => { refreshBtn.textContent = "Refresh Data"; }, 2000);
  } catch {
    refreshBtn.textContent = "Failed to refresh";
    setTimeout(() => { refreshBtn.textContent = "Refresh Data"; }, 2000);
  } finally {
    refreshBtn.disabled = false;
  }
}

function daysInMonth(ym) {
  const [y, m] = ym.split("-").map(x => parseInt(x, 10));
  if (!y || !m) return 30;
  return new Date(y, m, 0).getDate();
}

async function renderReporting() {
  const reportingCard = $("#reportingCard");
  const reportingTableBody = document.querySelector("#reportingTable tbody");
  const totalReqsCell = $("#totalReqs");
  const avgOverallCell = $("#avgOverall");
  const avgPerMonthText = $("#avgPerMonthText");

  try {
    const params = new URLSearchParams();
    const startDate = $("#reportStartDate")?.value;
    const endDate = $("#reportEndDate")?.value;
    if (startDate) params.set("startDate", startDate);
    if (endDate) params.set("endDate", endDate);
    params.set("_t", Date.now());
    const url = `/reporting.json?${params.toString()}`;

    const r = await fetch(url);
    const j = await r.json();
    if (!j.ok || !j.monthly || !j.monthly.length) {
      reportingCard?.classList.add("hidden");
      return;
    }

    if (reportingTableBody) reportingTableBody.innerHTML = "";
    let grand = 0, totalDays = 0;
    const monthsCount = j.monthly.length;
    const queryName = j.queryName || "N/A";

    for (const row of j.monthly) {
      const month = row.month;
      const count = Number(row.count || 0);
      const dim = daysInMonth(month);
      const avgDay = dim ? (count / dim) : 0;
      const tr = document.createElement("tr");
      tr.innerHTML = `<td>${month}</td><td data-label="API Usage">${count}</td><td data-label="Avg / Day">${avgDay.toFixed(2)}</td><td data-label="Filter">${queryName}</td>`;
      reportingTableBody?.appendChild(tr);
      grand += count;
      totalDays += dim;
    }

    if (totalReqsCell) totalReqsCell.textContent = grand;
    if (avgOverallCell) avgOverallCell.textContent = (totalDays ? (grand / totalDays) : 0).toFixed(2);
    const filterNameFooter = document.getElementById("filterNameFooter");
    if (filterNameFooter) filterNameFooter.textContent = queryName;
    const avgPerMonth = monthsCount ? (grand / monthsCount) : 0;
    if (avgPerMonthText) avgPerMonthText.textContent = `Average per month: ${avgPerMonth.toFixed(2)}`;

    // Chart
    const labels = j.monthly.map(x => x.month);
    const data = j.monthly.map(x => x.count);
    if (reportingChart) { reportingChart.destroy(); reportingChart = null; }

    const chartCanvas = document.getElementById("reportingChart");
    if (chartCanvas && window.Chart) {
      const datasets = [];

      // Per-filter lines when data is available
      if (j.monthlyByFilter && Object.keys(j.monthlyByFilter).length > 1) {
        const filterColors = [
          "#3b82f6", "#ef4444", "#10b981", "#f59e0b", "#8b5cf6",
          "#ec4899", "#06b6d4", "#84cc16", "#f97316", "#6366f1"
        ];
        const filterNames = Object.keys(j.monthlyByFilter).sort();
        filterNames.forEach((name, i) => {
          const filterMonthly = j.monthlyByFilter[name];
          const monthMap = {};
          for (const m of filterMonthly) monthMap[m.month] = m.count;
          const color = filterColors[i % filterColors.length];
          datasets.push({
            label: name,
            data: labels.map(l => monthMap[l] || 0),
            borderColor: color,
            backgroundColor: color + "20",
            borderWidth: 2,
            pointRadius: 3,
            tension: 0.1
          });
        });
        // Add total line (dashed)
        datasets.push({
          label: "Total",
          data,
          borderColor: "#6b7280",
          backgroundColor: "#6b728020",
          borderWidth: 2,
          borderDash: [6, 3],
          pointRadius: 2,
          tension: 0.1
        });
      } else {
        datasets.push({ label: "API Usage per Month", data });
      }

      reportingChart = new Chart(chartCanvas, {
        type: "line",
        data: { labels, datasets },
        options: { responsive: true }
      });
    }

    // Query breakdown
    const breakdownSection = document.getElementById("filterBreakdownSection");
    const breakdownTableBody = document.querySelector("#filterBreakdownTable tbody");
    const breakdownTotalCell = document.getElementById("breakdownTotal");
    const breakdownCompanyTotalCell = document.getElementById("breakdownCompanyTotal");

    if (j.queryBreakdown && Object.keys(j.queryBreakdown).length > 0) {
      if (breakdownTableBody) breakdownTableBody.innerHTML = "";
      const entries = Object.entries(j.queryBreakdown).sort((a, b) => b[1].count - a[1].count);
      const totalCount = entries.reduce((sum, [, v]) => sum + v.count, 0);
      const totalCompanies = entries.reduce((sum, [, v]) => sum + (v.companyCount || 0), 0);
      for (const [name, v] of entries) {
        const tr = document.createElement("tr");
        const pct = totalCount ? ((v.count / totalCount) * 100).toFixed(1) + "%" : "0%";
        const fetched = v.lastFetched ? new Date(v.lastFetched).toLocaleString() : "Never";
        tr.innerHTML = `<td>${name}</td><td data-label="Companies">${(v.companyCount || 0).toLocaleString()}</td><td data-label="API Usage">${v.count.toLocaleString()}</td><td data-label="% of Total">${pct}</td><td data-label="Last Fetched">${fetched}</td>`;
        breakdownTableBody?.appendChild(tr);
      }
      if (breakdownTotalCell) breakdownTotalCell.textContent = totalCount.toLocaleString();
      if (breakdownCompanyTotalCell) breakdownCompanyTotalCell.textContent = totalCompanies.toLocaleString();
      breakdownSection?.classList.remove("hidden");
    } else {
      breakdownSection?.classList.add("hidden");
    }

    reportingCard?.classList.remove("hidden");
  } catch {
    reportingCard?.classList.add("hidden");
  }
}

// ── Error Summary ──

function renderErrorSummary(errorSummary) {
  const card = $("#errorSummaryCard");
  const tbody = document.querySelector("#errorSummaryTable tbody");
  const totalCell = $("#errorSummaryTotal");

  if (!errorSummary || errorSummary.length === 0) {
    card?.classList.add("hidden");
    return;
  }

  if (tbody) tbody.innerHTML = "";
  let total = 0;
  for (const row of errorSummary) {
    const tr = document.createElement("tr");
    const descCell = document.createElement("td");
    descCell.textContent = row.description;
    const countCell = document.createElement("td");
    countCell.className = "col-count";
    countCell.textContent = row.company_count;
    const actionsCell = document.createElement("td");
    actionsCell.className = "col-actions";
    const viewBtn = document.createElement("button");
    viewBtn.className = "btn-secondary btn-small";
    viewBtn.textContent = "View";
    viewBtn.onclick = () => openErrorCompaniesModal(row.description, row.companies || []);
    actionsCell.appendChild(viewBtn);
    tr.appendChild(descCell);
    tr.appendChild(countCell);
    tr.appendChild(actionsCell);
    tbody?.appendChild(tr);
    total += row.company_count;
  }
  if (totalCell) totalCell.textContent = total;
  card?.classList.remove("hidden");
}

// ── Error Companies Modal ──

let ecModalCompanies = [];
let ecSortCol = "index";
let ecSortAsc = true;

function openErrorCompaniesModal(description, companies) {
  const modal = $("#errorCompaniesModal");
  if (!modal) return;
  const descEl = $("#errorCompaniesDescription");
  if (descEl) descEl.textContent = description;

  ecModalCompanies = companies.slice();
  ecSortCol = "index";
  ecSortAsc = true;
  renderModalTable();

  const sortIndex = $("#ecSortIndex");
  const sortName = $("#ecSortName");
  if (sortIndex) sortIndex.onclick = () => toggleModalSort("index");
  if (sortName) sortName.onclick = () => toggleModalSort("name");

  modal.classList.remove("hidden");

  const closeBtn = $("#errorCompaniesCloseBtn");
  if (closeBtn) closeBtn.onclick = closeErrorCompaniesModal;
}

function toggleModalSort(col) {
  if (ecSortCol === col) {
    ecSortAsc = !ecSortAsc;
  } else {
    ecSortCol = col;
    ecSortAsc = true;
  }
  renderModalTable();
}

function renderModalTable() {
  const tbody = document.querySelector("#errorCompaniesTable tbody");
  if (!tbody) return;

  const rows = ecModalCompanies.map((c, i) => ({ name: c, index: i + 1 }));

  if (ecSortCol === "name") {
    rows.sort((a, b) => a.name.localeCompare(b.name));
  } else {
    rows.sort((a, b) => a.index - b.index);
  }
  if (!ecSortAsc) rows.reverse();

  tbody.innerHTML = "";
  for (const row of rows) {
    const tr = document.createElement("tr");
    tr.innerHTML = `<td>${row.index}</td><td>${row.name}</td>`;
    tbody.appendChild(tr);
  }

  updateSortIcons();
}

function updateSortIcons() {
  const indexTh = $("#ecSortIndex");
  const nameTh = $("#ecSortName");
  const arrow = (active, asc) => {
    const cls = active ? "sort-icon active" : "sort-icon";
    const ch = active ? (asc ? "\u25B2" : "\u25BC") : "\u25B2";
    return `<span class="${cls}">${ch}</span>`;
  };
  if (indexTh) indexTh.innerHTML = `# ${arrow(ecSortCol === "index", ecSortAsc)}`;
  if (nameTh) nameTh.innerHTML = `Company/Instance ${arrow(ecSortCol === "name", ecSortAsc)}`;
}

function closeErrorCompaniesModal() {
  const modal = $("#errorCompaniesModal");
  if (modal) modal.classList.add("hidden");
}
