// schedule.js - Schedule configuration tab module

const $ = (s) => document.querySelector(s);

let editingScheduleId = null;
let originalScheduleName = null;
let notificationConfigs = [];
let availableQueries = [];
let messageTimeout = null;
let initialFormState = null;

// ---- Form state tracking for unsaved changes warning ----

function captureFormState() {
  const daysOfWeek = [];
  for (let i = 0; i <= 6; i++) {
    const cb = $(`#sched-day-${i}`);
    if (cb && cb.checked) daysOfWeek.push(String(i));
  }

  const attachments = [];
  for (const type of ['detailed', 'summary', 'exceptions']) {
    const cb = $(`#sched-attach-${type}`);
    if (cb && cb.checked) attachments.push(type);
  }

  // Get selected notification config IDs
  const notifConfigIds = getSelectedNotificationConfigIds().join(',');

  return {
    name: $('#sched-name')?.value || '',
    scheduleType: $('#sched-type')?.value || '',
    oneoffDate: $('#sched-oneoff-date')?.value || '',
    oneoffTime: $('#sched-oneoff-time')?.value || '',
    timeWeekly: $('#sched-time-weekly')?.value || '',
    timeMonthly: $('#sched-time-monthly')?.value || '',
    dayOfMonth: $('#sched-dayofmonth')?.value || '',
    daysOfWeek: daysOfWeek.join(','),
    cron: $('#sched-cron')?.value || '',
    queryFilter: $('#sched-query-filter')?.value || '',
    notifConfigIds: notifConfigIds,
    attachments: attachments.join(','),
    reportsOnly: $('#sched-reports-only')?.checked || false,
    fullRefresh: $('#sched-full-refresh')?.checked || false,
    startDate: $('#sched-start-date')?.value || '',
    completeMonths: $('#sched-complete-months')?.checked || false
  };
}

function getSelectedNotificationConfigIds() {
  const checkboxes = document.querySelectorAll('.sched-notif-toggle:checked');
  return Array.from(checkboxes).map(cb => parseInt(cb.dataset.id, 10));
}

function isFormDirty() {
  if (!initialFormState) return false;
  const currentState = captureFormState();
  return JSON.stringify(currentState) !== JSON.stringify(initialFormState);
}

async function handleBackToList() {
  if (isFormDirty()) {
    const confirmed = await window.showConfirmModal(
      'You have unsaved changes. Are you sure you want to go back?',
      'Unsaved Changes',
      { destructive: true }
    );
    if (!confirmed) return;
  }
  clearScheduleForm();
  showListView();
}

// ---- View switching ----

function showListView() {
  const listView = $('#sched-list-view');
  const formView = $('#sched-form-view');
  if (listView) listView.classList.remove('hidden');
  if (formView) formView.classList.add('hidden');
}

function showFormView(mode = 'add') {
  const listView = $('#sched-list-view');
  const formView = $('#sched-form-view');
  const formTitle = $('#sched-form-title');

  if (listView) listView.classList.add('hidden');
  if (formView) formView.classList.remove('hidden');

  if (formTitle) {
    formTitle.textContent = mode === 'edit' ? 'Edit Schedule' : 'Add Schedule';
  }
}

function showMessage(text, type = 'success', durationMs = 3000) {
  // Skip empty messages
  if (!text) return;

  // Use global status modal
  if (window.showStatusModal) {
    window.showStatusModal(text, type);
  }
}

// ---- Schedule type toggle ----

function toggleScheduleType() {
  const type = $('#sched-type').value;

  const oneoffSection = $('#sched-oneoff-section');
  const weeklySection = $('#sched-weekly-section');
  const monthlySection = $('#sched-monthly-section');
  const cronSection = $('#sched-cron-section');

  if (oneoffSection) oneoffSection.classList.toggle('hidden', type !== 'oneoff');
  if (weeklySection) weeklySection.classList.toggle('hidden', type !== 'weekly');
  if (monthlySection) monthlySection.classList.toggle('hidden', type !== 'monthly');
  if (cronSection) cronSection.classList.toggle('hidden', type !== 'cron');
}

// ---- Reports Only toggle interaction ----

function updateReportsOnlyToggle() {
  const reportsOnlyCheckbox = $('#sched-reports-only');
  const fullRefreshRow = $('#sched-full-refresh-row');
  const fullRefreshCheckbox = $('#sched-full-refresh');

  if (!reportsOnlyCheckbox) return;

  const isReportsOnly = reportsOnlyCheckbox.checked;

  // When Reports Only is checked, disable Full Refresh (not applicable)
  if (fullRefreshRow) {
    fullRefreshRow.classList.toggle('disabled', isReportsOnly);
  }
  if (fullRefreshCheckbox) {
    fullRefreshCheckbox.disabled = isReportsOnly;
    if (isReportsOnly) {
      fullRefreshCheckbox.checked = false;
    }
  }
}

// ---- Form management ----

function clearScheduleForm() {
  editingScheduleId = null;
  originalScheduleName = null;

  if ($('#sched-name')) $('#sched-name').value = '';
  if ($('#sched-type')) $('#sched-type').value = '';
  if ($('#sched-oneoff-date')) $('#sched-oneoff-date').value = '';
  if ($('#sched-oneoff-time')) $('#sched-oneoff-time').value = '08:00';
  if ($('#sched-time-weekly')) $('#sched-time-weekly').value = '08:00';
  if ($('#sched-time-monthly')) $('#sched-time-monthly').value = '08:00';
  if ($('#sched-dayofmonth')) $('#sched-dayofmonth').value = '1';
  if ($('#sched-cron')) $('#sched-cron').value = '';
  if ($('#sched-query-filter')) $('#sched-query-filter').value = 'all';

  // Clear notification config toggles
  document.querySelectorAll('.sched-notif-toggle').forEach(cb => cb.checked = false);

  // Reset attachment checkboxes to all checked
  for (const type of ['detailed', 'summary', 'exceptions']) {
    const cb = $(`#sched-attach-${type}`);
    if (cb) cb.checked = true;
  }

  // Reset reports only and full refresh
  if ($('#sched-reports-only')) $('#sched-reports-only').checked = false;
  if ($('#sched-full-refresh')) $('#sched-full-refresh').checked = false;
  updateReportsOnlyToggle();

  // Reset date range and complete months
  if ($('#sched-start-date')) $('#sched-start-date').value = '';
  if ($('#sched-complete-months')) $('#sched-complete-months').checked = false;

  // Clear day checkboxes
  for (let i = 0; i <= 6; i++) {
    const cb = $(`#sched-day-${i}`);
    if (cb) cb.checked = false;
  }

  showButtons('create');
  toggleScheduleType();

  // Capture initial state for dirty checking
  initialFormState = captureFormState();
}

function showButtons(mode) {
  const saveBtn = $('#sched-save-btn');
  const updateBtn = $('#sched-update-btn');
  const saveNewBtn = $('#sched-save-new-btn');
  const cancelBtn = $('#sched-cancel-btn');

  if (mode === 'edit') {
    if (saveBtn) saveBtn.classList.add('hidden');
    if (updateBtn) updateBtn.classList.remove('hidden');
    if (saveNewBtn) saveNewBtn.classList.remove('hidden');
    if (cancelBtn) cancelBtn.classList.remove('hidden');
  } else {
    if (saveBtn) saveBtn.classList.remove('hidden');
    if (updateBtn) updateBtn.classList.add('hidden');
    if (saveNewBtn) saveNewBtn.classList.add('hidden');
    if (cancelBtn) cancelBtn.classList.add('hidden');
  }
}

// ---- Load notification configs as toggles ----

async function loadNotificationConfigs() {
  const list = $('#sched-notif-list');
  const noConfigsMsg = $('#sched-no-notif-configs');

  try {
    const r = await fetch('/api/schedules/email-configs');
    const j = await r.json();

    if (j.success) {
      notificationConfigs = j.configs;

      if (list) {
        list.innerHTML = '';

        if (j.configs && j.configs.length > 0) {
          if (noConfigsMsg) noConfigsMsg.classList.add('hidden');
          list.classList.remove('hidden');

          for (const cfg of j.configs) {
            const providerLabel = cfg.provider === 'smtp' ? 'SMTP' : 'Graph';
            const row = document.createElement('div');
            row.className = 'toggle-row';
            row.innerHTML = `
              <span class="toggle-label">${cfg.name} (${providerLabel})</span>
              <label class="toggle-switch">
                <input type="checkbox" class="sched-notif-toggle" data-id="${cfg.id}" value="${cfg.id}" title="Send to ${cfg.name}">
                <span class="toggle-slider"></span>
              </label>
            `;
            list.appendChild(row);
          }
        } else {
          if (noConfigsMsg) noConfigsMsg.classList.remove('hidden');
          list.classList.add('hidden');
        }
      }
    }
  } catch (e) {
    console.warn('Error loading notification configs:', e);
    if (noConfigsMsg) noConfigsMsg.classList.remove('hidden');
    if (list) list.classList.add('hidden');
  }
}

async function loadQueries() {
  try {
    const r = await fetch('/queries');
    const j = await r.json();

    if (j.ok && j.queries) {
      availableQueries = j.queries;
      const select = $('#sched-query-filter');
      if (select) {
        select.innerHTML = '<option value="all">All Filters</option>';
        for (const q of j.queries) {
          select.innerHTML += `<option value="${q.name}">${q.name}</option>`;
        }
      }
    }
  } catch (e) {
    console.warn('Error loading queries:', e);
  }
}

// ---- Load schedules table ----

async function loadSchedules() {
  const tbody = document.querySelector('#schedTable tbody');
  const noSchedules = $('#schedNoSchedules');
  const table = $('#schedTable');

  try {
    const r = await fetch('/api/schedules');
    const j = await r.json();

    if (j.success && j.schedules && j.schedules.length > 0) {
      tbody.innerHTML = '';
      if (noSchedules) noSchedules.classList.add('hidden');
      if (table) table.classList.remove('hidden');

      // Check if any schedule is currently running
      const anyRunning = j.schedules.some(s => s.lastRunStatus === 'running');

      for (const sched of j.schedules) {
        const scheduleDesc = formatScheduleDescription(sched);
        const queryFilterDisplay = sched.queryFilter === 'all' ? 'All Filters' : sched.queryFilter;

        // Handle multiple notification config IDs
        let notifName = 'Not set';
        const configIds = sched.notificationConfigIds || (sched.notificationConfigId ? [sched.notificationConfigId] : []);
        if (configIds.length > 0) {
          const names = configIds.map(id => {
            const cfg = notificationConfigs.find(c => c.id === id);
            return cfg ? cfg.name : null;
          }).filter(Boolean);
          notifName = names.length > 0 ? names.join(', ') : 'Not set';
        }

        const lastRunStr = sched.lastRunAt
          ? new Date(sched.lastRunAt).toLocaleString('en-GB')
          : 'Never';
        const nextRunStr = sched.nextRunAt
          ? new Date(sched.nextRunAt).toLocaleString('en-GB')
          : 'Not scheduled';

        let statusHtml;
        if (!sched.enabled) {
          statusHtml = '<span class="status-badge status-disabled">Disabled</span>';
        } else if (sched.lastRunStatus === 'running') {
          statusHtml = '<span class="status-badge status-running">Running</span>';
        } else if (sched.lastRunStatus === 'stopped') {
          statusHtml = '<span class="status-badge status-stopped">Stopped</span>';
        } else if (sched.lastRunStatus === 'success') {
          statusHtml = '<span class="status-badge status-success">Success</span>';
        } else if (sched.lastRunStatus === 'error') {
          statusHtml = `<span class="status-badge status-error" title="${sched.lastRunMessage || ''}">Error</span>`;
        } else {
          statusHtml = '<span class="status-badge status-pending">Pending</span>';
        }

        const isThisRunning = sched.lastRunStatus === 'running';
        const disableToggle = anyRunning ? 'disabled' : '';
        // Disable edit/delete when schedule is enabled or any job is running
        const disableEditDelete = (anyRunning || sched.enabled) ? 'disabled' : '';

        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td data-label="Name">${sched.name}</td>
          <td data-label="Schedule">${scheduleDesc}</td>
          <td data-label="Filter">${queryFilterDisplay}</td>
          <td data-label="Email Config">${notifName}</td>
          <td data-label="Last Run">${lastRunStr}</td>
          <td data-label="Next Run">${nextRunStr}</td>
          <td data-label="Status">${statusHtml}</td>
          <td class="col-actions" data-label="Actions">
            <label class="toggle-switch toggle-inline" title="${sched.enabled ? 'Disable schedule' : 'Enable schedule'}">
              <input type="checkbox" class="toggle-enabled" ${sched.enabled ? 'checked' : ''} ${disableToggle}>
              <span class="toggle-slider"></span>
            </label>
            <button type="button" class="btn-small btn-secondary edit-btn" ${disableEditDelete} title="${sched.enabled ? 'Disable schedule to edit' : 'Edit schedule'}">Edit</button>
            ${isThisRunning
              ? '<button type="button" class="btn-small btn-danger stop-btn">Stop</button>'
              : ''}
            <button type="button" class="btn-small btn-danger delete-btn" ${disableEditDelete} title="${sched.enabled ? 'Disable schedule to delete' : 'Delete schedule'}">Delete</button>
          </td>
        `;
        tbody.appendChild(tr);

        tr.querySelector('.edit-btn').onclick = () => editSchedule(sched.id);
        const stopBtn = tr.querySelector('.stop-btn');
        if (stopBtn) stopBtn.onclick = () => stopSchedule();
        tr.querySelector('.toggle-enabled').onchange = () => toggleSchedule(sched.id);
        tr.querySelector('.delete-btn').onclick = () => deleteSchedule(sched.id);
      }
    } else {
      tbody.innerHTML = '';
      if (table) table.classList.add('hidden');
      if (noSchedules) noSchedules.classList.remove('hidden');
    }
  } catch (e) {
    tbody.innerHTML = '';
    if (noSchedules) {
      noSchedules.textContent = 'Error loading schedules: ' + e.message;
      noSchedules.classList.remove('hidden');
    }
  }
}

function formatScheduleDescription(sched) {
  if (sched.scheduleType === 'oneoff') {
    if (sched.oneoffDatetime) {
      const dt = new Date(sched.oneoffDatetime);
      return `Once on ${dt.toLocaleDateString('en-GB')} at ${dt.toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit' })}`;
    }
    return 'One-off (not set)';
  }

  if (sched.scheduleType === 'weekly') {
    const dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    const days = sched.daysOfWeek.map(d => dayNames[parseInt(d, 10)]).join(', ');
    return `${days} at ${sched.timeOfDay}`;
  }

  if (sched.scheduleType === 'monthly') {
    const day = sched.dayOfMonth;
    let dayStr;
    if (day === -1) {
      dayStr = 'Last day';
    } else {
      const suffix = ['th', 'st', 'nd', 'rd'][(day % 100 > 10 && day % 100 < 14) ? 0 : Math.min(day % 10, 3)];
      dayStr = `${day}${suffix}`;
    }
    return `Monthly on ${dayStr} at ${sched.timeOfDay}`;
  }

  if (sched.scheduleType === 'cron') {
    return `Cron: ${sched.cronExpression || 'Not set'}`;
  }

  // Legacy support for interval/daily
  if (sched.scheduleType === 'interval') {
    const hours = Math.floor(sched.intervalMinutes / 60);
    if (hours === 1) return 'Every hour';
    if (hours > 1) return `Every ${hours} hours`;
    return `Every ${sched.intervalMinutes} minutes`;
  }

  if (sched.scheduleType === 'daily') {
    return `Daily at ${sched.timeOfDay}`;
  }

  return 'Unknown';
}

// ---- Edit schedule ----

async function editSchedule(id) {
  try {
    const r = await fetch(`/api/schedules/${id}`);
    const j = await r.json();

    if (!j.success || !j.schedule) {
      showMessage('Failed to load schedule', 'error');
      return;
    }

    const sched = j.schedule;
    editingScheduleId = id;
    originalScheduleName = sched.name || '';

    $('#sched-name').value = sched.name || '';
    $('#sched-type').value = sched.scheduleType || '';
    toggleScheduleType();

    if (sched.scheduleType === 'oneoff') {
      if (sched.oneoffDatetime) {
        const dt = new Date(sched.oneoffDatetime);
        // Format date as YYYY-MM-DD for input[type="date"]
        const dateStr = dt.toISOString().split('T')[0];
        // Format time as HH:MM for input[type="time"]
        const timeStr = dt.toTimeString().slice(0, 5);
        if ($('#sched-oneoff-date')) $('#sched-oneoff-date').value = dateStr;
        if ($('#sched-oneoff-time')) $('#sched-oneoff-time').value = timeStr;
      }
    }

    if (sched.scheduleType === 'weekly') {
      if ($('#sched-time-weekly')) $('#sched-time-weekly').value = sched.timeOfDay || '08:00';
      if (sched.daysOfWeek) {
        for (let i = 0; i <= 6; i++) {
          const cb = $(`#sched-day-${i}`);
          if (cb) cb.checked = sched.daysOfWeek.includes(String(i));
        }
      }
    }

    if (sched.scheduleType === 'monthly') {
      if ($('#sched-time-monthly')) $('#sched-time-monthly').value = sched.timeOfDay || '08:00';
      if ($('#sched-dayofmonth')) $('#sched-dayofmonth').value = sched.dayOfMonth || 1;
    }

    if (sched.scheduleType === 'cron') {
      if ($('#sched-cron')) $('#sched-cron').value = sched.cronExpression || '';
    }

    $('#sched-query-filter').value = sched.queryFilter || 'all';

    // Set notification config toggles
    const configIds = sched.notificationConfigIds || (sched.notificationConfigId ? [sched.notificationConfigId] : []);
    document.querySelectorAll('.sched-notif-toggle').forEach(cb => {
      cb.checked = configIds.includes(parseInt(cb.dataset.id, 10));
    });

    // Populate attachment checkboxes
    const schedAttachments = sched.attachments || ['detailed', 'summary', 'exceptions'];
    for (const type of ['detailed', 'summary', 'exceptions']) {
      const cb = $(`#sched-attach-${type}`);
      if (cb) cb.checked = schedAttachments.includes(type);
    }

    // Populate reports only and full refresh toggles
    if ($('#sched-reports-only')) $('#sched-reports-only').checked = sched.reportsOnly || false;
    if ($('#sched-full-refresh')) $('#sched-full-refresh').checked = sched.fullRefresh || false;
    updateReportsOnlyToggle();

    // Populate date range and complete months
    if ($('#sched-start-date')) $('#sched-start-date').value = sched.reportStartDate || '';
    if ($('#sched-complete-months')) $('#sched-complete-months').checked = sched.completeMonthsOnly || false;

    showButtons('edit');
    showFormView('edit');

    // Capture initial state for dirty checking
    initialFormState = captureFormState();
  } catch (e) {
    showMessage('Error loading schedule: ' + e.message, 'error');
  }
}

// ---- Save schedule ----

async function saveSchedule() {
  const name = $('#sched-name').value.trim();
  const scheduleType = $('#sched-type').value;

  if (!name) {
    showMessage('Please enter a schedule name', 'error');
    return;
  }
  if (!scheduleType) {
    showMessage('Please select a schedule type', 'error');
    return;
  }

  // Check duplicate names
  try {
    const existingR = await fetch('/api/schedules');
    const existingJ = await existingR.json();
    if (existingJ.success && existingJ.schedules) {
      const existingNames = existingJ.schedules
        .filter(s => s.id !== editingScheduleId)
        .map(s => s.name.toLowerCase());
      if (existingNames.includes(name.toLowerCase())) {
        showMessage(`A schedule named "${name}" already exists`, 'error');
        return;
      }
    }
  } catch (e) {
    console.warn('Error checking duplicate names:', e);
  }

  const notifConfigIds = getSelectedNotificationConfigIds();
  if (notifConfigIds.length === 0) {
    showMessage('Please select at least one email configuration', 'error');
    return;
  }

  const queryFilter = $('#sched-query-filter').value || 'all';

  // Collect selected attachments
  const attachments = [];
  for (const type of ['detailed', 'summary', 'exceptions']) {
    const cb = $(`#sched-attach-${type}`);
    if (cb && cb.checked) attachments.push(type);
  }
  if (attachments.length === 0) {
    showMessage('Please select at least one report attachment', 'error');
    return;
  }

  const reportsOnly = $('#sched-reports-only')?.checked || false;
  const fullRefresh = $('#sched-full-refresh')?.checked || false;
  const reportStartDate = $('#sched-start-date')?.value || '';
  const completeMonthsOnly = $('#sched-complete-months')?.checked || false;

  const data = {
    name,
    scheduleType,
    queryFilter,
    attachments,
    reportsOnly,
    fullRefresh,
    reportStartDate,
    completeMonthsOnly,
    notificationConfigIds: notifConfigIds
  };

  if (scheduleType === 'oneoff') {
    const oneoffDate = $('#sched-oneoff-date').value;
    const oneoffTime = $('#sched-oneoff-time').value || '08:00';
    if (!oneoffDate) {
      showMessage('Please select a date for the one-off schedule', 'error');
      return;
    }
    // Combine date and time into ISO datetime
    data.oneoffDatetime = `${oneoffDate}T${oneoffTime}:00`;

    // Validate that the datetime is in the future
    const scheduledTime = new Date(data.oneoffDatetime);
    if (scheduledTime <= new Date()) {
      showMessage('One-off schedule must be set in the future', 'error');
      return;
    }
  }

  if (scheduleType === 'weekly') {
    data.timeOfDay = $('#sched-time-weekly').value;
    const days = [];
    for (let i = 0; i <= 6; i++) {
      const cb = $(`#sched-day-${i}`);
      if (cb && cb.checked) days.push(String(i));
    }
    if (days.length === 0) {
      showMessage('Please select at least one day of the week', 'error');
      return;
    }
    data.daysOfWeek = days;
  }

  if (scheduleType === 'monthly') {
    data.timeOfDay = $('#sched-time-monthly').value;
    data.dayOfMonth = parseInt($('#sched-dayofmonth').value, 10);
  }

  if (scheduleType === 'cron') {
    const cronExpr = $('#sched-cron').value.trim();
    if (!cronExpr) {
      showMessage('Please enter a cron expression', 'error');
      return;
    }
    data.cronExpression = cronExpr;
  }

  try {
    const url = editingScheduleId
      ? `/api/schedules/${editingScheduleId}`
      : '/api/schedules';
    const method = editingScheduleId ? 'PUT' : 'POST';

    const r = await fetch(url, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(data)
    });
    const j = await r.json();

    if (j.success) {
      const isNew = !editingScheduleId;
      const baseMsg = isNew ? 'Schedule saved' : 'Schedule updated';
      showMessage(`${baseMsg} successfully. Remember to enable it using the toggle.`);
      clearScheduleForm();
      showListView();
      loadSchedules();
    } else {
      showMessage(j.error || 'Failed to save schedule', 'error');
    }
  } catch (e) {
    showMessage('Error: ' + e.message, 'error');
  }
}

function saveScheduleAsNew() {
  const currentName = $('#sched-name').value.trim();
  if (originalScheduleName && currentName.toLowerCase() === originalScheduleName.toLowerCase()) {
    showMessage('Please change the schedule name before using "Save As New"', 'error');
    return;
  }
  editingScheduleId = null;
  saveSchedule();
}

// ---- Delete schedule ----

async function deleteSchedule(id) {
  const confirmed = await window.showConfirmModal(
    'Are you sure you want to delete this schedule? This cannot be undone.',
    'Delete Schedule'
  );
  if (!confirmed) return;

  try {
    const r = await fetch(`/api/schedules/${id}`, { method: 'DELETE' });
    const j = await r.json();

    if (j.success) {
      showMessage('Schedule deleted successfully');
      loadSchedules();
    } else {
      showMessage(j.error || 'Failed to delete schedule', 'error');
    }
  } catch (e) {
    showMessage('Error: ' + e.message, 'error');
  }
}

// ---- Stop running schedule ----

async function stopSchedule() {
  const confirmed = await window.showConfirmModal(
    'Stop the currently running scheduled job?',
    'Stop Job'
  );
  if (!confirmed) return;

  try {
    const r = await fetch('/api/schedules/stop', { method: 'POST' });
    const j = await r.json();

    if (j.success) {
      if (j.reset) {
        // Job wasn't actually running - stale status was reset
        showMessage(j.message || 'Stale running status was reset.', 'success', 8000);
        loadSchedules();
      } else {
        // Job was actually running and stop signal was sent
        showMessage('Stop signal sent. Waiting for job to stop...', 'success', 15000);
        loadSchedules();
        // Poll until status is no longer "running"
        pollUntilStopped();
      }
    } else {
      showMessage(j.error || 'Failed to stop schedule', 'error');
    }
  } catch (e) {
    showMessage('Error: ' + e.message, 'error');
  }
}

let stopPollInterval = null;

function pollUntilStopped() {
  // Clear any existing poll
  if (stopPollInterval) clearInterval(stopPollInterval);

  let attempts = 0;
  stopPollInterval = setInterval(async () => {
    attempts++;
    try {
      const r = await fetch('/api/schedules');
      const j = await r.json();
      const stillRunning = j.success && j.schedules && j.schedules.some(s => s.lastRunStatus === 'running');

      if (!stillRunning) {
        clearInterval(stopPollInterval);
        stopPollInterval = null;
        showMessage('Job stopped.', 'success', 5000);
        loadSchedules();
      } else if (attempts >= 30) {
        // Give up after ~60 seconds
        clearInterval(stopPollInterval);
        stopPollInterval = null;
        loadSchedules();
      }
    } catch {
      // Ignore fetch errors during polling
    }
  }, 2000);
}

// ---- Toggle schedule ----

async function toggleSchedule(id) {
  try {
    const r = await fetch(`/api/schedules/${id}/toggle`, { method: 'POST' });
    const j = await r.json();

    if (j.success) {
      showMessage(j.enabled ? 'Schedule enabled' : 'Schedule disabled');
      loadSchedules();
    } else {
      showMessage(j.error || 'Failed to toggle schedule', 'error');
    }
  } catch (e) {
    showMessage('Error: ' + e.message, 'error');
  }
}

// ---- Tab init / cleanup ----

export async function initSchedule() {
  // Wire up event handlers
  const typeSelect = $('#sched-type');
  if (typeSelect) typeSelect.onchange = toggleScheduleType;

  // Reports Only toggle - disables Full Refresh when checked
  const reportsOnlyToggle = $('#sched-reports-only');
  if (reportsOnlyToggle) reportsOnlyToggle.onchange = updateReportsOnlyToggle;

  const saveBtn = $('#sched-save-btn');
  if (saveBtn) saveBtn.onclick = saveSchedule;

  const updateBtn = $('#sched-update-btn');
  if (updateBtn) updateBtn.onclick = saveSchedule;

  const saveNewBtn = $('#sched-save-new-btn');
  if (saveNewBtn) saveNewBtn.onclick = saveScheduleAsNew;

  // Add button - show form for adding new schedule
  const addBtn = $('#sched-add-btn');
  if (addBtn) addBtn.onclick = () => {
    clearScheduleForm();
    showFormView('add');
  };

  // Back button - return to list view (with unsaved changes check)
  const backBtn = $('#sched-back-btn');
  if (backBtn) backBtn.onclick = handleBackToList;

  await loadQueries();
  await loadNotificationConfigs();
  await loadSchedules();
}

export function cleanupSchedule() {
  editingScheduleId = null;
  initialFormState = null;
  if (stopPollInterval) {
    clearInterval(stopPollInterval);
    stopPollInterval = null;
  }
  if (messageTimeout) {
    clearTimeout(messageTimeout);
    messageTimeout = null;
  }
}
