// tabs.js - Tab switching and partial loading module

// Partial cache to avoid re-fetching
const partialCache = {};

// Current active tab
let currentTab = 'schedule';

// Tab initialization functions (set by other modules)
const tabInitializers = {};
const tabCleanups = {};

/**
 * Register a tab initializer function
 */
export function registerTabInit(tabName, initFn) {
  tabInitializers[tabName] = initFn;
}

/**
 * Register a tab cleanup function
 */
export function registerTabCleanup(tabName, cleanupFn) {
  tabCleanups[tabName] = cleanupFn;
}

/**
 * Load a partial HTML file
 */
async function loadPartial(tabName) {
  // Check cache first
  if (partialCache[tabName]) {
    return partialCache[tabName];
  }

  // Fetch from server
  const response = await fetch(`/partials/${tabName}.html`);
  if (!response.ok) {
    throw new Error(`Failed to load partial: ${tabName}`);
  }
  const html = await response.text();

  // Cache for next use
  partialCache[tabName] = html;
  return html;
}

/**
 * Check if a partial is already loaded
 */
export function isPartialLoaded(tabName) {
  return !!partialCache[tabName];
}

/**
 * Switch to a different tab
 */
export async function switchTab(tabName) {
  // Skip if this tab is already active and loaded (prevents double-init from hashchange)
  const target = document.getElementById(`${tabName}-tab`);
  if (target?.classList.contains('active') && partialCache[tabName]) return;

  // Close settings dropdown
  const dropdown = document.getElementById('settings-dropdown');
  if (dropdown) {
    dropdown.classList.remove('show');
  }

  // Close resources dropdown
  const resourcesDropdown = document.getElementById('resources-dropdown');
  if (resourcesDropdown) {
    resourcesDropdown.classList.remove('show');
  }

  // Close mobile more menu
  const mobileMoreMenu = document.getElementById('mobile-more-menu');
  if (mobileMoreMenu) {
    mobileMoreMenu.classList.remove('show');
  }

  // Close mobile resources menu
  const mobileResourcesMenu = document.getElementById('mobile-resources-menu');
  if (mobileResourcesMenu) {
    mobileResourcesMenu.classList.remove('show');
  }

  // Run cleanup for current tab if leaving
  if (currentTab !== tabName && tabCleanups[currentTab]) {
    tabCleanups[currentTab]();
  }

  // Hide all tab contents and remove active class from tabs
  document.querySelectorAll('.tab-content').forEach(content => {
    content.classList.remove('active');
  });

  // Remove active class from main tabs
  document.querySelectorAll('.tabs > .tab').forEach(tab => {
    tab.classList.remove('active');
  });

  // Remove active class from dropdown items
  document.querySelectorAll('.tab-dropdown-item').forEach(item => {
    item.classList.remove('active');
  });

  // Remove active class from mobile nav items
  document.querySelectorAll('.mobile-nav-item').forEach(item => {
    item.classList.remove('active');
  });

  // Get the tab container
  const tabContainer = document.getElementById(`${tabName}-tab`);
  if (!tabContainer) {
    console.error(`Tab container not found: ${tabName}-tab`);
    return;
  }

  // Load partial if not already loaded
  if (!partialCache[tabName]) {
    tabContainer.innerHTML = '<p class="loading">Loading...</p>';
    try {
      const html = await loadPartial(tabName);
      tabContainer.innerHTML = html;

      // Initialize tab content
      if (tabInitializers[tabName]) {
        await tabInitializers[tabName]();
      }
    } catch (e) {
      tabContainer.innerHTML = `<p class="err">Failed to load: ${e.message}</p>`;
      return;
    }
  } else {
    // Re-initialize if already loaded (for data refresh)
    if (tabInitializers[tabName]) {
      await tabInitializers[tabName]();
    }
  }

  // Show the selected tab content
  tabContainer.classList.add('active');

  // Highlight active desktop tab
  const topLevelTabs = ['schedule', 'reports'];
  const adminTabs = ['config', 'notifications', 'logs'];
  const resourcesTabs = ['guide', 'about'];

  if (topLevelTabs.includes(tabName)) {
    document.querySelectorAll('.tabs > .tab').forEach(tab => {
      if (tab.getAttribute('onclick') === `switchTab('${tabName}')`) {
        tab.classList.add('active');
      }
    });
  } else if (adminTabs.includes(tabName)) {
    // Highlight the Administration dropdown
    const adminDropdown = document.querySelector('.tab-dropdown > .tab[onclick*="toggleSettingsDropdown"]');
    if (adminDropdown) adminDropdown.classList.add('active');

    const dropdownItem = document.querySelector(`.tab-dropdown-item[data-tab="${tabName}"]`);
    if (dropdownItem) dropdownItem.classList.add('active');
  } else if (resourcesTabs.includes(tabName)) {
    // Highlight the Resources dropdown
    const resourcesDropdown = document.querySelector('.tab-dropdown > .tab[onclick*="toggleResourcesDropdown"]');
    if (resourcesDropdown) resourcesDropdown.classList.add('active');

    const dropdownItem = document.querySelector(`.tab-dropdown-item[data-tab="${tabName}"]`);
    if (dropdownItem) dropdownItem.classList.add('active');
  }

  // Highlight active mobile nav item
  if (topLevelTabs.includes(tabName)) {
    const mobileItem = document.querySelector(`.mobile-nav-item[data-tab="${tabName}"]`);
    if (mobileItem) mobileItem.classList.add('active');
  } else if (adminTabs.includes(tabName)) {
    // For admin tabs, highlight the "more" button
    const mobileMoreItem = document.querySelector('.mobile-nav-item[data-tab="more"]');
    if (mobileMoreItem) mobileMoreItem.classList.add('active');
  } else if (resourcesTabs.includes(tabName)) {
    // For resources tabs, highlight the "resources" button
    const mobileResourcesItem = document.querySelector('.mobile-nav-item[data-tab="resources"]');
    if (mobileResourcesItem) mobileResourcesItem.classList.add('active');
  }

  // Update current tab
  currentTab = tabName;

  // Update URL hash
  window.location.hash = tabName;
}

/**
 * Toggle the settings/admin dropdown menu (desktop)
 */
export function toggleSettingsDropdown(event) {
  event.stopPropagation();
  const dropdown = document.getElementById('settings-dropdown');
  dropdown.classList.toggle('show');

  // Close resources dropdown if open
  const resourcesDropdown = document.getElementById('resources-dropdown');
  if (resourcesDropdown) resourcesDropdown.classList.remove('show');

  // Auto-close when clicking outside
  const closeDropdown = (e) => {
    if (!e.target.closest('.tab-dropdown')) {
      dropdown.classList.remove('show');
      document.removeEventListener('click', closeDropdown);
    }
  };

  if (dropdown.classList.contains('show')) {
    setTimeout(() => document.addEventListener('click', closeDropdown), 0);
  }
}

/**
 * Toggle the resources dropdown menu (desktop)
 */
export function toggleResourcesDropdown(event) {
  event.stopPropagation();
  const dropdown = document.getElementById('resources-dropdown');
  dropdown.classList.toggle('show');

  // Close settings dropdown if open
  const settingsDropdown = document.getElementById('settings-dropdown');
  if (settingsDropdown) settingsDropdown.classList.remove('show');

  // Auto-close when clicking outside
  const closeDropdown = (e) => {
    if (!e.target.closest('.tab-dropdown')) {
      dropdown.classList.remove('show');
      document.removeEventListener('click', closeDropdown);
    }
  };

  if (dropdown.classList.contains('show')) {
    setTimeout(() => document.addEventListener('click', closeDropdown), 0);
  }
}

/**
 * Toggle the mobile more menu
 */
export function toggleMobileMore(event) {
  event.stopPropagation();
  const moreMenu = document.getElementById('mobile-more-menu');
  moreMenu.classList.toggle('show');

  // Close resources menu if open
  const resourcesMenu = document.getElementById('mobile-resources-menu');
  if (resourcesMenu) resourcesMenu.classList.remove('show');

  // Auto-close when clicking outside
  const closeMenu = (e) => {
    if (!e.target.closest('.mobile-more-menu') && !e.target.closest('.mobile-nav-item[data-tab="more"]')) {
      moreMenu.classList.remove('show');
      document.removeEventListener('click', closeMenu);
    }
  };

  if (moreMenu.classList.contains('show')) {
    setTimeout(() => document.addEventListener('click', closeMenu), 0);
  }
}

/**
 * Toggle the mobile resources menu
 */
export function toggleMobileResources(event) {
  event.stopPropagation();
  const resourcesMenu = document.getElementById('mobile-resources-menu');
  resourcesMenu.classList.toggle('show');

  // Close more menu if open
  const moreMenu = document.getElementById('mobile-more-menu');
  if (moreMenu) moreMenu.classList.remove('show');

  // Auto-close when clicking outside
  const closeMenu = (e) => {
    if (!e.target.closest('.mobile-more-menu') && !e.target.closest('.mobile-nav-item[data-tab="resources"]')) {
      resourcesMenu.classList.remove('show');
      document.removeEventListener('click', closeMenu);
    }
  };

  if (resourcesMenu.classList.contains('show')) {
    setTimeout(() => document.addEventListener('click', closeMenu), 0);
  }
}

/**
 * Load the initial tab (called on page load)
 */
export async function loadInitialTab(defaultTab = 'schedule') {
  // Check URL hash for tab navigation
  let hash = window.location.hash.substring(1);
  const validTabs = ['schedule', 'reports', 'config', 'notifications', 'logs', 'guide', 'about'];

  // Migrate old hashes
  if (hash === 'usage' || hash === 'manual') {
    hash = 'schedule';
    window.location.hash = 'schedule';
  }

  if (hash && validTabs.includes(hash)) {
    await switchTab(hash);
  } else {
    await switchTab(defaultTab);
  }
}

/**
 * Get the current active tab name
 */
export function getCurrentTab() {
  return currentTab;
}
